// Copyright 1998-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OIF/OIAnimationInstance.h>

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OIF/OIAnimation.h>
#import <OIF/OIAnimationFrame.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OIF/OIAnimationInstance.m,v 1.18 2000/03/25 06:44:28 wjs Exp $")

@interface OIAnimationInstance (Private)
- (void)displayNextFrame;
- (void)scheduledDisplayNextFrame;
- (void)cancelScheduledEvent;
@end

@implementation OIAnimationInstance

static OFScheduler *animationScheduler;

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
        return;
    initialized = YES;

    animationScheduler = [[OFScheduler mainScheduler] subscheduler];
    [animationScheduler retain];
}

- initWithAnimation:(OIAnimation *)anAnimation;
{
    if (![super init])
        return nil;

    animation = [anAnimation retain];
    remainingLoops = [animation loopCount];

    return self;
}

- (void)dealloc;
{
    [frame removeObserver:self];
    [animation release];
    [super dealloc];
}

// Called by the animation, could possibly be in another thread.

- (void)animationEnded;
{
    if (remainingLoops == 0 || nextFrame <= 1)
        return;

    if (--remainingLoops > 0 && nextFrame != NSNotFound) {
        nextFrame = 0;
        [self displayNextFrame];
    }
}

- (void)animationReceivedFrame:(OIAnimationFrame *)aFrame;
{
    if (nextFrame == NSNotFound)
        return;

    nextFrame++;
    if (frame != aFrame) {
        [frame removeObserver:self];
        frame = aFrame;
        if (!haveSize && [frame hasSize])
            [self setSize:[frame size]];
        [self updateImage:[frame image]];
        [frame addObserver:self];
    }

    if ([observerValues count] != 0) {
        OBASSERT(nextFrameEvent == nil);
        nextFrameEvent = [[animationScheduler scheduleSelector:@selector(scheduledDisplayNextFrame) onObject:self withObject:nil afterTime:[frame delayInterval]] retain];
    }
}

// OIImage subclass

- (void)startAnimation;
{
    [self cancelScheduledEvent];
    remainingLoops = [animation loopCount];
    nextFrame = 0;
    [self displayNextFrame];
}

- (void)stopAnimation;
{
    [self cancelScheduledEvent];
    nextFrame = NSNotFound;
}


- (void)addObserver:(id <OIImageObserver>)anObserver;
{
    BOOL jumpstartAnimation;

    jumpstartAnimation = [observerValues count] == 0;
    [super addObserver:anObserver];
    if (jumpstartAnimation) {
        // We've added our first observer, time to animate
        [self startAnimation];
    }
}

- (void)removeObserver:(id <OIImageObserver>)anObserver;
{
    [super removeObserver:anObserver];
    if ([observerValues count] == 0) {
        // We've removed our last observer, no need to animate
        [self stopAnimation];
    }
}

// OIImageObserver protocol

- (void)imageDidSize:(OIImage *)anOmniImage;
{
    if (!haveSize)
        [self setSize:[anOmniImage size]];
}

- (void)imageDidUpdate:(OIImage *)anOmniImage;
{
    [self updateImage:[anOmniImage image]];
}

- (void)imageDidAbort:(OIImage *)anOmniImage;
{
    [self cancelScheduledEvent];
    // Pass the abort along to our observers
    [self abortImage];
}

// NSObject subclass (Debugging)

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    [debugDictionary setObject:[NSNumber numberWithInt:remainingLoops] forKey:@"remainingLoops"];
    [debugDictionary setObject:[NSNumber numberWithInt:nextFrame] forKey:@"nextFrame"];

    return debugDictionary;
}

@end

@implementation OIAnimationInstance (Private)

- (void)displayNextFrame;
{
    if (nextFrame == NSNotFound)
        return;

    [animation animationInstance:self wantsFrame:nextFrame];
}

- (void)scheduledDisplayNextFrame;
{
    if (nextFrameEvent) {
        // We're done with the event that just called us
        [nextFrameEvent release];
        nextFrameEvent = nil;
    }
    [self displayNextFrame];
}

- (void)cancelScheduledEvent;
{
    if (nextFrameEvent) {
        [animationScheduler abortEvent:nextFrameEvent];
        [nextFrameEvent release];
        nextFrameEvent = nil;
    }
}

@end
