// Copyright 1998-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OIF/OIImage.h>

#import <OmniFoundation/OFObject-Queue.h> // Working around compiler bug
#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OWF/OWF.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OIF/OIImage.m,v 1.24 2000/07/14 07:26:50 kc Exp $")

@implementation OIImage

static OWContentType *contentType;

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
        return;
    initialized = YES;

    contentType = [OWContentType contentTypeForString:@"omni/image"];
}

+ (OWContentType *)contentType;
{
    return contentType;
}

// Gamma

+ (double)sourceGamma;
    // Standard is sRGB, approximately 2.2
{
    return 2.2;
}

+ (double)targetGamma;
    // Standard Macintosh monitors use 1.8
{
    return [[NSUserDefaults standardUserDefaults] floatForKey:@"OIGammaTarget"];
}

+ (double)gammaCorrection;
    // +sourceGamma / +targetGamma
{
    return [self sourceGamma] / [self targetGamma];
}

+ (void)fillGammaCorrectionTable:(OFByte[256])gammaCorrectionTable withSamplesOfGamma:(double)gamma;
{
    unsigned int index;

    index = 256;
    while (index--)
        gammaCorrectionTable[index] = pow(((double)(index) / 255.0), gamma) * 255;
}

// Init and dealloc

- init;
{
    if (![super initWithName:@"Image"])
	return nil;

    haveSize = NO;
    imageSize = NSZeroSize;
    nsImage = nil;
    observerValues = [[NSMutableArray alloc] initWithCapacity:1];

    return self;
}

- (void)dealloc;
{
    [nsImage mainThreadPerformSelector:@selector(release)];
    [observerValues release];
    [sourceData release];
    [super dealloc];
}


// Info

- (BOOL)hasSize;
{
    return haveSize;
}

- (NSSize)size;
{
    return imageSize;
}

- (void)setSize:(NSSize)newSize;
{
    unsigned int observerIndex;
    
    haveSize = YES;
    imageSize = newSize;

    observerIndex = [observerValues count];
    while (observerIndex--) {
        OFObject <OIImageObserver> *observer;

        observer = [[observerValues objectAtIndex:observerIndex] nonretainedObjectValue];
        [observer imageDidSize:self];
    }
}

- (NSImage *)image;
{
    return nsImage;
}

- (void)updateImage:(NSImage *)anImage;
{
    unsigned int observerIndex;

    if (nsImage != anImage) {
        [nsImage release];
        nsImage = [anImage retain];
    }

    [nsImage recache];

    observerIndex = [observerValues count];
    while (observerIndex--) {
        OFObject <OIImageObserver> *observer;

        observer = [[observerValues objectAtIndex:observerIndex] nonretainedObjectValue];
	[observer imageDidUpdate:self];
    }
}

- (void)abortImage;
{
    unsigned int observerIndex;
    
    haveSize = YES; // Well, as much as we ever will...

    observerIndex = [observerValues count];
    while (observerIndex--) {
        OFObject <OIImageObserver> *observer;

        observer = [[observerValues objectAtIndex:observerIndex] nonretainedObjectValue];
	[observer imageDidAbort:self];
    }
}

- (void)startAnimation;
{
    // Only animating subclasses really care.
}

- (void)stopAnimation;
{
    // Only animating subclasses really care.
}

- (void)addObserver:(id <OIImageObserver>)anObserver;
{
    NSValue *value;

    value = [NSValue valueWithNonretainedObject:anObserver];
    OBASSERT([observerValues indexOfObject:value] == NSNotFound);

    [observerValues addObject:value];
}

- (void)removeObserver:(id <OIImageObserver>)anObserver;
{
    NSValue *value;

    value = [NSValue valueWithNonretainedObject:anObserver];
    OBASSERT([observerValues indexOfObject:value] != NSNotFound);

    // The NSValue will be equal to any NSValue that was created from the same object pointer
    [observerValues removeObject:value];
}

- (void)setSourceData:(id <OIImageSourceData>)imageSourceData;
{
    if (sourceData == imageSourceData)
	return;
    [sourceData release];
    sourceData = [imageSourceData retain];
}

// OWContent protocol

- (OWContentType *)contentType;
{
    return contentType;
}

- (OWCursor *)contentCursor;
{
    return nil;
}

- (unsigned long int)cacheSize;
{
    return [sourceData length];
}

- (BOOL)shareable;
{
    return YES;
}

- (BOOL)contentIsValid;
{
    return YES;
}


// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    [debugDictionary setObject:haveSize ? @"YES" : @"NO" forKey:@"haveSize"];
    if (nsImage)
	[debugDictionary setObject:nsImage forKey:@"nsImage"];
    [debugDictionary setObject:observerValues forKey:@"observerValues"];

    return debugDictionary;
}

@end
