// Copyright 1998-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OIF/OIImage.h>

#import <OmniFoundation/OFObject-Queue.h> // Working around compiler bug
#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OWF/OWF.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OIF/OIImage.m,v 1.29 2001/04/16 19:59:52 kc Exp $")

@implementation OIImage

static OWContentType *contentType;

+ (void)initialize;
{
    OBINITIALIZE;

    contentType = [OWContentType contentTypeForString:@"omni/image"];
}

+ (OWContentType *)contentType;
{
    return contentType;
}

// Gamma

+ (double)sourceGamma;
    // Standard is sRGB, approximately 2.2
{
    return 2.2;
}

+ (double)targetGamma;
    // Now that we have ColorSync, this is always sRGB also
{
    return 2.2;
}

+ (double)gammaCorrection;
    // +sourceGamma / +targetGamma
{
    return 1.0; // [self sourceGamma] / [self targetGamma]
}

+ (void)fillGammaCorrectionTable:(OFByte[256])gammaCorrectionTable withSamplesOfGamma:(double)gamma;
{
    unsigned int index;

    index = 256;
    if (gamma == 1.0) {
        // Now that this is the common case, let's optimize it a bit.  (Of course, ideally the caller wouldn't use a gamma table at all in this case.)
        while (index--)
            gammaCorrectionTable[index] = index;
    } else {
        while (index--)
            gammaCorrectionTable[index] = pow(((double)(index) / 255.0), gamma) * 255;
    }
}

// Init and dealloc

- init;
{
    if (![super initWithName:@"Image"])
	return nil;

    haveSize = NO;
    imageSize = NSZeroSize;
    nsImage = nil;
    nsImageLock = [[NSLock alloc] init];
    _observers = [[NSMutableArray alloc] initWithCapacity:1];
    observersLock = [[NSLock alloc] init];

    return self;
}

- (void)dealloc;
{
    unsigned int observerIndex, observerCount;

    [nsImage release];
    [nsImageLock release];
    observerCount = [_observers count];
    for (observerIndex = 0; observerIndex < observerCount; observerIndex++) {
        id <OFWeakRetain> anObserver;
        
        anObserver = [_observers objectAtIndex:observerIndex];
        [anObserver decrementWeakRetainCount];
    }
    [_observers release];
    [observersLock release];
    [sourceData release];
    [super dealloc];
}


// Info

- (BOOL)hasSize;
{
    return haveSize;
}

- (NSSize)size;
{
    return imageSize;
}

- (void)setSize:(NSSize)newSize;
{
    NSArray *observers;
    unsigned int observerIndex, observerCount;
    
    haveSize = YES;
    imageSize = newSize;

    observers = [self observers];
    observerCount = [observers count];
    for (observerIndex = 0; observerIndex < observerCount; observerIndex++) {
        OFObject <OIImageObserver> *observer;

        observer = [observers objectAtIndex:observerIndex];
        [observer imageDidSize:self];
    }
}

- (NSImage *)image;
{
    NSImage *returnedImage;

    [nsImageLock lock];
    returnedImage = [nsImage retain];
    [nsImageLock unlock];
    return [returnedImage autorelease];
}

- (void)updateImage:(NSImage *)anImage;
{
    NSArray *observers;
    unsigned int observerIndex, observerCount;

    [nsImageLock lock];
    NS_DURING {
        if (nsImage != anImage) {
            [nsImage release];
            nsImage = [anImage retain];
        }

        [nsImage recache];
    } NS_HANDLER {
        NSLog(@"-[OIImage updateImage:] caught exception: %@", [localException reason]);
    } NS_ENDHANDLER;
    [nsImageLock unlock];

    observers = [self observers];
    observerCount = [observers count];
    for (observerIndex = 0; observerIndex < observerCount; observerIndex++) {
        OFObject <OIImageObserver> *observer;

        observer = [observers objectAtIndex:observerIndex];
	[observer imageDidUpdate:self];
    }
}

- (void)abortImage;
{
    NSArray *observers;
    unsigned int observerIndex, observerCount;
    
    haveSize = YES; // Well, as much as we ever will...

    observers = [self observers];
    observerCount = [observers count];
    for (observerIndex = 0; observerIndex < observerCount; observerIndex++) {
        OFObject <OIImageObserver> *observer;

        observer = [observers objectAtIndex:observerIndex];
	[observer imageDidAbort:self];
    }
}

- (void)startAnimation;
{
    // Only animating subclasses really care.
}

- (void)stopAnimation;
{
    // Only animating subclasses really care.
}

- (NSArray *)observers;
    // Returns a snapshot of the observers array
{
    NSArray *observers;

    [observersLock lock];
    observers = [[NSArray alloc] initWithArray:_observers];
    [observersLock unlock];
    return [observers autorelease];
}

- (unsigned int)observerCount;
    // Returns the number of observers for this image (more efficient than [[self observers] count])
{
    unsigned int observerCount;

    [observersLock lock];
    observerCount = [_observers count];
    [observersLock unlock];
    return observerCount;
}

- (void)addObserver:(id <OIImageObserver, OFWeakRetain>)anObserver;
    // Subscribed anObserver to receive messages described in the OIImageObserver protocol.  The new observer is  retained.  The new observer is responsible for unsubscribing itself so that it can eventually be deallocated.
{
    [observersLock lock];
    OBASSERT([_observers indexOfObjectIdenticalTo:anObserver] == NSNotFound);
    [_observers addObject:anObserver];
    [observersLock unlock];
    [anObserver incrementWeakRetainCount];
}

- (void)removeObserver:(id <OIImageObserver, OFWeakRetain>)anObserver;
    // Unsubscribes anObserver such that it will not receive the messages described in the OIImageObserver protocol and is no longer retained by the image.
{
    [anObserver decrementWeakRetainCount];
    [observersLock lock];
    OBASSERT([_observers indexOfObjectIdenticalTo:anObserver] != NSNotFound);
    [_observers removeObjectIdenticalTo:anObserver];
    [observersLock unlock];
}

- (void)setSourceData:(id <OIImageSourceData>)imageSourceData;
{
    [nsImageLock lock];
    if (sourceData != imageSourceData) {
        [sourceData release];
        sourceData = [imageSourceData retain];
    }
    [nsImageLock unlock];
}

// OWContent protocol

- (OWContentType *)contentType;
{
    return contentType;
}

- (OWCursor *)contentCursor;
{
    return nil;
}

- (unsigned long int)cacheSize;
{
    unsigned long int cacheSize;
    
    [nsImageLock lock];
    cacheSize = [sourceData length];
    [nsImageLock unlock];
    return cacheSize;
}

- (BOOL)shareable;
{
    return YES;
}

- (BOOL)contentIsValid;
{
    return YES;
}


// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    [debugDictionary setObject:haveSize ? @"YES" : @"NO" forKey:@"haveSize"];
    if (nsImage)
	[debugDictionary setObject:nsImage forKey:@"nsImage"];
    [debugDictionary setObject:[self observers] forKey:@"observers"];

    return debugDictionary;
}

@end
