// Copyright 1998-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OIF/OIAnimationInstance.h>

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OIF/OIAnimation.h>
#import <OIF/OIAnimationFrame.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OIF/OIAnimationInstance.m,v 1.28 2001/09/13 20:30:13 kc Exp $")

@interface OIAnimationInstance (Private)
- (void)_displayNextFrame;
- (void)_scheduledDisplayNextFrame;
- (void)_cancelScheduledEvent;
- (void)_setScheduledEvent:(OFScheduledEvent *)newEvent;
@end

@implementation OIAnimationInstance

static OFScheduler *animationScheduler;

+ (void)initialize;
{
    OBINITIALIZE;

    animationScheduler = [[OFScheduler mainScheduler] subscheduler];
    [animationScheduler retain];
}

- (id)initWithSourceContent:(id <OWContent>)sourceContent;
{
    OBRejectUnusedImplementation(self, _cmd);
    return nil; // NOT REACHED
}

- (id)initWithAnimation:(OIAnimation *)anAnimation;
{
    if ([super initWithSourceContent:nil] == nil)
        return nil;

    OFWeakRetainConcreteImplementation_INIT;
    animation = [anAnimation retain];
    remainingLoops = [anAnimation loopCount];
    nextFrameEventLock = [[NSLock alloc] init];
    
    return self;
}

- (void)dealloc;
{
    OFWeakRetainConcreteImplementation_DEALLOC;
    OBASSERT(frame == nil); // Our frame was already released by -invalidateWeakRetains
    [animation release];
    [nextFrameEventLock release];
    [nextFrameEvent release]; // Assert this is nil?
    
    [super dealloc];
}

// Called by the animation, could possibly be in another thread.

- (void)animationEnded;
{
    if (remainingLoops == 0 || nextFrame <= 1)
        return;

    if (--remainingLoops > 0 && nextFrame != NSNotFound) {
        nextFrame = 0;
        [self _displayNextFrame];
    }
}

- (void)animationReceivedFrame:(OIAnimationFrame *)aFrame;
{
    if (nextFrame == NSNotFound)
        return;

    nextFrame++;
    if (frame != aFrame) {
        [frame removeObserver:self];
        [frame release];
        frame = [aFrame retain];
        if (!haveSize && [frame hasSize])
            [self setSize:[frame size]];
        [self updateImage:[frame image]];
        [frame addObserver:self];
    }

    if ([self observerCount] != 0) {
        OFScheduledEvent *event;

        event = [animationScheduler scheduleSelector:@selector(_scheduledDisplayNextFrame) onObject:self withObject:nil afterTime:[frame delayInterval]];
        [self _setScheduledEvent:event];
    }
}

OFWeakRetainConcreteImplementation_IMPLEMENTATION

- (void)invalidateWeakRetains;
{
    [frame removeObserver:self];
    [frame release];
    frame = nil;
}

// OIImage subclass

- (id <OWContent>)sourceContent;
{
    return [animation sourceContent];
}

- (void)startAnimation;
{
    [self _cancelScheduledEvent];
    remainingLoops = [animation loopCount];
    nextFrame = 0;
    [self _displayNextFrame];
}

- (void)stopAnimation;
{
    [self _cancelScheduledEvent];
    nextFrame = NSNotFound;
}


- (void)addObserver:(id <OIImageObserver, OFWeakRetain>)anObserver;
{
    BOOL jumpstartAnimation;

    jumpstartAnimation = [self observerCount] == 0;
    [super addObserver:anObserver];
    if (jumpstartAnimation) {
        // We've added our first observer, time to animate
        [self startAnimation];
    }
}

- (void)removeObserver:(id <OIImageObserver, OFWeakRetain>)anObserver;
{
    [super removeObserver:anObserver];
    if ([self observerCount] == 0) {
        // We've removed our last observer, no need to animate
        [self stopAnimation];
    }
}

// OIImageObserver protocol

- (void)imageDidSize:(OIImage *)anOmniImage;
{
    if (!haveSize)
        [self setSize:[anOmniImage size]];
}

- (void)imageDidUpdate:(OIImage *)anOmniImage;
{
    [self updateImage:[anOmniImage image]];
}

- (void)imageDidAbort:(OIImage *)anOmniImage;
{
    [self _cancelScheduledEvent];
    // Pass the abort along to our observers
    [self abortImage];
}

// NSObject subclass (Debugging)

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    [debugDictionary setObject:[NSNumber numberWithInt:remainingLoops] forKey:@"remainingLoops"];
    [debugDictionary setObject:[NSNumber numberWithInt:nextFrame] forKey:@"nextFrame"];

    return debugDictionary;
}

@end

@implementation OIAnimationInstance (Private)

- (void)_displayNextFrame;
{
    if (nextFrame == NSNotFound)
        return;

    [animation animationInstance:self wantsFrame:nextFrame];
}

- (void)_scheduledDisplayNextFrame;
{
    [nextFrameEventLock lock];
    if (nextFrameEvent != nil) {
        // We're done with the event that just called us
        [nextFrameEvent release];
        nextFrameEvent = nil;
    }
    [nextFrameEventLock unlock];
    [self _displayNextFrame];
}

- (void)_cancelScheduledEvent;
{
    [self _setScheduledEvent:nil];
}

- (void)_setScheduledEvent:(OFScheduledEvent *)newEvent;
{
    OFScheduledEvent *oldEvent;

    [nextFrameEventLock lock];
    oldEvent = nextFrameEvent; // Inherit retain
    nextFrameEvent = [newEvent retain];
    [nextFrameEventLock unlock];
    if (oldEvent != nil) {
        [animationScheduler abortEvent:oldEvent];
        [oldEvent release];
    }
}

@end
