// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OWF/OWAuthorizationRealm.h>

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/Processors.subproj/Protocols.subproj/HTTP.subproj/OWAuthorizationRealm.m,v 1.11 2000/03/25 06:44:39 wjs Exp $")

@interface OWAuthorizationRealm (Private)
+ (void)controllerDidInit:(NSNotification *)notification;
+ (NSString *)savedCredentialsPathname;
//
- initWithName:(NSString *)aName forHostAddress:(NSString *)hostAddress;
@end

#warning There are a bunch of strings here which need to be made localizable

@implementation OWAuthorizationRealm

static NSMutableDictionary *realmDictionary;
static NSLock *realmDictionaryLock;
static NSMutableDictionary *schemeDictionary;
static NSLock *schemeDictionaryLock;
static NSMutableDictionary *savedCredentials;
static NSLock *savedCredentialsLock;

enum {CREDENTIALS_NEEDED, CREDENTIALS_READY};

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
	return;
    initialized = YES;

    realmDictionary = [[NSMutableDictionary alloc] init];
    realmDictionaryLock = [[NSLock alloc] init];
    schemeDictionary = [[NSMutableDictionary alloc] init];
    schemeDictionaryLock = [[NSLock alloc] init];

    savedCredentialsLock = [[NSLock alloc] init];
}

+ (void)didLoad;
{
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(controllerDidInit:) name:OFControllerDidInitNotification object:nil];
}

+ (void)registerSelector:(SEL)aSelector forScheme:(NSString *)scheme;
{
    [schemeDictionaryLock lock];
    [schemeDictionary setObject:[[OFImplementationHolder alloc] initWithSelector:aSelector] forKey:[scheme lowercaseString]];
    [schemeDictionaryLock unlock];
}

+ (OWAuthorizationRealm *)realmWithName:(NSString *)aName forHostAddress:(NSString *)hostAddress;
{
    return [[[self alloc] initWithName:aName forHostAddress:hostAddress] autorelease];
}

//

- (void)dealloc;
{
    [name release];
    [defaultsKey release];
    [credentialsByScheme release];
    [credentialsLock release];
    [cachedCredentials release];
    [super dealloc];
}

//

- (NSArray *)credentials;
{
    NSArray *credentials;
    NSEnumerator *credentialsEnumerator;
    NSArray *schemeCredentials;

    [credentialsLock lockWhenCondition:CREDENTIALS_READY];
    if ((credentials = [[cachedCredentials retain] autorelease])) {
	[credentialsLock unlock];
	return credentials;
    }
    cachedCredentials = [[NSMutableArray alloc] initWithCapacity:1];
    credentialsEnumerator = [credentialsByScheme objectEnumerator];
    while ((schemeCredentials = [credentialsEnumerator nextObject]))
	[cachedCredentials addObjectsFromArray:schemeCredentials];
    credentials = [[cachedCredentials retain] autorelease];
    [credentialsLock unlock];
    return credentials;
}

- (void)generateCredentialsForScheme:(NSString *)scheme challenge:(OWHeaderDictionary *)challenge reprompt:(BOOL)shouldReprompt;
{
    OFImplementationHolder *implementation;

    [credentialsLock lock];
    if (!shouldReprompt && [credentialsByScheme objectForKey:scheme]) {
	[credentialsLock unlock];
	return;
    }
    implementation = [schemeDictionary objectForKey:[scheme lowercaseString]];
    if (!implementation) {
	[credentialsLock unlock];
	[NSException raise:@"Unauthorized" format:@"Unsupported authentication scheme %@", scheme];
    }
    [credentialsLock unlockWithCondition:CREDENTIALS_NEEDED];
    [implementation executeOnObject:self withObject:scheme withObject:challenge];
    [credentialsLock lockWhenCondition:CREDENTIALS_READY];
    if (![credentialsByScheme objectForKey:scheme]) {
	[credentialsLock unlock];
        [NSException raise:@"Unauthorized" format:@"No credentials for scheme %@", scheme];
    }
    [credentialsLock unlock];
}

- (void)setCredentials:(NSArray *)credentials forScheme:(NSString *)scheme save:(BOOL)shouldSave;
{
    [credentialsLock lock];
    if (!credentialsByScheme && credentials)
	credentialsByScheme = [[NSMutableDictionary alloc] initWithCapacity:1];
    if (credentials)
	[credentialsByScheme setObject:credentials forKey:scheme];
    else
	[credentialsByScheme removeObjectForKey:scheme];
    [cachedCredentials release];
    cachedCredentials = nil;
    if (shouldSave) {
        [savedCredentialsLock lock];
        [savedCredentials setObject:credentialsByScheme forKey:defaultsKey];
        [savedCredentials writeToFile:[isa savedCredentialsPathname] atomically:YES];
        [[NSFileManager defaultManager] changeFileAttributes:[NSDictionary dictionaryWithObject:[NSNumber numberWithInt:0600] forKey:NSFilePosixPermissions] atPath:[isa savedCredentialsPathname]];
        [savedCredentialsLock unlock];
    }
    [credentialsLock unlockWithCondition:CREDENTIALS_READY];
}

// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    if (credentialsByScheme)
	[debugDictionary setObject:credentialsByScheme forKey:@"credentialsByScheme"];
    return debugDictionary;
}

@end

@implementation OWAuthorizationRealm (Private)

+ (void)controllerDidInit:(NSNotification *)notification;
{
    savedCredentials = [[NSMutableDictionary alloc] initWithContentsOfFile:[self savedCredentialsPathname]];
    if (!savedCredentials)
        savedCredentials = [[NSMutableDictionary alloc] init];
}

+ (NSString *)savedCredentialsPathname;
{
    return [[[[NSUserDefaults standardUserDefaults] objectForKey:@"OWLibraryDirectory"] stringByStandardizingPath] stringByAppendingPathComponent:@"Credentials.plist"];

}

- initWithName:(NSString *)aName forHostAddress:(NSString *)hostAddress;
{
    if (![super init])
	return nil;

    name = [aName retain];
    defaultsKey = [[NSString alloc] initWithFormat:@"%@: %@", aName, hostAddress];
    [savedCredentialsLock lock];
    credentialsByScheme = [[savedCredentials objectForKey:defaultsKey] mutableCopy];
    [savedCredentialsLock unlock];
    credentialsLock = [[NSConditionLock alloc] initWithCondition:CREDENTIALS_READY];

    return self;
}

@end
