// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OWF/OWCookieDomain.h>

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OWF/NSDate-OWExtensions.h>
#import <OWF/OWAddress.h>
#import <OWF/OWCookiePath.h>
#import <OWF/OWCookie.h>
#import <OWF/OWHeaderDictionary.h>
#import <OWF/OWHTTPSession.h>
#import <OWF/OWNetLocation.h>
#import <OWF/OWWebPipeline.h>
#import <OWF/OWURL.h>
#import <OmniExpat/xmlparse.h>


RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/Processors.subproj/Protocols.subproj/HTTP.subproj/OWCookieDomain.m,v 1.16 2000/07/15 00:32:22 krevis Exp $")


static NSLock              *domainLock;
static NSMutableDictionary *domainsByName;
static OFScheduledEvent    *saveEvent;

static NSArray             *shortTopLevelDomains;
static NSCharacterSet      *endNameSet, *endValueSet, *endDateSet, *endKeySet;
static NSTimeInterval       distantPastInterval;

static OWCookieDomainBehavior defaultBehavior = OWCookieDomainPromptBehavior;

static id    classDelegate;

static NSString *OWCookieImmunizedDomainPath = @"(Immunized Domain)";
static NSString *OW4CookieFileName           = @"Cookies.xml";
static NSString *OW3CookieFileName           = @"Cookies.plist";
NSString *OWCookiesChangedNotification       = @"OWCookiesChangedNotification";

BOOL  OWCookiesDebug = NO;

NSString *OWSetCookieHeader = @"set-cookie";


static inline const char *_stringForBehavior(OWCookieDomainBehavior behavior)
{
    const char *string;
    
    switch (behavior) {
        case OWCookieDomainAcceptBehavior:
            string = "accept";
            break;
        case OWCookieDomainRejectBehavior:
            string = "reject";
            break;
        case OWCookieDomainPromptBehavior:
        default:
            string = "prompt";
            break;
    }
    
    return string;
}

static inline OWCookieDomainBehavior _behaviorForString(NSString *behaviorString)
{
    if ([behaviorString isEqualToString: @"accept"])
        return OWCookieDomainAcceptBehavior;
    if ([behaviorString isEqualToString: @"reject"])
        return OWCookieDomainRejectBehavior;
    if ([behaviorString isEqualToString: @"prompt"])
        return OWCookieDomainPromptBehavior;
    return OWCookieDomainDefaultBehavior;
}

static inline OWCookieDomainBehavior _behaviorForCString(const char *behaviorString)
{
    if (!strcmp(behaviorString, "accept"))
        return OWCookieDomainAcceptBehavior;
    if (!strcmp(behaviorString, "reject"))
        return OWCookieDomainRejectBehavior;
    if (!strcmp(behaviorString, "prompt"))
        return OWCookieDomainPromptBehavior;
    return OWCookieDomainDefaultBehavior;
}


@interface OWCookieDomain (PrivateAPI)
+ (void)controllerDidInit:(NSNotification *)notification;
+ (void)saveCookies;
+ (NSString *)cookiePath: (NSString *) fileName;
+ (void)locked_didChange;
+ (void) notifyCookiesChanged;
- (void)addCookie: (OWCookie *) cookie andNotify: (BOOL) shouldNotify;
+ (OWCookieDomain *) domainNamed: (NSString *) name
                        behavior: (OWCookieDomainBehavior) behavior
                       andNotify: (BOOL) shouldNotify;
- (OWCookiePath *) locked_pathNamed: (NSString *) pathName shouldCreate: (BOOL) shouldCreate;
+ (unsigned int)minimumDomainComponentsForTopLevelDomain:(NSString *)aTopLevelDomain;
+ (NSArray *)searchDomainsForDomain:(NSString *)aDomain;
+ (OWCookie *)cookieFromHeaderValue:(NSString *)headerValue defaultDomain:(NSString *)defaultDomain defaultPath:(NSString *)defaultPath;
- (void) locked_addApplicableCookies: (NSMutableArray *) cookies
                             forPath: (NSString *) aPath
                         urlIsSecure: (BOOL) secure;
+ (BOOL)locked_readOW3Cookies;
+ (BOOL)locked_readOW4Cookies;
- (id) initWithDomain: (NSString *) domain behavior: (OWCookieDomainBehavior) behavior;
- (BOOL) promptForCookie: (OWCookie *) cookie pipeline: (OWPipeline *) pipeline;
@end


@implementation OWCookieDomain

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
	return;
    initialized = YES;

    domainLock = [[NSRecursiveLock alloc] init];
    domainsByName = [[NSMutableDictionary alloc] init];
    
    endNameSet = [[NSCharacterSet characterSetWithCharactersInString:@"=;, \t\r\n"] retain];
    endDateSet = [[NSCharacterSet characterSetWithCharactersInString:@";\r\n"] retain];
    endValueSet = [[NSCharacterSet characterSetWithCharactersInString:@";, \t\r\n"] retain];
    endKeySet = [[NSCharacterSet characterSetWithCharactersInString:@"=;, \t\r\n"] retain];
    
    distantPastInterval = [[NSDate distantPast] timeIntervalSinceReferenceDate];
}

+ (void)didLoad;
{
    NSNotificationCenter *center;
    
    center = [NSNotificationCenter defaultCenter];
    [center addObserver:self selector:@selector(controllerDidInit:) name:OFControllerDidInitNotification object:nil];
    [center addObserver:self selector:@selector(controllerWillTerminate:) name:OFControllerWillTerminateNotification object:nil];
    
}

+ (void)readDefaults;
{
    NSUserDefaults *userDefaults;
    NSString *value;
    
    [domainLock lock];

    userDefaults = [NSUserDefaults standardUserDefaults];
    [shortTopLevelDomains release];
    shortTopLevelDomains = [[userDefaults arrayForKey:@"OWShortTopLevelDomains"] retain];

    // Upgrade our defaults -- check if the old default exists, and if so, what it was
    if ((value = [userDefaults objectForKey: @"OWHTTPRefuseAllCookies"])) {
        if ([userDefaults boolForKey: @"OWHTTPRefuseAllCookies"])
            [userDefaults setInteger: OWCookieDomainRejectBehavior
                              forKey: @"OWHTTPCookieDefaultBehavior"];
        [userDefaults removeObjectForKey: @"OWHTTPRefuseAllCookies"];
        [userDefaults synchronize];
    }
    
    value = [userDefaults stringForKey: @"OWHTTPCookieDefaultBehavior"];
    if ([value isEqualToString: @"accept"])
        defaultBehavior = OWCookieDomainAcceptBehavior;
    else if ([value isEqualToString: @"reject"])
        defaultBehavior = OWCookieDomainRejectBehavior;
    else
        defaultBehavior = OWCookieDomainPromptBehavior;

    [domainLock unlock];
}

+ (void) registerCookiesFromPipeline: (OWPipeline *) pipeline
                         headerValue: (NSString *) headerValue
{
    NSString *defaultDomain, *defaultPath;
    OWURL *url;
    OWCookie *cookie;
    OWCookieDomain *domain;
    OWCookieDomainBehavior behavior;
    BOOL shouldAccept;

    url = [(OWAddress *)[pipeline lastAddress] url];
    if (!url)
        return;

    defaultDomain = [[url parsedNetLocation] hostname];
    defaultPath = [@"/" stringByAppendingString:[url path]];
#if 0
    // This is incorrect, but since our search algorithm is also incorrect, I think this might produce better behavior
    if ([defaultPath length] > 1)
        defaultPath = [OWURL stringByDeletingLastPathComponentFromPath:defaultPath];
// #else
    // It's unclear exactly what we're supposed to use for a default path. This seems to follow the only document that specifies any actual behavior. I'm deviating from the specified behavior in cases which I think they forgot to consider.

    defaultPathComponents = [OWURL pathComponentsForPath:defaultPath];
    if ([defaultPathComponents count] /* &&
        [[defaultPathComponents lastObject] length] > 0 */ ) {
        defaultPathComponents = [defaultPathComponents mutableCopy];
        [defaultPathComponents removeLastObject];
        if ([defaultPathComponents count] < 2)
            defaultPath = @"/";
        else
            defaultPath = [defaultPathComponents componentsJoinedByString:@"/"];
    }
#else
    // this is *also* incorrect, in a different way
    {
        NSRange slashRange = [defaultPath rangeOfString:@"/" options:NSBackwardsSearch];
        if (slashRange.length != 0 && slashRange.location > 0) {
            defaultPath = [defaultPath substringToIndex:slashRange.location];
        }
    }
#endif

    OBASSERT(defaultDomain != nil);

    if (OWCookiesDebug)
        NSLog(@"COOKIES: Register url=%@ domain=%@ path=%@ header=%@", [url shortDescription], defaultDomain, defaultPath, headerValue);

    cookie = [self cookieFromHeaderValue:headerValue defaultDomain:defaultDomain defaultPath:defaultPath];
    if (!cookie)
        return;
    
    // The cookie itself can specify a domain, so get the domain that
    // ends up in the actual cookie instance.
    if (OWCookiesDebug)
        NSLog(@"COOKIES: url=%@, adding cookie = %@", [url shortDescription], cookie);
        
    domain = [self domainNamed: [cookie domain]];
    behavior = [domain behavior];
    if (OWCookiesDebug)
        NSLog(@"COOKIES: behavior is %s", _stringForBehavior(behavior));

    switch (behavior) {
        case OWCookieDomainAcceptBehavior:
            shouldAccept = YES;
            break;
        case OWCookieDomainRejectBehavior:
            shouldAccept = NO;
            break;
        case OWCookieDomainPromptBehavior:
        default:
            shouldAccept = [domain promptForCookie: cookie
                                          pipeline: pipeline];
            break;
    }
    
    if (shouldAccept)
        [domain addCookie: cookie];
}

+ (void)registerCookiesFromPipeline: (OWPipeline *) pipeline
                   headerDictionary: (OWHeaderDictionary *)headerDictionary;
{
    NSArray *valueArray;
    unsigned int valueIndex, valueCount;

    valueArray = [headerDictionary stringArrayForKey: OWSetCookieHeader];
    if (!valueArray)
	return;

    valueCount = [valueArray count];
    for (valueIndex = 0; valueIndex < valueCount; valueIndex++) {
        NSString *headerValue;
        
        headerValue = [valueArray objectAtIndex:valueIndex];
        [self registerCookiesFromPipeline: pipeline
                              headerValue: headerValue];
    }
}

+ (NSArray *)cookiesForURL:(OWURL *)url;
{
    NSMutableArray *cookies;
    NSArray *searchDomains;
    unsigned int domainIndex, domainCount;
    NSString *hostname, *searchDomain, *path;
    OWCookieDomain *domain;


    hostname = [[[url parsedNetLocation] hostname] lowercaseString];
    path = [@"/" stringByAppendingString:[url path]];
    searchDomains = [self searchDomainsForDomain: hostname];

    if (OWCookiesDebug)
        NSLog(@"COOKIES: url=%@ hostname=%@, path=%@ --> domains=%@", url, hostname, path, searchDomains);

    cookies = [NSMutableArray array];
    [domainLock lock];
    domainCount = [searchDomains count];
    for (domainIndex = 0; domainIndex < domainCount; domainIndex++) {
        searchDomain = [searchDomains objectAtIndex:domainIndex];
        domain = [domainsByName objectForKey: searchDomain];
        [domain locked_addApplicableCookies: cookies forPath: path urlIsSecure: [url isSecure]];
    }
    
    [domainLock unlock];

    if (OWCookiesDebug)
        NSLog(@"COOKIES: -cookiesForURL:%@ --> %@", [url shortDescription], cookies);

    return cookies;
}


+ (void) didChange;
{
    [domainLock lock];
    [self locked_didChange];
    [domainLock unlock];
}

+ (NSArray *) allDomains;
{
    NSArray *domains;
    
    [domainLock lock];
    domains = [[domainsByName allValues] sortedArrayUsingSelector: @selector(compare:)];
    [domainLock unlock];
    
    return domains;
}

+ (OWCookieDomain *) domainNamed: (NSString *) name;
{
    return [self domainNamed: name behavior: OWCookieDomainDefaultBehavior andNotify: YES];
}

+ (void) deleteDomain: (OWCookieDomain *) domain;
{
    [domainLock lock];
    [domainsByName removeObjectForKey: [domain name]];
    [self locked_didChange];
    [domainLock unlock];
}

+ (void) deleteCookie: (OWCookie *) cookie;
{
    OWCookieDomain *domain;
    
    [domainLock lock];
    // Its domain might have been deleted already which is why
    // this method exists -- the caller can't call +domainNamed:
    // and delete the cookie from there since that might recreate
    // a deleted domain.
    domain = [domainsByName objectForKey: [cookie domain]];
    [domain removeCookie: cookie];
    [domainLock unlock];
}

+ (void) setDelegate: (id) delegate;
{
    classDelegate = delegate;
}

+ (id) delegate;
{
    return classDelegate;
}

+ (OWCookieDomainBehavior) defaultBehavior;
{
    return defaultBehavior;
}

+ (void) setDefaultBehavior: (OWCookieDomainBehavior) behavior;
{
    NSUserDefaults *defaults;
    NSString *string;
    
    defaultBehavior = behavior;
    string = [NSString stringWithCString: _stringForBehavior(defaultBehavior)];
    defaults = [NSUserDefaults standardUserDefaults];
    [defaults setObject: string forKey: @"OWHTTPCookieDefaultBehavior"];
    [defaults synchronize];
}

- (NSString *) name;
{
    return _name;
}

- (OWCookieDomainBehavior) internalBehavior;
{
    return _behavior;
}


- (void) setInternalBehavior: (OWCookieDomainBehavior) behavior;
{
    // Don't really need this lock for the ivar set, but do for the method call
    [domainLock lock];
    _behavior = behavior;
    [isa locked_didChange];
    [domainLock unlock];
}

- (OWCookieDomainBehavior) behavior;
{
    if (_behavior == OWCookieDomainDefaultBehavior)
        return defaultBehavior;
    return _behavior;
}

- (NSArray *) paths;
{
    NSArray *paths;
    
    [domainLock lock];
    paths = [[NSArray alloc] initWithArray: _cookiePaths];
    [domainLock unlock];
    
    return [paths autorelease];
}

- (OWCookiePath *) pathNamed: (NSString *) pathName;
{
    OWCookiePath *path;
    
    [domainLock lock];
    path = [[self locked_pathNamed: pathName shouldCreate: YES] retain];
    [domainLock unlock];
    
    return [path autorelease];
}

//
// Saving
//

- (void) appendXML: (OFDataBuffer *) xmlBuffer;
{
    unsigned int pathIndex, pathCount;
    unsigned int cookieIndex, cookieCount;
    NSMutableArray *cookies;
    
    cookies = [NSMutableArray array];
    
    [domainLock lock];

    // The paths are not represented in the XML file (since they are
    // ususally the default and there are usually few enough cookies
    // per path that it would be a waste.
    pathCount = [_cookiePaths count];
    for (pathIndex = 0; pathIndex < pathCount; pathIndex++) {
        OWCookiePath *path;
        
        path = [_cookiePaths objectAtIndex: pathIndex];
        [path addNonExpiredCookiesToArray: cookies usageIsSecure: YES];
    }
    
    // Don't archive domains with zero cookies and a default behavior.
    cookieCount = [cookies count];
    if (cookieCount || _behavior != OWCookieDomainDefaultBehavior) {
        OFDataBufferAppendCString(xmlBuffer, "<domain name=\"");
        // This *shouldn't* have entities in it, but ...
        OFDataBufferAppendXMLQuotedString(xmlBuffer, (CFStringRef)_name);
        if (_behavior != OWCookieDomainDefaultBehavior) {
            OFDataBufferAppendCString(xmlBuffer, "\" behavior=\"");
            OFDataBufferAppendCString(xmlBuffer, _stringForBehavior(_behavior));
        }
        OFDataBufferAppendCString(xmlBuffer, "\">\n");
    
        for (cookieIndex = 0; cookieIndex < cookieCount; cookieIndex++) {
            OWCookie *cookie;
            
            // -addNonExpiredCookiesToArray:usageIsSecure: filters out the cookies
            // with expiration dates that have passed, but it does NOT filter out
            // cookies with no expiration date (those that expire at the end of the
            // session, and thus should not be written out).
            // This extra filtering here could cause us to write a domain with
            // zero cookies, but that will be fixed the next time we write the
            // file, most likely.
            cookie = [cookies objectAtIndex: cookieIndex];
            if ([cookie expirationDate])
                [cookie appendXML: xmlBuffer];
        }
        
        OFDataBufferAppendCString(xmlBuffer, "</domain>\n");
    }
    
    [domainLock unlock];
}

//
// Convenience methods that loop over all the paths
//

- (void) addCookie: (OWCookie *) cookie;
{
    [self addCookie: cookie andNotify: YES];
}

- (void) removeCookie: (OWCookie *) cookie;
{
    OWCookiePath *path;
    
    [domainLock lock];
    path = [self locked_pathNamed: [cookie path] shouldCreate: NO];
    [path removeCookie: cookie];
    [domainLock unlock];
}

- (NSArray *) cookies;
{
    NSMutableArray *cookies;
    unsigned int pathIndex, pathCount;
    
    cookies = [NSMutableArray array];
    [domainLock lock];
    pathCount = [_cookiePaths count];
    for (pathIndex = 0; pathIndex < pathCount; pathIndex++)
        [[_cookiePaths objectAtIndex: pathIndex] addNonExpiredCookiesToArray: cookies usageIsSecure: YES];
    [domainLock unlock];
    
    return cookies;
}

- (NSComparisonResult) compare: (id) otherObject;
{
    if (![otherObject isKindOfClass: isa])
        return NSOrderedAscending;
    
    return [_name compare: [(OWCookieDomain *)otherObject name]];
}

@end



@implementation OWCookieDomain (PrivateAPI)

+ (void)controllerDidInit:(NSNotification *)notification;
{
    [self readDefaults];
    
    [domainLock lock];
    
    // Read the cookies, prefering the newest file is it exists
    if (![self locked_readOW4Cookies]) {
        if ([self locked_readOW3Cookies]) {
            // If we got an old file, make sure we save a new file immediately
            [self saveCookies];
        }
    }

    [domainLock unlock];
}

+ (void)controllerWillTerminate: (NSNotification *) note;
{
    [self saveCookies];
}

+ (void)saveCookies;
{
    NSString *cookieFilename;
    NSArray *domains;
    OFDataBuffer xmlBuffer;
    unsigned int domainIndex, domainCount;
    
    // This must get executed in the main thread so that the notification
    // gets posted in the main thread (since that is where the cookie preferences
    // panel is listening).
    OBPRECONDITION([NSThread inMainThread]);

    if (OWCookiesDebug)
        NSLog(@"COOKIES: Saving");

    if (!(cookieFilename = [self cookiePath: OW4CookieFileName])) {
        if (OWCookiesDebug)
            NSLog(@"COOKIES: Unable to compute cookie path");
        return;
    }

    [domainLock lock];
    
    [saveEvent release];
    saveEvent = nil;
    
    OFDataBufferInit(&xmlBuffer);
#warning TJW -- I still need to write a DTD for this file and put it on our web site
    OFDataBufferAppendCString(&xmlBuffer,
    "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
    "<!DOCTYPE OmniWebCookies SYSTEM \"http://www.omnigroup.com/DTDs/OmniWebCookies.dtd\">\n"
    "<OmniWebCookies>\n");

    domains = [[domainsByName allValues] sortedArrayUsingSelector: @selector(compare:)];
    domainCount = [domains count];
    for (domainIndex = 0; domainIndex < domainCount; domainIndex++)
        [[domains objectAtIndex: domainIndex] appendXML: &xmlBuffer];

    OFDataBufferAppendCString(&xmlBuffer, "</OmniWebCookies>\n");

    OFDataBufferFlush(&xmlBuffer);
    [OFDataBufferData(&xmlBuffer) writeToFile: cookieFilename atomically: YES];
    
    OFDataBufferRelease(&xmlBuffer);
    
    [domainLock unlock];
}

+ (NSString *)cookiePath: (NSString *) fileName
{
    NSString *directory;
    NSFileManager *fileManager;
    BOOL isDirectory;

    directory = [[[NSUserDefaults standardUserDefaults] objectForKey:@"OWLibraryDirectory"] stringByStandardizingPath];
    fileManager = [NSFileManager defaultManager];
    if (![fileManager fileExistsAtPath:directory isDirectory:&isDirectory] || !isDirectory)
        return nil;

    return [directory stringByAppendingPathComponent: fileName];
}

+ (void) locked_didChange;
{
    OFScheduler *mainScheduler;
    
    mainScheduler = [OFScheduler mainScheduler];
    
    // Kill the old scheduled event and schedule one for later
    if (saveEvent) {
        [mainScheduler abortEvent: saveEvent];
        [saveEvent release];
    }

    saveEvent = [[mainScheduler scheduleSelector: @selector(saveCookies)
                                        onObject: self
                                      withObject: nil
                                       afterTime: 60.0] retain];

    if (OWCookiesDebug)
        NSLog(@"COOKIES: Did change, saveEvent = %@", saveEvent);
        
    [self mainThreadPerformSelectorOnce: @selector(notifyCookiesChanged)];
}

+ (void) notifyCookiesChanged;
{
    [[NSNotificationCenter defaultCenter] postNotificationName: OWCookiesChangedNotification
                                                        object: nil];
}

+ (OWCookieDomain *) domainNamed: (NSString *) name
                        behavior: (OWCookieDomainBehavior) behavior
                       andNotify: (BOOL) shouldNotify;
{
    OWCookieDomain *domain;
    
    [domainLock lock];
    if (!(domain = [domainsByName objectForKey: name])) {
        domain = [[self alloc] initWithDomain: name behavior: behavior];
        [domainsByName setObject: domain forKey: name];
        [domain release];
        if (shouldNotify)
            [self locked_didChange];
    }
    [domainLock unlock];
    
    return domain;
}

- (void)addCookie: (OWCookie *) cookie andNotify: (BOOL) shouldNotify;
{
    OWCookiePath *path;
    
    [domainLock lock];
    path = [self locked_pathNamed: [cookie path] shouldCreate: YES];
    [path addCookie: cookie andNotify: shouldNotify];
    [domainLock unlock];
}

- (OWCookiePath *) locked_pathNamed: (NSString *) pathName shouldCreate: (BOOL) shouldCreate;
{
    unsigned int pathIndex;
    OWCookiePath *path;
    
    [domainLock lock];

    pathIndex = [_cookiePaths count];
    while (pathIndex--) {
        path = [_cookiePaths objectAtIndex: pathIndex];
        if ([[path path] isEqualToString: pathName]) {
            [path retain];
            goto found;
        }
    }

    if (shouldCreate) {
        path = [[OWCookiePath alloc] initWithPath: pathName];
        [_cookiePaths insertObject: path inArraySortedUsingSelector: @selector(compare:)];
    } else
        path = nil;

found:
    [domainLock unlock];
    
    return [path autorelease];
}

+ (unsigned int)minimumDomainComponentsForTopLevelDomain:(NSString *)aTopLevelDomain;
{
    return [shortTopLevelDomains containsObject:aTopLevelDomain] ? 2 : 3;
}

+ (NSArray *)searchDomainsForDomain:(NSString *)aDomain;
{
    NSMutableArray *searchDomains;
    NSMutableArray *domainComponents;
    unsigned int domainComponentCount;
    unsigned int minimumDomainComponents;

    domainComponents = [[aDomain componentsSeparatedByString:@"."] mutableCopy];
    domainComponentCount = [domainComponents count];
    minimumDomainComponents = [self minimumDomainComponentsForTopLevelDomain:[domainComponents lastObject]];
    searchDomains = [NSMutableArray arrayWithCapacity:domainComponentCount];
    [searchDomains addObject:aDomain];
    if (domainComponentCount < minimumDomainComponents) {
        [domainComponents release];
	return searchDomains;
    }
    domainComponentCount -= minimumDomainComponents;
    while (domainComponentCount--) {
	NSString *searchDomain;

	[domainComponents removeObjectAtIndex:0];
	searchDomain = [domainComponents componentsJoinedByString:@"."];
	[searchDomains addObject:[@"." stringByAppendingString:searchDomain]];
	[searchDomains addObject:searchDomain];
    }
    [domainComponents release];
    return searchDomains;
}

+ (OWCookie *)cookieFromHeaderValue:(NSString *)headerValue defaultDomain:(NSString *)defaultDomain defaultPath:(NSString *)defaultPath;
{
    NSString *aName, *aValue;
    NSDate *aDate = nil;
    NSString *aDomain = defaultDomain, *aPath = defaultPath;
    BOOL isSecure = NO;
    NSScanner *scanner;
    NSString *aKey;

    scanner = [NSScanner scannerWithString:headerValue];
    if (![scanner scanUpToCharactersFromSet:endNameSet intoString:&aName] ||
	![scanner scanString:@"=" intoString:NULL] ||
	![scanner scanUpToCharactersFromSet:endValueSet intoString:&aValue])
	return nil;
    [scanner scanCharactersFromSet:endKeySet intoString:NULL];
    while ([scanner scanUpToCharactersFromSet:endKeySet intoString:&aKey]) {
	aKey = [aKey lowercaseString];
	[scanner scanString:@"=" intoString:NULL];
	if ([aKey isEqualToString:@"expires"]) {
	    NSString *dateString = nil;

	    [scanner scanUpToCharactersFromSet:endDateSet intoString:&dateString];
            if (dateString) {
                aDate = [NSDate dateWithHTTPDateString:dateString];
                if (!aDate) {
                    NSCalendarDate *yearFromNowDate;

                    NSLog(@"OWCookie: could not parse expiration date, expiring cookie in one year");
                    yearFromNowDate = [[NSCalendarDate calendarDate] dateByAddingYears:1 months:0 days:0 hours:0 minutes:0 seconds:0];
                    [yearFromNowDate setCalendarFormat:[OWHTTPSession preferredDateFormat]];
                    aDate = yearFromNowDate;
                }
            }
	} else if ([aKey isEqualToString:@"domain"]) {
	    [scanner scanUpToCharactersFromSet:endValueSet intoString:&aDomain];
            if (aDomain != nil) {
                NSMutableArray *domainComponents;
                unsigned int domainComponentCount;

                domainComponents = [[aDomain componentsSeparatedByString:@"."] mutableCopy];
                domainComponentCount = [domainComponents count];
                if (domainComponentCount > 0 && [[domainComponents objectAtIndex:0] isEqualToString:@""]) {
                    // ".co.uk" -> ("", "co", "uk"):  we shouldn't count that initial empty component
                    domainComponentCount--;
                }
                
                if (OWCookiesDebug)
                    NSLog(@"COOKIES: domainComponents = %@, minimum = %d", domainComponents, [self minimumDomainComponentsForTopLevelDomain:[domainComponents lastObject]]);
                    
                if (domainComponentCount < [self minimumDomainComponentsForTopLevelDomain:[domainComponents lastObject]]) {
                    // Sorry, you can't create cookies for "com" or "co.uk"
                    aDomain = defaultDomain;
                }
            }
	} else if ([aKey isEqualToString:@"path"]) {
	    [scanner scanUpToCharactersFromSet:endValueSet intoString:&aPath];
	} else if ([aKey isEqualToString:@"secure"]) {
	    isSecure = YES;
	}
	[scanner scanCharactersFromSet:endKeySet intoString:NULL];
    }
    
    return [[[OWCookie alloc] initWithDomain:aDomain path:aPath name:aName value:aValue expirationDate:aDate secure:isSecure] autorelease];
}

- (void) locked_addApplicableCookies: (NSMutableArray *) cookies
                             forPath: (NSString *) aPath
                         urlIsSecure: (BOOL) secure;
{
    unsigned int pathIndex;
    OWCookiePath *path;
    
    pathIndex = [_cookiePaths count];
    while (pathIndex--) {
        path = [_cookiePaths objectAtIndex: pathIndex];
        if (![path appliesToPath: aPath])
            continue;
        
        [path addNonExpiredCookiesToArray: cookies usageIsSecure: secure];
    }
}
    
+ (BOOL)locked_readOW3Cookies;
{
    NSMutableDictionary *registrationDict;
    NSEnumerator *domainEnumerator;
    NSString *aDomain;
    BOOL success;
    NSString *cookieFilename;
    
    cookieFilename = [self cookiePath: OW3CookieFileName];
    
    if (!cookieFilename)
        return NO;

    [domainLock lock];
    
    NS_DURING {
        registrationDict = [[NSMutableDictionary alloc] initWithContentsOfFile:cookieFilename];
        domainEnumerator = [registrationDict keyEnumerator];
        while ((aDomain = [domainEnumerator nextObject])) {
            NSString *aPath;
            NSEnumerator *pathEnumerator;
            NSMutableDictionary *domainDict;

            domainDict = [registrationDict objectForKey:aDomain];
            pathEnumerator = [domainDict keyEnumerator];
            while ((aPath = [pathEnumerator nextObject])) {
                NSString *aName;
                NSEnumerator *nameEnumerator;
                NSMutableDictionary *pathDict;

                // Convert old style immunizations to new domain behaviors
                if ([aPath isEqualToString: OWCookieImmunizedDomainPath]) {
                    OWCookieDomain *domain;
                    
                    domain = [self domainNamed: aDomain behavior: OWCookieDomainDefaultBehavior andNotify: NO];
                    [domain setInternalBehavior: OWCookieDomainRejectBehavior];
                    continue;
                }
                
                pathDict = [domainDict objectForKey:aPath];
                nameEnumerator = [pathDict keyEnumerator];
                while ((aName = [nameEnumerator nextObject])) {
                    OWCookie *cookie;
                    OWCookieDomain *domain;
                    NSString *value, *expirationString;
                    NSDate *date;
                    BOOL secure;
                    NSDictionary *dict;
                    
                    // Don't notify on this registration since we aren't dirty here.
                    // We've inlined the OW3 cookie unarchiving logic here which is
                    // somewhat different from the current stuff (which is in OWCookie itself).
                    dict = [pathDict objectForKey:aName];
                    value = [dict objectForKey: @"value"];
                    expirationString = [dict objectForKey:@"expirationTime"];
                    
                    if (expirationString) {
                        NSTimeInterval timeInterval;
                
                        timeInterval = [expirationString doubleValue];
                        date = [NSCalendarDate dateWithTimeIntervalSinceReferenceDate:timeInterval];
                    } else {
                        // Cookies with no specified expiration were supposed to have
                        // expired before being saved.
                        continue;
                    }
                    
                    secure = [[dict objectForKey:@"secure"] isEqualToString:@"YES"];
                    cookie = [[OWCookie alloc] initWithDomain: aDomain
                                                         path: aPath
                                                         name: aName
                                                        value: value
                                               expirationDate: date
                                                       secure: secure];
                                        
                    domain = [self domainNamed: [cookie domain]];
                    if (OWCookiesDebug)
                        NSLog(@"COOKIES: Reading OW3, Domain = %@, adding cookie = %@", [cookie domain], cookie);
                    [domain addCookie: cookie andNotify: NO];
                    [cookie release];
                }
            }
        }
        success = YES;
    } NS_HANDLER {
        // No saved cookies
        registrationDict = nil;
        success = NO;
    } NS_ENDHANDLER;
    [registrationDict release];
    
    [domainLock unlock];
    
    return success;
}

//
// OW4.x XML Cookie file parsing
//


typedef struct _OW4CookieParseInfo {
    OWCookieDomain *domain;
    NSTimeInterval  parseTime;
} OW4CookieParseInfo;



// Under DP4 CFXMLParser does not resolve character entities (like &#60;).  This
// means that it is not easy to read and write files that contain characters that
// are supposed to be quoted --  &<>"' 
// We could post process the strings read from CFXMLParser to resolve the quoted
// characters, but for now I'll just use expat (in the OmniExpat framework).
//
// There is CFXMLParser-based implmenetation of locked_readOW4Cookies in version 1.1 of this file.

void startElement(void *userData,
                  const XML_Char *name,
		  const XML_Char **atts)
{
#if 0
    unsigned int attIndex = 0;
    
    printf("start %s", name);
    while (atts[attIndex])
        printf(" %s", atts[attIndex++]);
    printf("\n");
#endif

    if (!strcmp(name, "domain")) {
        OWCookieDomain *domain;
        OWCookieDomainBehavior behavior = OWCookieDomainDefaultBehavior;
        NSString *name = nil;
        
        while (*atts) {
            if (!strcmp(*atts, "name"))
                name = [[NSString alloc] initWithUTF8String: *(++atts)];
            else if (!strcmp(*atts, "behavior"))
                behavior = _behaviorForCString(*(++atts));
            atts++;
        }
        
        if (!name)
            // Invalid
            domain = nil;
        else
            domain = [OWCookieDomain domainNamed: name behavior: behavior andNotify: NO];
            
        [name release];
        ((OW4CookieParseInfo *)userData)->domain = domain;
    } else if (!strcmp(name, "cookie")) {
        OWCookie *cookie;
        OWCookieDomain *domain;
        NSString *path = OWCookieGlobalPath, *name = nil, *value = nil;
        NSTimeInterval expireInterval = distantPastInterval;
        NSDate *expires;
        BOOL secure = NO;
        
        domain = ((OW4CookieParseInfo *)userData)->domain;
        if (!domain)
            // Cookie elements must be inside a domain element
            return;
            
        while (*atts) {
            if (!strcmp(*atts, "name"))
                name = [[NSString alloc] initWithUTF8String: *(++atts)];
            else if (!strcmp(*atts, "path"))
                path = [[NSString alloc] initWithUTF8String: *(++atts)];
            else if (!strcmp(*atts, "value"))
                value = [[NSString alloc] initWithUTF8String: *(++atts)];
            else if (!strcmp(*atts, "expires"))
                expireInterval = atof(*(++atts));
            else if (!strcmp(*atts, "secure"))
                secure = strcmp(*(++atts), "YES") == 0;
            atts++;
        }
        
        // Cookies without names are invalid.  Also, skip cookies that have expired.
        if (name && expireInterval > ((OW4CookieParseInfo *)userData)->parseTime) {
            expires = [[NSDate alloc] initWithTimeIntervalSinceReferenceDate: expireInterval];
            cookie = [[OWCookie alloc] initWithDomain: [domain name]
                                                 path: path
                                                 name: name
                                                value: value
                                       expirationDate: expires
                                               secure: secure];

            [name release];
            [path release];
            [value release];
            [expires release];

            [domain addCookie: cookie andNotify: NO];
            [cookie release];
        }
        
    } else {
        // Ignore it
    }
}

void endElement(void *userData,
                const XML_Char *name)
{
#if 0
    printf("end %s\n", name);
#endif
    
    if (!strcmp(name, "domain"))
        ((OW4CookieParseInfo *)userData)->domain = nil;
}

+ (BOOL) locked_readOW4Cookies;
{
    NSString *filename;
    NSData *xmlData;
    XML_Parser parser;
    OW4CookieParseInfo info;
    
    filename = [self cookiePath: OW4CookieFileName];
    xmlData = [[[NSData alloc] initWithContentsOfFile: filename] autorelease];
    if (!xmlData)
        return NO;
        
    memset(&info, 0, sizeof(info));
    info.parseTime = [NSDate timeIntervalSinceReferenceDate];
    
    parser = XML_ParserCreate("UTF-8");
    XML_SetElementHandler(parser, startElement, endElement);
    XML_SetUserData(parser, &info);
    
    // Pass the entire file in one buffer, passing '1' to indicate that this is the last buffer
    if (!XML_Parse(parser, [xmlData bytes], [xmlData length], 1)) {
        NSLog(@"OWCookieDomain XML Parse: XML_Parse returned %d, %s, Line: %d, Column: %d",
                XML_GetErrorCode(parser),
                XML_ErrorString(XML_GetErrorCode(parser)),
                XML_GetCurrentLineNumber(parser),
                XML_GetCurrentColumnNumber(parser));
        return NO;
    }
    
    return YES;
}


- (id) initWithDomain: (NSString *) domain behavior: (OWCookieDomainBehavior) behavior;
{
    _promptLock = [[NSLock alloc] init];
    _name = [domain copy];
    _cookiePaths = [[NSMutableArray alloc] init];
    _behavior = behavior;
    
    return self;
}

- (void) dealloc;
{
    [_promptLock release];
    [_name release];
    [_cookiePaths release];
    [super dealloc];
}

- (BOOL) promptForCookie: (OWCookie *) cookie pipeline: (OWPipeline *) pipeline;
{
    OWCookiePromptResponse response;
    BOOL shouldAccept;
    
    if (!classDelegate) {
        if (OWCookiesDebug)
            NSLog(@"COOKIES: No delegate -- will accept");
        return YES;
    }
    
    // It is important that we are NOT locked when executing this
    // callout since this method could block for a while and we want
    // other processors to be able to continue.
    // BUT, we do want to ensure that no other prompts for this
    // particular domain are processed while we are already prompting
    // This would annoy the user.
    [_promptLock lock];
    
    // Check the behavior again now that we have the prompt lock.
    switch ([self behavior]) {
        case OWCookieDomainAcceptBehavior:
            [_promptLock unlock];
            return YES;
        case OWCookieDomainRejectBehavior:
            [_promptLock unlock];
            return NO;
        case OWCookieDomainPromptBehavior:
        default:
            // Still need to prompt
            break;
    }
    
    NS_DURING {
        response = [classDelegate cookieDomain: self
                       promptResponseForCookie: cookie
                                  fromPipeline: pipeline];
    } NS_HANDLER {
        NSLog(@"Ignoring exception raised while prompting for cookie behavior: %@", localException);
    } NS_ENDHANDLER;
    
    switch (response) {
        case OWCookiePromptRejectResponse:
            shouldAccept = NO;
            break;
        case OWCookiePromptAlwaysRejectResponse:
            shouldAccept = NO;
            [self setInternalBehavior: OWCookieDomainRejectBehavior];
            break;
        case OWCookiePromptAlwaysAcceptResponse:
            shouldAccept = YES;
            [self setInternalBehavior: OWCookieDomainAcceptBehavior];
            break;
        default:
        case OWCookiePromptAcceptResponse:
            shouldAccept = YES;
            break;
    }
    
    if (OWCookiesDebug)
        NSLog(@"COOKIES: Delegate says to %@ (%d)", shouldAccept ? @"accept" : @"reject", response);
        
    // Don't unlock until we've actually set the behavior.
    [_promptLock unlock];
    return shouldAccept;
}


- (NSMutableDictionary *) debugDictionary;
{
    NSMutableDictionary *dict;
    
    dict = [super debugDictionary];
    [dict setObject: _name forKey: @"name"];
    [dict setObject: [NSString stringWithCString: _stringForBehavior(_behavior)] forKey: @"behavior"];
    [dict setObject: _cookiePaths forKey: @"cookiePaths"];
    
    return dict;
}

@end

