// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWCSSStyleSheet.h"

#import <OmniBase/OmniBase.h>
#import <Foundation/Foundation.h>
#import <OmniFoundation/OmniFoundation.h>

#import "OWAddress.h"
#import "OWContentType.h"
#import "OWCSSSelector.h"
#import "OWCSSSelectorGroup.h"
#import "OWDataStream.h"
#import "OWDataStreamCharacterProcessor.h"
#import "OWObjectStreamCursor.h"
#import "OWSGMLProcessor.h"
#import "OWSGMLTag.h"
#import "OWSimpleTarget.h"
#import "OWWebPipeline.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/CSS.subproj/OWCSSStyleSheet.m,v 1.35 2000/12/23 00:03:05 corwin Exp $")

@implementation OWCSSStyleSheet

static OWContentType *cssContentType;
static CSBitmap whitespaceBitmap;
static CSBitmap semicolonAndWhitespaceBitmap;
static CSBitmap importDelimiterBitmap;
static CSBitmap selectorDelimiterBitmap;
static CSBitmap classTypeDelimiterBitmap;
static CSBitmap colonAndWhitespaceBitmap;


+ (void)initialize;
{
    static BOOL alreadyInitialized = NO;
    NSMutableCharacterSet *classTypeDelimiterSet;

    [super initialize];
    if (alreadyInitialized)
        return;
    alreadyInitialized = YES;

    cssContentType = [OWContentType contentTypeForString:@"text/css"];
    whitespaceBitmap = bitmapForCharacterSetDoRetain([NSCharacterSet whitespaceAndNewlineCharacterSet], YES);
    importDelimiterBitmap = bitmapForCharacterSetDoRetain([NSCharacterSet 	characterSetWithCharactersInString:@"\"("], YES);
    selectorDelimiterBitmap = bitmapForCharacterSetDoRetain([NSCharacterSet 	characterSetWithCharactersInString:@",{"], YES);
    semicolonAndWhitespaceBitmap = bitmapForCharacterSetDoRetain([NSCharacterSet 	characterSetWithCharactersInString:@"; \t\n\r"], YES);
    colonAndWhitespaceBitmap = bitmapForCharacterSetDoRetain([NSCharacterSet characterSetWithCharactersInString:@": \t\n\r"], YES);
    classTypeDelimiterSet = [[NSCharacterSet whitespaceAndNewlineCharacterSet] mutableCopy];
    [classTypeDelimiterSet addCharactersInString:@".#:"];
    classTypeDelimiterBitmap = bitmapForCharacterSetDoRetain(classTypeDelimiterSet, YES);
    [classTypeDelimiterSet release];
}

- init;
{        
    processedStyleSheets = [[NSMutableArray alloc] init];
    // Tags and pseudoclasses are NOT case sensitive, classes and IDs are.
    lastSelectorsByTag = OFCreateCaseInsensitiveKeyMutableDictionary();
    lastSelectorsByClass = [[NSMutableDictionary alloc] init];
    lastSelectorsByID = [[NSMutableDictionary alloc] init];
    lastSelectorsByPseudoclass = OFCreateCaseInsensitiveKeyMutableDictionary();
    currentPosition = 0;
    return self;
}

- initWithSGMLProcessor:(OWSGMLProcessor *)proc;
{
    [self init];
    
    nonretained_processor = proc;
    baseAddress = [[nonretained_processor baseAddress] retain];
    
    return self;
}

- (void)dealloc;
{
    [processedStyleSheets release];
    [lastSelectorsByTag release];
    [lastSelectorsByClass release];
    [lastSelectorsByID release];
    [lastSelectorsByPseudoclass release];
    [baseAddress release];
    [super dealloc];
}

- (void)processStyleSheetContent:(NSString *)styleSheet
{
    [processedStyleSheets addObject:styleSheet];
    [self parseStyleSheet:styleSheet];
}

- (void)parseStyleSheet:(NSString *)sheet;
{
    OFStringScanner *sheetScanner;
    BOOL importsAllowed = YES;

    if (sheet == nil || [sheet length] == 0)
        return; // Let's avoid all this if we have no style sheet

//#define SAVE_SHEETS
#ifdef SAVE_SHEETS
    NSLog(@"Saving style sheet '%@' to /tmp/last-sheet", sheet);
    [sheet writeToFile: @"/tmp/last-sheet" atomically:YES];
#endif

#warning Rather than removing comments from the style sheet string and then parsing it, we should skip comments as we parse
    sheetScanner = [[OFStringScanner alloc] initWithString:[self removeComments:sheet]];
        
    // NSLog(@"sheet: '%@'", [self removeComments:sheet]);
    while (scannerScanUpToCharacterNotInCSBitmap(sheetScanner, whitespaceBitmap)) {
        if (scannerPeekCharacter(sheetScanner) == '@') {
            NSString *type;
            scannerSkipPeekedCharacter(sheetScanner);
            type = [sheetScanner readFullTokenWithDelimiterCSBitmap:whitespaceBitmap];
            if ([type compare:@"import" options:NSCaseInsensitiveSearch] == NSOrderedSame) {
                NSString *urlString = @"";

                // NSLog(@"Hey, got an import at-rule");
                if (importsAllowed) {
                    if (scannerScanUpToCharacterInCSBitmap(sheetScanner, importDelimiterBitmap)) {
                        unichar delimiter = scannerReadCharacter(sheetScanner);
                        NSString *importedScript;
                        if (delimiter == '"') {
                            urlString = [sheetScanner readFullTokenWithDelimiterCharacter:(unichar)'"'];
                        } else if (delimiter == '(') {
                            urlString = [sheetScanner readFullTokenWithDelimiterCharacter:(unichar)')'];
                        }
                        // NSLog(@"urlString = '%@', go get it.", urlString);
                        importedScript = [self getStyleSheetFromURLString:urlString];
                        if (importedScript)
                            [self parseStyleSheet:importedScript];
                    } else {
                        // NSLog(@"Import was poorly formed, or I'm stupid.");
                    }
                    // NSLog(@"Not only that, I've been instructed to import '%@'", urlString);
                } else {
                    // NSLog(@"Too bad imports aren't allowed now.");
                }
            } else {
                // NSLog(@"Hey, got some other at-rule: '%@'", type);
                // NSLog(@"I'll just blow it away");
            }
            scannerScanUpToCharacter(sheetScanner, (unichar)';');
            scannerSkipPeekedCharacter(sheetScanner);
        } else {
            NSString *declaration;
            NSDictionary *declarationDict;
            NSString *selectors = @"";
            BOOL moreSelectorGroups = YES;
            unsigned int beforeSelectorLocation = scannerScanLocation(sheetScanner);
            unsigned int afterDeclaration = 0;
            scannerScanUpToCharacter(sheetScanner, (unichar)'{');
            scannerSkipPeekedCharacter(sheetScanner);
            declaration = [sheetScanner readFullTokenWithDelimiterCharacter:(unichar)'}'];
            scannerSkipPeekedCharacter(sheetScanner);
            afterDeclaration = scannerScanLocation(sheetScanner);
            [sheetScanner setScanLocation:beforeSelectorLocation];
            // NSLog(@"directives: '%@'", declaration);
            importsAllowed = NO;
            declarationDict = [OWCSSStyleSheet parseDeclaration:declaration];
            while (moreSelectorGroups && (selectors = [sheetScanner readFullTokenWithDelimiterCSBitmap:selectorDelimiterBitmap])) {
                NSArray *selectorList;
                OWCSSSelectorGroup *group;
                
                if (scannerReadCharacter(sheetScanner) == '{') {
                    moreSelectorGroups = NO;
                }
                selectorList = [self parseSelectorString:selectors];
                group = [[OWCSSSelectorGroup alloc] initWithSelectorList:selectorList withDeclarations:declarationDict withPosition:currentPosition++];
                [self addSelectorGroup:group];
                [group release];
            }
            [sheetScanner setScanLocation:afterDeclaration];
        }
    }
#if 1
    //NSLog(@"done with style sheet.");
    //NSLog(@"description of lastSelectorsByTag:'%@'", lastSelectorsByTag);
    //NSLog(@"description of lastSelectorsByClass:'%@'", lastSelectorsByClass);
    //NSLog(@"description of lastSelectorsByID:'%@'", lastSelectorsByID);
    //NSLog(@"description of lastSelectorsByPseudoClass:'%@'", lastSelectorsByPseudoclass);
#endif
    [sheetScanner release];
}

- (NSString *)getStyleSheetFromURLString:(NSString *)urlString;
{
    OWAddress *sourceAddress;
    OWSimpleTarget *fetchTarget;
    id <OWContent> fetchedContent;
    OWDataStream *scriptDataStream;
    NSData *scriptData;
    NSString *scriptString;

    if (urlString == nil)
        return nil;

    sourceAddress = [baseAddress addressForRelativeString:urlString];
    if (sourceAddress == nil)
        return nil;

    fetchTarget = [[OWSimpleTarget alloc] initWithParentContentInfo:[[nonretained_processor pipeline] contentInfo] targetContentType:cssContentType];
    [fetchTarget setAcceptsAlternateContent:YES];
    [OWWebPipeline startPipelineWithContent:sourceAddress target:fetchTarget];
    fetchedContent = [fetchTarget resultingContent];
    [fetchTarget release];
    if (![fetchedContent isKindOfClass:[OWDataStream class]])
        return nil;

    scriptDataStream = (OWDataStream *)fetchedContent;
    scriptData = [[scriptDataStream newCursor] readAllData];
#warning We should obey the string encoding of our style sheet
    // We should really obey the charset parameter of the source content type rather than using the default string encoding.  Better yet, we should read from a character stream rather than making this all a big string right now.
    scriptString = [NSString stringWithData:scriptData encoding:[OWDataStreamCharacterProcessor defaultStringEncoding]];
    return scriptString;
}

- (NSArray *)parseSelectorString:(NSString *)selectors;
{
    NSMutableArray *result;
    OFStringScanner *selectorScanner;
    
    result = [NSMutableArray array];
    selectorScanner = [[OFStringScanner alloc] initWithString:selectors];
        
    while (scannerScanUpToCharacterNotInCSBitmap(selectorScanner, whitespaceBitmap)) {
        OWCSSSelector *selector;
        NSString *token;
        unichar delimiterCharacter;
        
        token = [selectorScanner readFullTokenWithDelimiterCSBitmap:classTypeDelimiterBitmap];
/*
        if (! [token isEqualToString:@""]) {
            NSLog(@"Hey, this rule applies to selector '%@' and the following:", temp);
        } else {
            NSLog(@"Hey, this rule applies to selectors with the following:");
        }
*/
        selector = [[OWCSSSelector alloc] init];
        [selector setTagName:[token lowercaseString]];
        
        delimiterCharacter = scannerReadCharacter(selectorScanner);
        while (delimiterCharacter != OFCharacterScannerEndOfDataCharacter && !characterIsMemberOfCSBitmap(whitespaceBitmap, delimiterCharacter)) {
            token = [selectorScanner readFullTokenWithDelimiterCSBitmap:classTypeDelimiterBitmap];
            switch (delimiterCharacter) {
                case '.':
                    // NSLog(@"\ta class called '%@'", temp);
                    [selector setClassName:token];
                    break;
                case '#':
                    // NSLog(@"\tan ID called '%@'", temp);
                    [selector setIDName:token];
                    break;
                case ':':
                    // NSLog(@"\ta pseudoclass called '%@'", temp);
                    [selector setPseudoClassName:token];
                    break;
                default:
                    // NSLog(@"\t??? mysterio state.");
                    OBASSERT(NO /* not reached */);
                    break;
            }
            delimiterCharacter = scannerReadCharacter(selectorScanner);
        }
        [result addObject:selector];
        [selector release];
    }
    
    [selectorScanner release];
    
    return result;
}

// Removes C-style comments, and removes SGML comment markers (but not the text between them)
- (NSString *)removeComments:(NSString *)sourceString;
{
    NSMutableString *postProcessed;
    int lastLocation;
    BOOL scanned;
    OFStringScanner *commentScanner;
    
    postProcessed = [[NSMutableString alloc] initWithCapacity:[sourceString length]];
    [postProcessed autorelease];
    
    // First scan for and remove any C-style comments in the string, copying the uncommented portions into the output buffer (postProcessed)
    commentScanner = [[OFStringScanner alloc] initWithString:sourceString];
    lastLocation = 0;
    for (;;) {
        scanned = [commentScanner scanUpToString:@"/*"];
        [postProcessed appendString:[sourceString substringWithRange:NSMakeRange(lastLocation, scannerScanLocation(commentScanner) - lastLocation)]];
        if (!scanned)
            break;
        [commentScanner scanString:@"/*" peek:NO];
        [commentScanner scanUpToString:@"*/"];
        [commentScanner scanString:@"*/" peek:NO];
        lastLocation = scannerScanLocation(commentScanner);
    }
    
    [commentScanner release];

    // Remove any SGML-style comment markers from the string. This is somewhat inefficient in the general case, but in the common case (one marker at the beginning, one at the end) it should be nearly as efficient as a cleverer algorithm.
    
    // The exact semantics of this are a little odd; I'm duplicating the effect of Corwin's algorithm.
    lastLocation = 0;
    for (;;) {
        NSRange foundRange = [postProcessed rangeOfString:@"<!--" options:0 range:NSMakeRange(lastLocation, [postProcessed length] - lastLocation)];
        if (foundRange.length == 0)
            break;
        lastLocation = foundRange.location;
        [postProcessed deleteCharactersInRange:foundRange];
        
        foundRange = [postProcessed rangeOfString:@"-->" options:0 range:NSMakeRange(lastLocation, [postProcessed length] - lastLocation)];
        if (foundRange.length > 0) {
            [postProcessed deleteCharactersInRange:foundRange];
        }
    }

    return postProcessed;
}

+ (NSDictionary *)parseDeclaration:(NSString *)declaration;
{
#warning +parseDeclaration: currently ignores things like "!important"
    NSMutableDictionary *result;
    OFStringScanner *declarationScanner;
    
    result = [NSMutableDictionary dictionary];
    declarationScanner = [[OFStringScanner alloc] initWithString:declaration];
    while (scannerHasData(declarationScanner)) {
        NSString *property;
        NSString *value;
        
        if (!scannerScanUpToCharacterNotInCSBitmap(declarationScanner, semicolonAndWhitespaceBitmap))
            break; // returns NO on failure, eg. EOF
            
        property = [declarationScanner readFullTokenWithDelimiterCSBitmap:colonAndWhitespaceBitmap forceLowercase:YES];
        
        if (!scannerScanUpToCharacterNotInCSBitmap(declarationScanner, colonAndWhitespaceBitmap))
            break; // badly-formed declaration...

        value = [declarationScanner readFullTokenWithDelimiterCSBitmap:semicolonAndWhitespaceBitmap];

        // NSLog(@"\tdirective: '%@', args: '%@'", property, value);
        if (value != nil && property != nil)
            [result setObject:value forKey:property];
    }
    [declarationScanner release];
    return result;
}

- (void)addSelectorGroup:(OWCSSSelectorGroup *)group;
{
    OWCSSSelector *lastSelector = [group lastSelector];
    NSString *key;
    NSMutableArray *list;
    
    // NSLog(@"group = '%@'", group);
    if ((key = [lastSelector pseudoClassName]) && [key length]) {
        if (! (list = [lastSelectorsByPseudoclass objectForKey:key])) {
            // NSLog(@"adding '%@' to the pseudoclass list", key);
            list = [[NSMutableArray alloc] init];
            [lastSelectorsByPseudoclass setObject:list forKey:key];
            [list release];
        }
    } else if ((key = [lastSelector tagName]) && [key length]) {
        if (!(list = [lastSelectorsByTag objectForKey:key])) {
            // NSLog(@"adding '%@' to the tag list", key);
            list = [[NSMutableArray alloc] init];
            [lastSelectorsByTag setObject:list forKey:key];
            [list release];
        }
    } else if ((key = [lastSelector className]) && [key length]) {
        if (!(list = [lastSelectorsByClass objectForKey:key])) {
            // NSLog(@"adding '%@' to the class list", key);
            list = [[NSMutableArray alloc] init];
            [lastSelectorsByClass setObject:list forKey:key];
            [list release];
        }
    } else if ((key = [lastSelector IDName]) && [key length]) {
        if (!(list = [lastSelectorsByID objectForKey:key])) {
            // NSLog(@"adding '%@' to the ID list", key);
            list = [[NSMutableArray alloc] init];
            [lastSelectorsByID setObject:list forKey:key];
            [list release];
        }
    } else {
        // NSLog(@"I see an empty selector. Discarding.");
        return;
    }

    [list addObject:group];
    [list sortUsingSelector:NSSelectorFromString(@"compare:")];
    //NSLog(@"list = '%@'", [list description]);
}

- (BOOL)checkTagStack:(NSArray *)tagStack forSelectorGroup:(OWCSSSelectorGroup *)group;
{
    NSArray *selectorList = [group selectorList];
    NSEnumerator *selectorEnumerator = [selectorList reverseObjectEnumerator];
    OWCSSSelector *selector;
    NSEnumerator *tagEnumerator = [tagStack reverseObjectEnumerator];
    // Skip the first one, since if we're here it matches the last tag.
    [selectorEnumerator nextObject];
    [tagEnumerator nextObject];
    while ((selector = [selectorEnumerator nextObject])) {
        // technically, this could be lazier.
        NSString *tagName = [selector tagName];
        NSString *className = [selector className];
        NSString *IDName = [selector IDName];
        OWSGMLTag *tag;
        BOOL stillChecking = YES;
        while (stillChecking && (tag = [tagEnumerator nextObject])) {
            if (! [tagName isEqualToString:@""]) {
                stillChecking = ! [tagName isEqualToString:[tag name]];
#if 0
            NSLog(@"looking for '%@', got '%@'", tagName, [tag name]);
#endif
            }
            if (! [className isEqualToString:@""]) {
                stillChecking = ! [className isEqualToString:[tag valueForAttribute:@"class"]];
            }
            if (! [IDName isEqualToString:@""]) {
                stillChecking = ! [IDName isEqualToString:[tag valueForAttribute:@"id"]];
            }
        }
        if (stillChecking) {
            return NO;
        }
    }
    return YES;
}

- (NSDictionary *)findDeclarationsForTag:(NSString *)name withStack:(NSArray *)tagStack;
{
    NSArray *matchingGroups;

    matchingGroups = [lastSelectorsByTag objectForKey:name];
    // NSLog(@"checking to find '%@'", name);
    if (matchingGroups != nil) {
        unsigned int groupIndex, groupCount;
        NSMutableDictionary *result = [NSMutableDictionary dictionary];

        groupCount = [matchingGroups count];
        for (groupIndex = 0; groupIndex < groupCount; groupIndex++) {
            OWCSSSelectorGroup *group;
            OWCSSSelector *lastSelector;
            OWSGMLTag *lastTag;

            group = [matchingGroups objectAtIndex:groupIndex];
            lastSelector = [group lastSelector];
            lastTag = [tagStack lastObject];
            // NSLog(@"tag = '%@'", lastTag);
            // NSLog(@"selector = '%@'", lastSelector);
#warning first part of conditional should be part of -checkTagStack:: method
            if ([lastSelector matchesTag:lastTag]  && (([group count] == 0) || [self checkTagStack:tagStack forSelectorGroup:group])) {
                [result addEntriesFromDictionary:[group declarations]];
            }
        }
        return result;
    } else {
        return nil;
    }
}

- (NSDictionary *)findDeclarationsForClass:(NSString *)name withStack:(NSArray *)tagStack;
{
    NSArray *matchingGroups;

    matchingGroups = [lastSelectorsByClass objectForKey:name];
    // NSLog(@"checking to find '%@'", name);
    if (matchingGroups != nil) {
        unsigned int groupIndex, groupCount;
        NSMutableDictionary *result = [NSMutableDictionary dictionary];

        groupCount = [matchingGroups count];
        for (groupIndex = 0; groupIndex < groupCount; groupIndex++) {
            OWCSSSelectorGroup *group;
            OWCSSSelector *lastSelector;
            OWSGMLTag *lastTag;

            group = [matchingGroups objectAtIndex:groupIndex];
            lastSelector = [group lastSelector];
            lastTag = [tagStack lastObject];
            // NSLog(@"tag = '%@'", lastTag);
            // NSLog(@"selector = '%@'", lastSelector);
#warning first part of conditional should be part of -checkTagStack:: method
            if ([lastSelector matchesTag:lastTag] && (([group count] == 0) || [self checkTagStack:tagStack forSelectorGroup:group])) {
                [result addEntriesFromDictionary:[group declarations]];
            }
        }
        return result;
    } else {
        return nil;
    }
}

- (NSDictionary *)findDeclarationsForID:(NSString *)name withStack:(NSArray *)tagStack;
{
    NSArray *matchingGroups;

    matchingGroups = [lastSelectorsByID objectForKey:name];
    //NSLog(@"checking to find '%@'", name);
    if (matchingGroups != nil) {
        unsigned int groupIndex, groupCount;
        NSMutableDictionary *result = [NSMutableDictionary dictionary];

        groupCount = [matchingGroups count];
        for (groupIndex = 0; groupIndex < groupCount; groupIndex++) {
            OWCSSSelectorGroup *group;
            OWCSSSelector *lastSelector;
            OWSGMLTag *lastTag;

            group = [matchingGroups objectAtIndex:groupIndex];
            lastSelector = [group lastSelector];
            lastTag = [tagStack lastObject];
            //NSLog(@"tag = '%@'", lastTag);
            //NSLog(@"selector = '%@'", lastSelector);
#warning first part of conditional should be part of -checkTagStack:: method
            if ([lastSelector matchesTag:lastTag] && (([group count] == 0) || [self checkTagStack:tagStack forSelectorGroup:group])) {
                [result addEntriesFromDictionary:[group declarations]];
            }
        }
        return result;
    } else {
        return nil;
    }
}

- (NSDictionary *)findDeclarationsForPseudoClass:(NSString *)name withStack:(NSArray *)tagStack;
{
    NSArray *matchingGroups;

    matchingGroups = [lastSelectorsByPseudoclass objectForKey:name];
    // NSLog(@"checking to find '%@'", name);
    if (matchingGroups != nil) {
        unsigned int groupIndex, groupCount;
        NSMutableDictionary *result = [NSMutableDictionary dictionary];

        groupCount = [matchingGroups count];
        for (groupIndex = 0; groupIndex < groupCount; groupIndex++) {
            OWCSSSelectorGroup *group;
            OWCSSSelector *lastSelector;
            OWSGMLTag *lastTag;

            group = [matchingGroups objectAtIndex:groupIndex];
            lastSelector = [group lastSelector];
            lastTag = [tagStack lastObject];
            // NSLog(@"tag = '%@'", lastTag);
            // NSLog(@"selector = '%@'", lastSelector);
#warning first part of conditional should be part of -checkTagStack:: method
            if ([lastSelector matchesTag:lastTag] && (([group count] == 0) || [self checkTagStack:tagStack forSelectorGroup:group])) {
                [result addEntriesFromDictionary:[group declarations]];
            }
        }
        return result;
    } else {
        return nil;
    }
}


@end
