//
// Copyright 2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.
//
//  Created by wiml on Wed Jul 26 2000.
//

#import <OWF/OWDataStreamCharacterProcessor.h>

#import <Foundation/Foundation.h>
#import <CoreFoundation/CFString.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OWF/OWDataStream.h>
#import <OWF/OWDataStreamCursor.h>
#import <OWF/OWDataStreamCharacterCursor.h>
#import <OWF/OWHeaderDictionary.h>
#import <OWF/OWParameterizedContentType.h>
#import <OWF/OWPipeline.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/Processors.subproj/OWDataStreamCharacterProcessor.m,v 1.8 2000/10/25 03:41:03 kc Exp $")

@implementation OWDataStreamCharacterProcessor

+ (CFStringEncoding)defaultStringEncoding
{
    CFStringEncoding cfEncoding;
    
    cfEncoding = [self stringEncodingForDefault:[[NSUserDefaults standardUserDefaults] stringForKey:@"OWIncomingStringEncoding"]];
    
    if (cfEncoding == kCFStringEncodingInvalidId)
        cfEncoding = kCFStringEncodingWindowsLatin1;  /* we bend over for microsoft */

    return cfEncoding;
}

+ (CFStringEncoding)stringEncodingForDefault:(NSString *)encodingName
{
    NSStringEncoding stringEncoding;
    CFStringEncoding cfEncoding;
    
    // Note that this default can be either a string or an integer. Integers refer to NSStringEncoding values. Strings consist of a prefix, a space, and a string whose meaning depends on the prefix. Currently understood prefixes are "ietf" (indicating an IETF charset name) and "cf" (indicating a CoreFoundation encoding number). Previously understood prefixes were the names of OWStringDocoder-conformant classes, but we don't do that any more.
    
    cfEncoding = kCFStringEncodingInvalidId;
    if ([encodingName hasPrefix:@"iana "]) {
        NSString *ietfName = [encodingName substringFromIndex:5];
        cfEncoding = CFStringConvertIANACharSetNameToEncoding((CFStringRef)ietfName);
    } else if ([encodingName hasPrefix:@"cf "]) {
        cfEncoding = [[encodingName substringFromIndex:3] intValue];
    }
    
    if (cfEncoding != kCFStringEncodingInvalidId)
        return cfEncoding;
    
    stringEncoding = [encodingName intValue];
    // Note that 0 is guaranteed never to be a valid encoding by the semantics of +[NSString availableStringEncodings].
    if (stringEncoding != 0)
        return CFStringConvertNSStringEncodingToEncoding(stringEncoding);
    
    return kCFStringEncodingInvalidId;
}

+ (NSString *)defaultForCFEncoding:(CFStringEncoding)anEncoding;
{
    NSString *encodingName;
    
    if (anEncoding == kCFStringEncodingInvalidId)
        return @"0";
        
    encodingName = (NSString *)CFStringConvertEncodingToIANACharSetName(anEncoding);
    if (encodingName != nil && ![encodingName hasPrefix:@"x-"] && ![encodingName hasPrefix:@"X-"])
        return [@"iana " stringByAppendingString:encodingName];
    
    return [NSString stringWithFormat:@"cf %d", anEncoding];
}

+ (CFStringEncoding)stringEncodingForIANACharSetName:(NSString *)charset;
{
    CFStringEncoding cfEncoding;
    
    if ([charset hasPrefix:@"iso8859-"]) {
        // Some web sites use iso8859 rather than iso-8859.
        charset = [@"iso-8859-" stringByAppendingString:[charset substringFromIndex:[@"iso8859-" length]]];
    }
    if ([charset hasPrefix:@"x-mac-cf-"]) {
        cfEncoding = [[charset substringFromIndex:9] intValue];
        if (!CFStringIsEncodingAvailable(cfEncoding))
            cfEncoding = kCFStringEncodingInvalidId;
    } else if ([charset isEqualToString:@"iso-8859-1"]) {
        // Windows-1252 (Windows Latin 1) is a superset of iso-8859-1 (ISO Latin 1), and unfortunately many web sites request iso-8859-1 when they actually use characters that only exist in windows-1252, like '\226'.
        cfEncoding = kCFStringEncodingWindowsLatin1;
    } else {
        cfEncoding = CFStringConvertIANACharSetNameToEncoding((CFStringRef)charset);
    }
    
    if (cfEncoding == kCFStringEncodingInvalidId)
        NSLog(@"Warning: Cannot convert charset \"%@\" to string encoding; using default encoding instead", charset);

    return cfEncoding;
}

+ (CFStringEncoding)stringEncodingForContentType:(OWParameterizedContentType *)aType;
{
    NSString *charset;
    
    if (aType == nil)
        return kCFStringEncodingInvalidId;
    
    charset = [[aType parameters] lastObjectForKey:@"charset"];
    if (charset != nil)
        return [self stringEncodingForIANACharSetName:charset];
    else
        return kCFStringEncodingInvalidId;
}

+ (NSString *)charsetForCFEncoding:(CFStringEncoding)anEncoding
{
    CFStringRef charsetName;
    
    if (anEncoding == kCFStringEncodingInvalidId)
        return nil;
        
    charsetName = CFStringConvertEncodingToIANACharSetName(anEncoding);
    if (charsetName)
        return (NSString *)charsetName;
    
    return [NSString stringWithFormat:@"x-mac-cf-%d", anEncoding];
}

// Init and dealloc

- initWithPipeline:(OWPipeline *)aPipeline;
{
    OWCursor *dataCursor;
    
    if (![super initWithPipeline:aPipeline])
        return nil;

    dataCursor = [(id <OWOptionalContent>)[pipeline lastContent] contentCursor];
    if (!dataCursor) {
        [self release];
        return nil;
    }
    OBASSERT([dataCursor isKindOfClass:[OWDataStreamCursor class]]);
    characterCursor = [[OWDataStreamCharacterCursor alloc] initForDataCursor:(OWDataStreamCursor *)dataCursor];

    return self;
}

- (void)dealloc;
{
    [characterCursor release];
    [super dealloc];
}

// OWProcessor subclass

- (void)abortProcessing;
{
    [characterCursor abort];
    [super abortProcessing];
}

// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    if (characterCursor)
        [debugDictionary setObject:characterCursor forKey:@"characterCursor"];

    return debugDictionary;
}

@end
