// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWHTTPSession.h"

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OmniNetworking/OmniNetworking.h>

#import "NSDate-OWExtensions.h"
#import "OWAddress.h"
#import "OWAuthorizationServer.h"
#import "OWContentType.h"
#import "OWParameterizedContentType.h"
#import "OWCookieDomain.h"
#import "OWCookie.h"
#import "OWDataStream.h"
#import "OWHeaderDictionary.h"
#import "OWHTTPProcessor.h"
#import "OWHTTPSessionQueue.h"
#import "OWNetLocation.h"
#import "OWSourceProcessor.h"
#import "OWTimeStamp.h"
#import "OWUnknownDataStreamProcessor.h"
#import "OWURL.h"
#import "OWWebPipeline.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/Processors.subproj/Protocols.subproj/HTTP.subproj/OWHTTPSession.m,v 1.71 2000/11/27 06:29:30 kc Exp $")

@interface OWHTTPSession (Private)

+ (NSString *)stringForHeader:(NSString *)aHeader value:(id)aValue;

// Generating request
- (NSString *)commandStringForAddress:(OWAddress *)anAddress;
- (NSString *)acceptHeadersStringForTarget:(id <OWTarget>)aTarget;
- (NSString *)acceptLanguageHeadersString;
- (NSString *)referrerHeaderStringForPipeline:(OWWebPipeline *)aPipeline;
- (NSString *)pragmaHeaderStringForPipeline:(OWWebPipeline *)aPipeline;
- (NSString *)hostHeaderStringForURL:(OWURL *)aURL;
- (NSString *)rangeStringForProcessor:(OWHTTPProcessor *)aProcessor;
- (NSString *)keepAliveString;
- (NSString *)cookiesForURL:(OWURL *)aURL;
- (NSString *)contentTypeHeaderStringForAddress:(OWAddress *)anAddress;
- (NSString *)contentLengthHeaderStringForAddress:(OWAddress *)anAddress;
- (NSString *)contentStringForAddress:(OWAddress *)anAddress;

// Reading results
- (BOOL)readResponse;
- (void)readBodyAndIgnore:(BOOL)ignoreThis;
- (BOOL)readHead;
- (void)readTimeStamp;
- (void)readHeaders;
- (unsigned int)intValueFromHexString:(NSString *)aString;
- (void)readChunkedBodyIntoStream:(OWDataStream *)dataStream;
- (void)readStandardBodyIntoStream:(OWDataStream *)dataStream;
- (void)readClosingBodyIntoStream:(OWDataStream *)dataStream;

// Exception handling
- (void)notifyProcessor:(OWHTTPProcessor *)aProcessor ofSessionException:(NSException *)sessionException;

@end

@implementation OWHTTPSession

static BOOL OWHTTPDebug = NO;
static NSMutableDictionary *languageAbbreviationsDictionary;
static NSString *preferredDateFormat;
static NSString *acceptLanguageString;
static NSString *http10VersionString;
static NSString *http11VersionString;
static NSString *endOfLineString;
static NSString *primaryUserAgentInfo = nil;
static NSString *userAgentInfo = nil;
static NSString *userAgentHeaderFormat = nil;
static NSMutableArray *languageArray = nil;
static NSCharacterSet *spaceCharacterSet;
static OWContentType *textPlainContentType;
static OWContentType *applicationOctetStreamContentType;

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
        return;
    initialized = YES;

    http10VersionString = @"HTTP/1.0";
    http11VersionString = @"HTTP/1.1";
    endOfLineString = @"\r\n";
    languageAbbreviationsDictionary = [[NSMutableDictionary alloc] initWithCapacity:8];
    spaceCharacterSet = [[NSCharacterSet characterSetWithCharactersInString:@" "] retain];
    textPlainContentType = [OWContentType contentTypeForString:@"text/plain"];
    applicationOctetStreamContentType = [OWContentType contentTypeForString:@"application/octet-stream"];
}

// OFBundleRegistryTarget informal protocol

+ (void)registerItemName:(NSString *)itemName bundle:(NSBundle *)bundle description:(NSDictionary *)description;
{
    if ([itemName isEqualToString:@"languageAbbreviations"]) {
        [languageAbbreviationsDictionary addEntriesFromDictionary:description];
    } else if ([itemName isEqualToString:@"dateFormats"]) {
        preferredDateFormat = [description objectForKey:@"preferredDateFormat"];
    }
}

+ (void)didLoad;
{
    [[OFController sharedController] addObserver: self];
}

+ (void)controllerDidInitialize:(NSNotification *)notification;
{
    NSDictionary *softwareVersionDictionary;
    NSString *primaryAgentKey, *primaryAgentVersion;

    softwareVersionDictionary = [OFBundleRegistry softwareVersionDictionary];
    primaryAgentKey = [[NSProcessInfo processInfo] processName];
    primaryAgentVersion = [softwareVersionDictionary objectForKey:primaryAgentKey];
    if (primaryAgentVersion && [primaryAgentVersion length] > 0)
        primaryUserAgentInfo = [[NSString stringWithFormat:@"%@/%@", primaryAgentKey, primaryAgentVersion] retain];
    else
        primaryUserAgentInfo = [primaryAgentKey copy];

    [self readDefaults];
}

+ (Class)socketClass;
{
    return [ONTCPSocket class];
}

+ (int)defaultPort;
{
    return 80;
}

+ (void)readDefaults;
{
    NSArray *systemLanguages = nil;
    unsigned int systemLanguageIndex;
    unsigned int systemLanguageCount;
    NSUserDefaults *defaults;

    defaults = [NSUserDefaults standardUserDefaults];

    userAgentHeaderFormat = [defaults stringForKey:@"OWHTTPUserAgentHeaderFormat"];
    userAgentInfo = [[NSString stringWithFormat: userAgentHeaderFormat, primaryUserAgentInfo] retain];

    OWHTTPDebug = [defaults boolForKey:@"OWHTTPDebug"];
    [OWHeaderDictionary setDebug:OWHTTPDebug];

    // Check NSLanguages to see which languages we accept
    systemLanguages = [[NSUserDefaults standardUserDefaults] stringArrayForKey:@"NSLanguages"];

    // otherwise, fall back on English
    if (!systemLanguages || [systemLanguages count] == 0)
        systemLanguages = [NSArray arrayWithObjects:@"English", nil];

    if (languageArray)
        [languageArray release];
    languageArray = [[NSMutableArray alloc]
            initWithCapacity:[systemLanguages count]];
    systemLanguageCount = [systemLanguages count];
    for (systemLanguageIndex = 0; systemLanguageIndex < systemLanguageCount; systemLanguageIndex++) {
        NSString *languageAbbreviation;

        languageAbbreviation = [languageAbbreviationsDictionary objectForKey:[systemLanguages objectAtIndex:systemLanguageIndex]];
        if (languageAbbreviation)
            [languageArray addObject:languageAbbreviation];
    }

    acceptLanguageString = [[self stringForHeader:@"Accept-Language"
            value:[languageArray componentsJoinedByString:@", "]] retain];
}

+ (void)setDebug:(BOOL)shouldDebug;
{
    OWHTTPDebug = shouldDebug;
}

+ (BOOL)shouldSpoofNetLocation:(OWNetLocation *)whatServer;
{
    NSString *hostname;
    NSUserDefaults *defaults;
    NSArray *spoofServers;
    unsigned int spoofServerIndex, spoofServerCount;
    NSString *spoofDomain;

    defaults = [NSUserDefaults standardUserDefaults];

    if ([defaults boolForKey:@"OWHTTPSpoofAllServers"])
        return YES;

    hostname = [whatServer hostname];
    spoofServers = [defaults arrayForKey:@"OWHTTPSpoofServers"];
    spoofServerCount = [spoofServers count];
    for (spoofServerIndex = 0; spoofServerIndex < spoofServerCount; spoofServerIndex++) {
        spoofDomain = [spoofServers objectAtIndex:spoofServerIndex];
        if ([hostname hasSuffix:spoofDomain])
            return YES;
    }
    return NO;
}

+ (NSString *)userAgentInfoForServerAtNetLocation:(OWNetLocation *)whatServer
{
    return userAgentInfo;
}

+ (NSString *)preferredDateFormat;
{
    return preferredDateFormat;
}

+ (NSArray *)acceptLanguages;
{
    return languageArray;
}

- initWithAddress:(OWAddress *)anAddress inQueue:(OWHTTPSessionQueue *)aQueue;
{
    OWURL *proxyURL, *realURL;
    
    if (![super init])
        return nil;

    queue = aQueue;
    proxyURL = [anAddress proxyURL];
    realURL = [anAddress url];
    flags.connectingViaProxyServer = (proxyURL != realURL);
    proxyLocation = [[proxyURL parsedNetLocation] retain];
    processorQueue = [[NSMutableArray alloc] initWithCapacity:[queue maximumNumberOfRequestsToPipeline]];
    flags.pipeliningRequests = NO;
    failedRequests = 0;

    return self;
}

- (void)dealloc;
{
    [self disconnectAndRequeueProcessors];
    [proxyLocation release];
    proxyLocation = nil; // Why?  Thread-safety issues?  We should fix those instead--this isn't a reliable cure, since the other thread could access proxyLocation after we release but before we reset it.
    [processorQueue release];
    processorQueue = nil;
    [super dealloc];
}

- (void)runSession;
{
    do {
        NSException *sessionException = nil;
        OWHTTPProcessor *aProcessor;

        OMNI_POOL_START {
            NS_DURING {
                BOOL continueSession = YES;

                while (continueSession) {
                    OMNI_POOL_START {
                        continueSession = [self sendRequest];
                        if (!continueSession)
                            break;
                        aProcessor = [processorQueue objectAtIndex:0];
                        if ([self fetchForProcessor:aProcessor inPipeline:[aProcessor pipeline]]) {
                            [processorQueue removeObjectAtIndex:0];
                        } else {
                            [self disconnectAndRequeueProcessors];
                        }
                    } OMNI_POOL_END;
                }
            } NS_HANDLER {
                sessionException = localException;
            } NS_ENDHANDLER;
            if (sessionException != nil) {
                unsigned int processorIndex, processorCount;

                // Notify processors
                processorCount = [processorQueue count];
                for (processorIndex = 0; processorIndex < processorCount; processorIndex++) {
                    aProcessor = [processorQueue objectAtIndex:processorIndex];
                    [self notifyProcessor:aProcessor ofSessionException:sessionException];
                }
                [processorQueue removeAllObjects];
                [self disconnectAndRequeueProcessors];
                if (processorCount == 0) {
                    // -sendRequest raised an exception before we even started looking for processors (e.g., when trying to connect to a server which is down or doesn't exist).  Send the exception to all queued processors.
                    while ((aProcessor = [queue nextProcessor])) {
                        [self notifyProcessor:aProcessor ofSessionException:sessionException];
                    }
                }
            }
        } OMNI_POOL_END;
    } while (![queue sessionIsIdle:self]);
}

- (BOOL)prepareConnectionForProcessor:(OWProcessor *)aProcessor;
{
    // The HTTPS plug-in subclasses this method to support SSL-Tunneling
    return YES;
}

- (void)abortProcessingForProcessor:(OWProcessor *)aProcessor;
{
    // Does this really abort the right processor?
    if ([processorQueue containsObject:aProcessor]) {
        fetchFlags.aborting = YES;
        [(ONInternetSocket *)[socketStream socket] abortSocket];
    }
}

- (void)setStatusString:(NSString *)newStatus;
{
    unsigned int index, count;

    // Set the status for all processors in this session
    for (index = 0, count = [processorQueue count]; index < count; index++)
        [[processorQueue objectAtIndex:index] setStatusString:newStatus];
    // And for any processors waiting on us
    [queue session:self hasStatusString:newStatus];
}

- (void)setStatusFormat:(NSString *)aFormat, ...;
{
    NSString *newStatus;
    va_list argList;

    va_start(argList, aFormat);
    newStatus = [[NSString alloc] initWithFormat:aFormat arguments:argList];
    va_end(argList);
    [self setStatusString:newStatus];
    [newStatus release];
}

// OBObject subclass

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    if (fetchAddress)
        [debugDictionary setObject: fetchAddress forKey:@"fetchAddress"];
    if (socketStream)
        [debugDictionary setObject:socketStream forKey:@"socketStream"];
    if (headerDictionary)
        [debugDictionary setObject:headerDictionary forKey:@"headerDictionary"];

    return debugDictionary;
}

@end


@implementation OWHTTPSession (SubclassesOnly)

- (void)connect;
{
    NSString *port;
    ONInternetSocket *socket;
    ONHost *host;
    Class socketClass;
    
    
    [self setStatusFormat:NSLocalizedStringFromTableInBundle(@"Finding %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
    port = [proxyLocation port];
    host = [ONHost hostForHostname:[proxyLocation hostname]];
    [self setStatusFormat:NSLocalizedStringFromTableInBundle(@"Contacting %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];

    // Sadly, we have two socket methods, +[ONInternetSocket socket] and -[ONSocketStream socket], and the compiler doesn't know which it's getting without the final cast.  (There's no way to cast +socketClass to disambiguate ahead of time.)
    socketClass = [isa socketClass];
    socket = (ONInternetSocket *)objc_msgSend(socketClass, @selector(socket));
    [socket setReadBufferSize:32 * 1024];

    OBASSERT(!socketStream);
    socketStream = [[ONSocketStream alloc] initWithSocket:socket];
    [socket connectToHost:host port:port ? [port intValue] : [isa defaultPort]];

    [self setStatusFormat:NSLocalizedStringFromTableInBundle(@"Contacted %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
    if (OWHTTPDebug)
        NSLog(@"%@: Connected to %@", [isa description], [proxyLocation displayString]);
}

- (void)disconnectAndRequeueProcessors;
{
    unsigned int index, count;

    // drop all processors
    for (index = 0, count = [processorQueue count]; index < count; index++)
        [queue queueProcessor:[processorQueue objectAtIndex:index]];
    [processorQueue removeAllObjects];

    // If we have a socket stream, release it
    [socketStream release];
    socketStream = nil;
}

- (BOOL)fetchForProcessor:(OWHTTPProcessor *)aProcessor inPipeline:(OWPipeline *)aPipeline;
{
    NSException *sessionException = nil;
    BOOL finishedProcessing = NO;
     
    nonretainedProcessor = aProcessor;
    nonretainedPipeline = aPipeline;
    [nonretainedProcessor processBegin];
    
    fetchAddress = [[nonretainedPipeline lastAddress] retain];
    fetchURL = [[fetchAddress url] retain];
    headerDictionary = [[OWHeaderDictionary alloc] init];
    interruptedDataStream = [[nonretainedProcessor dataStream] retain];

    NS_DURING {
        if ([[fetchAddress methodString] isEqualToString:@"HEAD"])
            finishedProcessing = [self readHead];
        else
            finishedProcessing = [self readResponse];
        failedRequests = 0;
    } NS_HANDLER {
        if (flags.pipeliningRequests && [[localException reason] isEqualToString:@"Unable to read from socket: Connection reset by peer"]) {
            // This HTTP 1.1 connection was reset by the server
            if ([interruptedDataStream bufferedDataLength] < 1024) {
                failedRequests++;
                if (failedRequests > 0) {
                    // We've been dropped by this server without getting much data:  let's try a traditional HTTP/1.0 connection instead.
                    // NSLog(@"%@: Switching to HTTP/1.0", OBShortObjectDescription(self));
                    [queue setServerCannotHandlePipelinedRequestsReliably];
                    failedRequests = 0;
                }
            } else {
                // Well, we got _some_ data...
                failedRequests = 0;
            }
        } else if (interruptedDataStream) {
            // Abort the data stream and reraise the exception
            sessionException = [localException retain];
            [interruptedDataStream dataAbort];
        } else {
            // We've caught an exception, but it didn't interrupt a data stream.  The only exception I've seen here is "Unable to read from socket: Connection reset by peer".  Let's try again rather than notifying the processor of this exception.
            // NSLog(@"%@: Caught exception: %@", OBShortObjectDescription(self), [localException reason]);
        }
    } NS_ENDHANDLER;            

    fetchFlags.aborting = NO;
    if (sessionException) {
        [self notifyProcessor:nonretainedProcessor ofSessionException:sessionException];
    } else if (finishedProcessing) {
        [nonretainedProcessor processEnd];
        [nonretainedProcessor retire];        
    }

    // get rid of variables for this fetch
    [fetchAddress release];
    fetchAddress = nil;
    [fetchURL release];
    fetchURL = nil;
    [headerDictionary release];
    headerDictionary = nil;
    [interruptedDataStream release];
    interruptedDataStream = nil;
    return finishedProcessing;
}

- (NSString *)requestStringForProcessor:(OWHTTPProcessor *)aProcessor;
{
    NSMutableString *requestString;
    NSString *requestMethod;
    NSString *tempString;
    OWWebPipeline *aPipeline;
    OWAddress *anAddress;
    OWURL *aURL;

    aPipeline = (OWWebPipeline *)[aProcessor pipeline];
    anAddress = (OWAddress *)[aPipeline lastAddress];
    aURL = [anAddress url];
    requestMethod = [anAddress methodString];
    requestString = [NSMutableString stringWithCapacity:2048];
    [requestString appendString:[self commandStringForAddress:anAddress]];
    [requestString appendString:[self rangeStringForProcessor:aProcessor]];
    [requestString appendString:[self keepAliveString]];
    [requestString appendString:[self referrerHeaderStringForPipeline:aPipeline]];
    [requestString appendString:[self userAgentHeaderStringForURL:aURL]];
    [requestString appendString:[self pragmaHeaderStringForPipeline:aPipeline]];
    [requestString appendString:[self hostHeaderStringForURL:aURL]];
    [requestString appendString:[self acceptHeadersStringForTarget:[aPipeline target]]];
    [requestString appendString:[self acceptLanguageHeadersString]];
    [requestString appendString:[self authorizationStringForAddress:anAddress]];
    [requestString appendString:[self cookiesForURL:aURL]];
    // TODO: For bookmarks refreshes, we should support If-Modified-Since:
    if ([requestMethod isEqualToString:@"POST"] ||
        [requestMethod isEqualToString:@"PUT"]) {
        [requestString appendString:[self contentTypeHeaderStringForAddress:anAddress]];
        [requestString appendString:[self contentLengthHeaderStringForAddress:anAddress]];

        // Blank line signals end of headers
        [requestString appendString:endOfLineString];

        // We should probably make -requestData return this, and then this code wouldn't have to be here.  Which means that the above statement could be collapsed to be the same as the else clause.
        if ((tempString = [self contentStringForAddress:anAddress]))
            [requestString appendString:tempString];
    } else {
        // Blank line signals end of headers
        [requestString appendString:endOfLineString];
    }
    return requestString;
}

- (NSString *)authorizationStringForAddress:(OWAddress *)anAddress;
{
    NSMutableArray *allCredentials;
    NSString *credentialString;
    OWAuthorizationServer *authorizationServer;

    flags.foundCredentials = NO;
    flags.foundProxyCredentials = NO;
    allCredentials = [[NSMutableArray alloc] init];

    authorizationServer = [[OWAuthorizationServer serverForAddress:anAddress] retain];
    if (authorizationServer) {
        NSArray *credentials;

        credentials = [authorizationServer credentialsForPath:[[anAddress url] path]];
        if (credentials && [credentials count] > 0) {
            [allCredentials insertObjectsFromArray:credentials atIndex:0];
            flags.foundCredentials = YES;
        }
    }

    if (flags.connectingViaProxyServer) {
        OWAuthorizationServer *proxyAuthorizationServer;
        OWURL *proxyURL;

        proxyURL = [anAddress proxyURL];
        proxyAuthorizationServer = [[OWAuthorizationServer serverForProxy:[proxyURL netLocation]] retain];
        if (proxyAuthorizationServer) {
            NSArray *credentials;

            credentials = [proxyAuthorizationServer credentialsForPath:[[anAddress url] path]];
            if (credentials && [credentials count] > 0) {
                [allCredentials insertObjectsFromArray:credentials atIndex:0];
                flags.foundProxyCredentials = YES;
            }
        }
    }

    if ([allCredentials count] > 0) {
        credentialString = [[allCredentials componentsJoinedByString:endOfLineString] stringByAppendingString:endOfLineString];
    } else {
        credentialString = @"";
    }

    [allCredentials release];

    return credentialString;
}

- (NSString *)userAgentHeaderStringForURL:(OWURL *)aURL;
{
    NSString *userAgent = [isa userAgentInfoForServerAtNetLocation:[aURL parsedNetLocation]];
    if (userAgent == nil)
        return @"";
    else
        return [isa stringForHeader:@"User-Agent" value:userAgent];
}

- (BOOL)sendRequest;
{    
    if (![(ONInternetSocket *)[socketStream socket] isWritable]) {
        [self disconnectAndRequeueProcessors];
        [self connect];
    }
    NS_DURING {
        [self sendRequests];
    } NS_HANDLER {
        NSString *exceptionReason;

        exceptionReason = [localException reason];
        if ([exceptionReason isEqualToString:@"Unable to write to socket: Connection reset by peer"] ||
            [exceptionReason isEqualToString:@"Unable to write to socket: Broken pipe"]) {
            // We'll try again in a bit
        } else {
            // Reraise the exception
            [localException raise];
        }
    } NS_ENDHANDLER;
    return ([processorQueue count] != 0);
}

- (BOOL)sendRequests;
{
    NSData *requestData;
    NSString *requestString;
    OWAddress *anAddress;
    OWHTTPProcessor *aProcessor = nil;
    unsigned int index, count;
    BOOL shouldPipelineRequests;
    unsigned int maximumNumberOfRequestsToSend;

    // figure out how many requests to send
    shouldPipelineRequests = [queue shouldPipelineRequests];
    count = [processorQueue count];
    // We've already sent requests for the processors in processorQueue
    index = count;
    if (shouldPipelineRequests) {
        maximumNumberOfRequestsToSend = [queue maximumNumberOfRequestsToPipeline];
        flags.pipeliningRequests = YES;
    } else {
        maximumNumberOfRequestsToSend = 1;
    }
    // Fill our queue
    while (count < maximumNumberOfRequestsToSend) {
        aProcessor = [queue nextProcessor];
        if (!aProcessor)
            break;
        switch ([aProcessor status]) {
            case OWProcessorStarting:
            case OWProcessorQueued:
                [aProcessor processBegin];
                // Fall through to running state
            case OWProcessorRunning:
                [processorQueue addObject:aProcessor];
                [aProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Preparing to request document from %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
                count++;
                break;
            case OWProcessorAborting:
                [aProcessor retire];
                // Fall through to retired state
            case OWProcessorRetired:
                break;
        }
    }

    // Send requests for each new processor in the queue
    for (; index < count; index++) {
        aProcessor = [processorQueue objectAtIndex:index];
        if ([self prepareConnectionForProcessor:aProcessor]) {
            anAddress = (OWAddress *)[[aProcessor pipeline] lastAddress];
            requestString = [self requestStringForProcessor:aProcessor];
            if (OWHTTPDebug)
                NSLog(@"%@ Tx: %@", [[anAddress url] scheme], requestString);
            [aProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Requesting document from %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
            [socketStream writeString:requestString];
            if ((requestData = [[anAddress methodDictionary] objectForKey:@"Content-Data"]))
                [socketStream writeData:requestData];
        }
    }
    return (index != 0);
}

@end

@implementation OWHTTPSession (Private)

+ (NSString *)stringForHeader:(NSString *)aHeader value:(id)aValue;
{
    NSMutableString *header;
    NSString *value;

    value = [aValue description];
    header = [[NSMutableString alloc] initWithCapacity:[aHeader length] + 2 + [value length] + [endOfLineString length]];
    [header appendString:aHeader];
    [header appendString:@": "];
    [header appendString:value];
    [header appendString:endOfLineString];
    [header autorelease];
    return header;
}

//
// Generating request
//

- (NSString *)commandStringForAddress:(OWAddress *)anAddress;
{
    NSMutableString *command;
    OWURL *aURL;
    NSString *fetchPath;

    aURL = [anAddress url];
    command = [NSMutableString stringWithCapacity:128];
    [command appendFormat:@"%@ ", [anAddress methodString]];
    if (flags.connectingViaProxyServer)
        fetchPath = [aURL proxyFetchPath];
    else
        fetchPath = [aURL fetchPath];
    if ([fetchPath containsCharacterInSet:spaceCharacterSet]) {
        // Don't send a malformed request just because someone forgot to quote the spaces in their URL
        fetchPath = [fetchPath stringByReplacingCharactersInSet:spaceCharacterSet withString:@"%20"];
    }
    [command appendString:fetchPath];
    [command appendFormat:@" %@", flags.pipeliningRequests ? http11VersionString : http10VersionString];
    [command appendString:endOfLineString];

    return command;
}

// Note --- this method uses no ivars.
- (NSString *)acceptHeadersStringForTarget:(id <OWTarget>)aTarget;
{
    NSString *headerString;
    NSMutableArray *acceptsArray;
    NSArray *encodings;
    NSEnumerator *possibleTypesEnumerator;
    OWContentType *possibleType;
    
    if ([[NSUserDefaults standardUserDefaults] boolForKey:@"OWHTTPFakeAcceptHeader"]) {
        headerString = [isa stringForHeader:@"Accept" value:@"image/gif, image/x-xbitmap, image/jpeg, image/pjpeg, image/png, image/tiff, multipart/x-mixed-replace, */*"];
    } else {
        NSString *wildcardContentTypeString;
        // Determine the real Accept header

        acceptsArray = [[NSMutableArray alloc] initWithCapacity:16];
        possibleTypesEnumerator = [[[aTarget targetContentType] indirectSourceContentTypes] objectEnumerator];
        while ((possibleType = [possibleTypesEnumerator nextObject]))
            if ([possibleType isPublic] && [possibleType isInteresting])
                [acceptsArray addObject:[possibleType contentTypeString]];

        // TODO: Be clever about when we include the wildcard content type
        // also: should we have covers for the boring content types (image/*, maybe)?
        wildcardContentTypeString = [[OWContentType wildcardContentType] contentTypeString];
        if ([acceptsArray indexOfObject:wildcardContentTypeString] == NSNotFound)
            [acceptsArray addObject:wildcardContentTypeString];

        headerString = [isa stringForHeader:@"Accept" value:[acceptsArray componentsJoinedByString:@", "]];
        [acceptsArray release];
    }

    // Determine the Accept-Encoding header if any
    encodings = [OWContentType contentEncodings];
    acceptsArray = [[NSMutableArray alloc] initWithCapacity:[encodings count]];
    possibleTypesEnumerator = [encodings objectEnumerator];
    while ((possibleType = [possibleTypesEnumerator nextObject])) {
        NSString *encodingString;
        if (![possibleType isPublic])
            continue;
        // TODO: Discard encodings we know about but can't handle (see comment above +contentEncodings)
//        if (![[possibleType links] count])
//            continue;
        encodingString = [possibleType contentTypeString];
        // TODO: More generic way to perform this test
        if ([encodingString isEqualToString:@"encoding/none"])
            continue;
        if (![encodingString hasPrefix:@"encoding/"])
            continue; // this should never happen
        [acceptsArray addObject:[encodingString substringFromIndex:9]]; // remove the "encoding/"
    }
    if ([acceptsArray count])
        headerString = [headerString stringByAppendingString:[isa stringForHeader:@"Accept-Encoding" value:[acceptsArray componentsJoinedByString:@", "]]];
    [acceptsArray release];

    return headerString;
}

- (NSString *)acceptLanguageHeadersString;
{
    return acceptLanguageString;
}

- (NSString *)referrerHeaderStringForPipeline:(OWWebPipeline *)aPipeline;
{
    OWAddress *referringAddress;
    NSString *referrerString;

    if ([aPipeline respondsToSelector:@selector(referringAddress)])
        referringAddress = [(OWWebPipeline *)aPipeline referringAddress];
    else
        return @"";
    
    referrerString = [referringAddress addressString];
    if (!referrerString)
        return @"";
    return [isa stringForHeader:@"Referer" /* [sic] */ value:referrerString];
}

- (NSString *)pragmaHeaderStringForPipeline:(OWWebPipeline *)aPipeline;
{
    if ([aPipeline respondsToSelector:@selector(proxyCacheDisabled)] &&
        [aPipeline proxyCacheDisabled])
        return [isa stringForHeader:@"Pragma" value:@"no-cache"];
    else
        return @"";
}

- (NSString *)hostHeaderStringForURL:(OWURL *)aURL;
{
    NSString *netLocation;

    netLocation = [aURL netLocation];
    if (netLocation)
        return [isa stringForHeader:@"Host" value:netLocation];
    else
        return @"";
}

- (NSString *)rangeStringForProcessor:(OWHTTPProcessor *)aProcessor;
{
    OWDataStream *dataStream;

    if ((dataStream = [aProcessor dataStream]) && [dataStream bufferedDataLength]) {
        OBASSERT([queue serverUnderstandsPipelinedRequests]);
        return [isa stringForHeader:@"Range" value:[NSString stringWithFormat:@"bytes=%d-", [dataStream bufferedDataLength]]];
    } else
        return @"";
}

- (NSString *)keepAliveString;
{
    if (flags.connectingViaProxyServer)
        return @"";
    else
        return [isa stringForHeader:@"Connection" value:@"Keep-Alive"];
}

- (NSString *)cookiesForURL:(OWURL *)aURL;
{
    NSMutableString *cookieString = nil;
    NSArray *cookies;
    unsigned int cookieIndex, cookieCount;

    cookies = [OWCookieDomain cookiesForURL:aURL];
    if (!cookies || (cookieCount = [cookies count]) == 0)
        return @"";
    for (cookieIndex = 0; cookieIndex < cookieCount; cookieIndex++) {
        OWCookie *cookie;

        cookie = [cookies objectAtIndex:cookieIndex];
        if (!cookieString)
            cookieString = [[[NSMutableString alloc] initWithString:@"Cookie: "] autorelease];
        else
            [cookieString appendString:@"; "];
        [cookieString appendString:[cookie name]];
        [cookieString appendString:@"="];
        [cookieString appendString:[cookie value]];
    }
    [cookieString appendString:endOfLineString];
    return cookieString;
}

- (NSString *)contentTypeHeaderStringForAddress:(OWAddress *)anAddress;
{
    NSMutableString *valueString;
    NSDictionary *addressMethodDictionary;
    NSString *boundaryString;
    NSString *contentTypeHeaderString;
    NSString *contentType;

    addressMethodDictionary = [anAddress methodDictionary];
    contentType = [addressMethodDictionary objectForKey:@"Content-Type"];
    if (!contentType)
        return @"";
    valueString = [[NSMutableString alloc] initWithString:contentType];
    boundaryString = [addressMethodDictionary objectForKey:@"Boundary"];
    if (boundaryString)
        [valueString appendFormat:@"; boundary=%@", boundaryString];

    contentTypeHeaderString = [isa stringForHeader:@"Content-Type" value:valueString];
    [valueString release];
    return contentTypeHeaderString;
}

- (NSString *)contentLengthHeaderStringForAddress:(OWAddress *)anAddress;
{
    NSDictionary *addressMethodDictionary;

    addressMethodDictionary = [anAddress methodDictionary];
    return [isa stringForHeader:@"Content-Length" value:[NSNumber numberWithInt:[[addressMethodDictionary objectForKey:@"Content-String"] length] + [[addressMethodDictionary objectForKey:@"Content-Data"] length]]];
}

- (NSString *)contentStringForAddress:(OWAddress *)anAddress;
{
    NSString *methodContentString;

    methodContentString = [[anAddress methodDictionary] objectForKey:@"Content-String"];
    if (!methodContentString)
        return nil;
    return [methodContentString stringByAppendingString:endOfLineString];
}

//
// Reading results
//

- (BOOL)readResponse;
{
    NSString *line;
    NSScanner *scanner;
    float httpVersion;
    HTTPStatus httpStatus;
    NSString *commentString;
    OWAuthorizationServer *authorizationServer;
    OWAuthorizationServer *proxyAuthorizationServer;

    if (fetchFlags.aborting)
        [NSException raise:@"User Aborted" format:NSLocalizedStringFromTableInBundle(@"User Stopped", @"OWF", [self bundle], httpsession error)];
        
    [nonretainedProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Awaiting document from %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];

beginReadResponse:    
    
    line = [socketStream peekLine];
    while (line != nil && [line isEqualToString:@""]) {
        // Skipping past leading newlines in the response fixes a problem I was seeing talking to a SmallWebServer/2.0 (used in some bulletin boards like the one at Clan Fat, http://pub12.ezboard.com/bfat).  I think what might have happened is that they miscalculated their content length in an earlier request, and sent us an extra newline following the counted bytes.  The result was that every other request to the server would fail.
        // Note:  if we're actually talking to an HTTP 0.9 server, it's possible we're losing blank lines at the beginning of the content they're sending us.  But since I haven't seen any HTTP 0.9 servers in a long, long time...
        [socketStream readLine]; // Skip past the empty line
        line = [socketStream peekLine]; // And peek at the next one
    }
    if (line == nil)
        return NO;
    if (OWHTTPDebug)
        NSLog(@"%@ Rx: %@", [fetchURL scheme], line);
    scanner = [NSScanner scannerWithString:line];

    if (![scanner scanString:@"HTTP" intoString:NULL]) {
        // 0.9 server:  good luck!
        NSLog(@"%@ is ancient, good luck!", [proxyLocation shortDisplayString]);
        [nonretainedProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"%@ is ancient, good luck!", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
        [headerDictionary addString:@"www/unknown" forKey:@"content-type"];
        [nonretainedPipeline setHeaderDictionary:headerDictionary fromURL:fetchURL];
        [OWCookieDomain registerCookiesFromPipeline:nonretainedPipeline headerDictionary:headerDictionary];
        [self readBodyAndIgnore:NO];
        return YES;
    }

    [socketStream readLine]; // Skip past the line we're already parsing
    [scanner scanString:@"/" intoString:NULL];
    [scanner scanFloat:&httpVersion];
    if (OWHTTPDebug)
        NSLog(@"Rx: %@", [fetchAddress addressString]);
    if (httpVersion > 1.0) {
        [queue setServerUnderstandsPipelinedRequests];
    }

    [scanner scanInt:(int *)&httpStatus];
    if (![scanner scanUpToString:@"\n" intoString:&commentString])
        commentString = @"";

processStatus:
    switch (httpStatus) {

        // 100 codes - Informational
        
        case HTTP_STATUS_CONTINUE:
            // read the headers, ignore 'em, start over
            [self readHeaders];
            goto beginReadResponse;
            
        // 200 codes - Success: Got MIME object

        case HTTP_STATUS_OK:
        case HTTP_STATUS_PARTIAL_CONTENT:
            [self readHeaders];
            [self readBodyAndIgnore:NO];
            break;

        case HTTP_STATUS_NO_CONTENT:
            // Netscape 4.0 just ignores request if it returns "NO_CONTENT"
            // was [NSException raise:@"NoContent" format:@"Server returns no content"];
            break;		// Don't read headers and body

        // 300 codes - Temporary error (various forms of redirection)

#warning Should double-check our HTTP 1.1 handling
        case HTTP_STATUS_MULTIPLE_CHOICES:
            break;

        // TODO: Handle permanent vs. temporary redirection
        case HTTP_STATUS_MOVED_PERMANENTLY:
        case HTTP_STATUS_MOVED_TEMPORARILY:
            {
                NSString *newLocationString;
                OWAddress *newLocation;

                [self readHeaders];
                newLocationString =
                    [headerDictionary lastStringForKey:@"location"];
                if (!newLocationString)
                    [NSException raise:@"Redirect failure" format:NSLocalizedStringFromTableInBundle(@"Location header missing on redirect", @"OWF", [self bundle], httpsession error)];
                newLocation = [fetchAddress addressForRelativeString:newLocationString];
                if ([newLocation isEqual:fetchAddress])
                    [NSException raise:@"Redirect loop" format:NSLocalizedStringFromTableInBundle(@"Redirect loop", @"OWF", [self bundle], httpsession error)];
                [nonretainedProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Redirected to %@", @"OWF", [self bundle], httpsession status), newLocationString];
                if (httpStatus == HTTP_STATUS_MOVED_PERMANENTLY)
                    [nonretainedPipeline permanentRedirection:newLocation];
                [nonretainedPipeline addSourceContent:newLocation];
                // TODO: Should we cacheContent for permanent redirections? If not, maybe -cacheContent should do the permanent redirection stuff that -permanentRedirection: does.
                if ([[newLocation addressString] isEqualToString:[[fetchAddress addressString] stringByAppendingString:@"/"]])
                    [nonretainedPipeline cacheContent];
                [nonretainedPipeline startProcessingContent];
                [self readBodyAndIgnore:YES];
            }
            break;

        case HTTP_STATUS_SEE_OTHER:
            break;

        case HTTP_STATUS_NOT_MODIFIED:
            break;

        case HTTP_STATUS_USE_PROXY:
            break;

        // 400 codes - Permanent error

        case HTTP_STATUS_BAD_REQUEST:
        case HTTP_STATUS_PAYMENT_REQUIRED:
        case HTTP_STATUS_FORBIDDEN:
        case HTTP_STATUS_NOT_FOUND:
            [nonretainedPipeline processor:nonretainedProcessor hasErrorName:commentString reason:[NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Server returns \"%@\" (%d)", @"OWF", [self bundle], httpsession error), commentString, httpStatus]];
            [self readHeaders];
            [nonretainedPipeline contentError];
            [self readBodyAndIgnore:NO];
            break;

        case HTTP_STATUS_UNAUTHORIZED:
            [nonretainedProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Authorizing %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
            [self readHeaders];
            authorizationServer = [[OWAuthorizationServer newServerForAddress:fetchAddress] retain];
            NS_DURING {
                [authorizationServer generateCredentialsForChallenge:headerDictionary path:[fetchURL path] reprompt:flags.foundCredentials];
            } NS_HANDLER {
                [nonretainedPipeline processor:nonretainedProcessor hasErrorName:[localException displayName] reason:[localException reason]];
                [nonretainedPipeline contentError];
                [self readBodyAndIgnore:NO];
                break;
            } NS_ENDHANDLER;
            [nonretainedPipeline startProcessingContent];
            [self readBodyAndIgnore:YES];
            break;


        case HTTP_STATUS_PROXY_AUTHENTICATION_REQUIRED:
            [nonretainedProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Authorizing %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
            [self readHeaders];
            proxyAuthorizationServer = [[OWAuthorizationServer newServerForProxy:[proxyLocation displayString]] retain];
            NS_DURING {
                [proxyAuthorizationServer generateCredentialsForChallenge:headerDictionary path:[fetchURL path] reprompt:flags.foundProxyCredentials];
            } NS_HANDLER {
                [nonretainedPipeline processor:nonretainedProcessor hasErrorName:[localException displayName] reason:[localException reason]];
                [nonretainedPipeline contentError];
                [self readBodyAndIgnore:NO];
                break;
            } NS_ENDHANDLER;
            [nonretainedPipeline startProcessingContent];
            [self readBodyAndIgnore:YES];
            break;

        case HTTP_STATUS_REQUEST_TIMEOUT:
            return NO; // Try again

        // 500 codes - Server error
        case HTTP_STATUS_INTERNAL_SERVER_ERROR:
        case HTTP_STATUS_NOT_IMPLEMENTED:
        case HTTP_STATUS_BAD_GATEWAY:
        case HTTP_STATUS_SERVICE_UNAVAILABLE:
        case HTTP_STATUS_GATEWAY_TIMEOUT:
            [nonretainedPipeline processor:nonretainedProcessor hasErrorName:commentString reason:[NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Server returns \"%@\" (%d)", @"OWF", [self bundle], httpsession error), commentString, httpStatus]];
            [self readHeaders];
            [nonretainedPipeline contentError];
            [self readBodyAndIgnore:NO];
            break;

        // Unrecognized client code, treat as x00

        default:
            {
                HTTPStatus equivalentStatus;

                equivalentStatus = httpStatus - httpStatus % 100;
                if (equivalentStatus == httpStatus)
                    httpStatus = HTTP_STATUS_NOT_IMPLEMENTED;
                else
                    httpStatus = equivalentStatus;
            }
            goto processStatus;
    }
    return YES;
}

- (void)readBodyAndIgnore:(BOOL)ignoreThis;
{
    OWParameterizedContentType *parameterizedContentType;

    if (ignoreThis) {
#warning The implementation of reading bodies currently requires that a data stream exist, so we create one that will just get thrown away. We could be more efficient by having separate methods for each type of body to read and ignore the results.    
        interruptedDataStream = [[OWDataStream alloc] init];
    } else if (!interruptedDataStream) {
        OWContentType *testContentType;

        interruptedDataStream = [[OWDataStream alloc] init];
        parameterizedContentType = [headerDictionary parameterizedContentType];
        testContentType = [parameterizedContentType contentType];
        if (testContentType == textPlainContentType || testContentType == applicationOctetStreamContentType) {
            // Treat text/plain and application/octet-stream as www/unknown.  A lot of web servers out there use these as default content types, which means they end up claiming that, say, .gnutar.gz files are text/plain, or ReadMe is application/octet-stream.  So...if we see a suspicious content type, let's just run it through our data detector and see whether it really is what it claims to be.  (Some HTML pages are also served as text/plain, but IE displays them as HTML even though they're valid plain text.  This also makes us compatible with that behavior.)
            parameterizedContentType = [[[OWParameterizedContentType alloc] initWithContentType:[OWUnknownDataStreamProcessor unknownContentType]] autorelease];
        }
        [interruptedDataStream setFullContentType:parameterizedContentType];
        [interruptedDataStream setContentEncoding:[headerDictionary contentEncoding]];
        [nonretainedPipeline addSourceContent:interruptedDataStream];
        [nonretainedPipeline cacheContent];
        [nonretainedPipeline startProcessingContent];
        if ([queue shouldPipelineRequests]) {
            // If it's not okay to pipeline requests, then we can't fetch partial ranges anyway, so we don't want to cache this data stream.
            [nonretainedProcessor setDataStream:interruptedDataStream];
        }
    }
    
    [nonretainedProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Reading document from %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];

    if ([[headerDictionary lastStringForKey:@"transfer-encoding"] isEqualToString:@"chunked"])
        [self readChunkedBodyIntoStream:interruptedDataStream];
    else if ([headerDictionary lastStringForKey:@"content-length"])
        [self readStandardBodyIntoStream:interruptedDataStream];
    else
        [self readClosingBodyIntoStream:interruptedDataStream];

    // NSLog(@"%@: ending data stream %@", OBShortObjectDescription(self), OBShortObjectDescription(interruptedDataStream));
    [interruptedDataStream dataEnd];
    // NSLog(@"%@: ended data stream %@", OBShortObjectDescription(self), OBShortObjectDescription(interruptedDataStream));
}

- (BOOL)readHead;
{
    NSString *line;
    NSScanner *scanner;
    float httpVersion;
    HTTPStatus httpStatus;
    NSString *commentString;
    OWTimeStamp *stamp = nil;

    [nonretainedProcessor setStatusFormat:NSLocalizedStringFromTableInBundle(@"Awaiting document info from %@", @"OWF", [self bundle], httpsession status), [proxyLocation shortDisplayString]];
    line = [socketStream peekLine];
    while (line != nil && [line isEqualToString:@""]) {
        // Skipping past leading newlines in the response fixes a problem I was seeing talking to a SmallWebServer/2.0 (used in some bulletin boards like the one at Clan Fat, http://pub12.ezboard.com/bfat).  I think what might have happened is that they miscalculated their content length in an earlier request, and sent us an extra newline following the counted bytes.  The result was that every other request to the server would fail.
        // Note:  if we're actually talking to an HTTP 0.9 server, it's possible we're losing blank lines at the beginning of the content they're sending us.  But since I haven't seen any HTTP 0.9 servers in a long, long time...
        [socketStream readLine]; // Skip past the empty line
        line = [socketStream peekLine]; // And peek at the next one
    }
    if (line == nil)
        return NO;
    if (OWHTTPDebug)
        NSLog(@"%@ Rx: %@", [fetchURL scheme], line);
    scanner = [NSScanner scannerWithString:line];

    if (![scanner scanString:@"HTTP" intoString:NULL]) {
        // 0.9 server, so can't determine timestamp
        stamp = [OWTimeStamp cacheDate:[NSDate distantPast] forAddress:fetchAddress];
        [nonretainedPipeline addContent:stamp];
        [nonretainedPipeline startProcessingContent];
        return YES;
    }

    [socketStream readLine]; // Skip past the line we're already parsing
    [scanner scanString:@"/" intoString:NULL];
    [scanner scanFloat:&httpVersion];
    if (OWHTTPDebug)
        NSLog(@"Rx: %@", [fetchAddress addressString]);
    if (httpVersion > 1.0) {
        [queue setServerUnderstandsPipelinedRequests];
    }

    [scanner scanInt:(int *)&httpStatus];
    if (![scanner scanUpToString:@"\n" intoString:&commentString])
        commentString = @"";

processStatus:
    switch (httpStatus) {

        // 200 codes - Got MIME object

        case HTTP_STATUS_OK:
        case HTTP_STATUS_CREATED:
        case HTTP_STATUS_ACCEPTED:
        case HTTP_STATUS_NON_AUTHORITATIVE_INFORMATION:
        case HTTP_STATUS_NO_CONTENT:
        case HTTP_STATUS_RESET_CONTENT:
        case HTTP_STATUS_PARTIAL_CONTENT:
            [self readHeaders];
            return YES;

        // 300 codes - Various forms of redirection

        case HTTP_STATUS_MULTIPLE_CHOICES:
            break;

        case HTTP_STATUS_MOVED_PERMANENTLY:
        case HTTP_STATUS_MOVED_TEMPORARILY:
            {
                OWAddress *newLocation;

                [self readHeaders];
                newLocation = [OWAddress addressForString:
                    [headerDictionary lastStringForKey:@"location"]];
                if (httpStatus == HTTP_STATUS_MOVED_PERMANENTLY)
                    [nonretainedPipeline permanentRedirection:newLocation];
                [nonretainedPipeline addSourceContent:newLocation];
                // TODO: Should we cacheContent even for temporary redirections? If not, maybe -cacheContent should do the permanent redirection stuff that -permanentRedirection: does.
                [nonretainedPipeline cacheContent];
                [nonretainedPipeline startProcessingContent];
            }
            break;

        case HTTP_STATUS_SEE_OTHER:
        case HTTP_STATUS_NOT_MODIFIED:
        case HTTP_STATUS_USE_PROXY:
            break;

        // 400 codes - Access Authorization problem

        case HTTP_STATUS_UNAUTHORIZED:
            // TODO: authorization

        case HTTP_STATUS_PAYMENT_REQUIRED:
        case HTTP_STATUS_FORBIDDEN:
        case HTTP_STATUS_NOT_FOUND:
            stamp = [OWTimeStamp cacheDate:[NSDate distantFuture] forAddress:fetchAddress];
            break;

        case HTTP_STATUS_REQUEST_TIMEOUT:
            return NO; // Try again

        case HTTP_STATUS_BAD_REQUEST:
            // fall through to 500 codes

        // 500 codes - Server error

        case HTTP_STATUS_INTERNAL_SERVER_ERROR:
        case HTTP_STATUS_BAD_GATEWAY:
        case HTTP_STATUS_NOT_IMPLEMENTED: 
        case HTTP_STATUS_SERVICE_UNAVAILABLE:
        case HTTP_STATUS_GATEWAY_TIMEOUT:
            // ignore it and try later?
            stamp = [OWTimeStamp cacheDate:[NSDate distantPast] forAddress:fetchAddress];
            break;

        // Unrecognized client code, treat as x00

        default:
            {
                HTTPStatus equivalentStatus;

                equivalentStatus = httpStatus - httpStatus % 100;
                if (equivalentStatus == httpStatus)
                    httpStatus = HTTP_STATUS_NOT_IMPLEMENTED;
                else
                    httpStatus = equivalentStatus;
            }
            goto processStatus;
    }
    [nonretainedPipeline addContent:stamp];
    [nonretainedPipeline startProcessingContent];
    return YES;
}

- (void)readTimeStamp;
{
    NSDate *date = nil;
    NSString *string;
    OWTimeStamp *stamp;

    string = [headerDictionary lastStringForKey:@"last-modified"];
    if (string) {
        date = [NSDate dateWithHTTPDateString:string];
    }
    if (!date)
        date = [NSDate distantPast];
    stamp = [OWTimeStamp cacheDate:date forAddress:fetchAddress];
}

- (void)readHeaders;
{
    [headerDictionary readRFC822HeadersFromSocketStream:socketStream];
    if (OWHTTPDebug)
        NSLog(@"Rx Headers:\n%@", headerDictionary);
    [nonretainedPipeline setHeaderDictionary:headerDictionary fromURL:fetchURL];
    [OWCookieDomain registerCookiesFromPipeline:nonretainedPipeline headerDictionary:headerDictionary];
    [self readTimeStamp];
}

- (unsigned int)intValueFromHexString:(NSString *)aString;
{
    unsigned int addition, result = 0;
    unsigned int index, length = [aString length];
    unichar c;
    
    for (index = 0; index < length; index++) {
        c = [aString characterAtIndex:index];
        if ((c >= '0') && (c <= '9'))
            addition = c - '0';
        else if ((c >= 'a') && (c <= 'f'))
            addition = c - 'a' + 10;
        else if ((c >= 'A') && (c <= 'F'))
            addition = c - 'A' + 10;
        else
            break;
        result *= 16;
        result += addition;
    }
    return result;
}

- (void)readChunkedBodyIntoStream:(OWDataStream *)dataStream;
{
    unsigned int contentLength, bytesLeft;
    NSAutoreleasePool *autoreleasePool = nil;
    void *dataStreamBuffer;
    unsigned int dataStreamBytesAvailable, socketBytesWritten;
    unsigned int byteCount, bytesInThisPool;

    while (YES) {
        autoreleasePool = [[NSAutoreleasePool alloc] init];

        if (!(contentLength = [self intValueFromHexString:[socketStream readLine]]))
            break;

        bytesInThisPool = 0;
        byteCount = 0;
        bytesLeft = contentLength;
        // NSLog(@"%@ readChunkedBody: start: processed bytes %d of %d for dataStream %@, bytesLeft = %d", OBShortObjectDescription(self), byteCount, contentLength, OBShortObjectDescription(dataStream), bytesLeft);
        [nonretainedProcessor processedBytes:byteCount ofBytes:contentLength];

        while (bytesLeft) {
            dataStreamBytesAvailable = MIN([dataStream appendToUnderlyingBuffer:&dataStreamBuffer], bytesLeft);
            socketBytesWritten = [socketStream readBytesWithMaxLength:dataStreamBytesAvailable intoBuffer:dataStreamBuffer];
            if (!socketBytesWritten)
                break;
                
            byteCount += socketBytesWritten;
            bytesLeft -= socketBytesWritten;
            bytesInThisPool += socketBytesWritten;
            [nonretainedProcessor processedBytes:byteCount ofBytes:contentLength];
            // NSLog(@"%@ readChunkedBody: processed bytes %d of %d for dataStream %@, bytesLeft = %d", OBShortObjectDescription(self), byteCount, contentLength, OBShortObjectDescription(dataStream), bytesLeft);
            [dataStream wroteBytesToUnderlyingBuffer:socketBytesWritten];
            // NSLog(@"%@ readChunkedBody: wrote data to %@", OBShortObjectDescription(self), OBShortObjectDescription(dataStream));
            if (bytesInThisPool > 64 * 1024) {
                [autoreleasePool release];
                autoreleasePool = [[NSAutoreleasePool alloc] init];
                bytesInThisPool = 0;
            }
            if (fetchFlags.aborting)
                [NSException raise:@"User Aborted" format:NSLocalizedStringFromTableInBundle(@"User Stopped", @"OWF", [self bundle], httpsession error)];
        }
        [socketStream readLine];
        [autoreleasePool release];
    }
    [headerDictionary readRFC822HeadersFromSocketStream:socketStream];
}

- (void)readStandardBodyIntoStream:(OWDataStream *)dataStream;
{
    unsigned int contentLength, bytesLeft;
    NSAutoreleasePool *autoreleasePool = nil;
    void *dataStreamBuffer;
    unsigned int dataStreamBytesAvailable, socketBytesWritten;
    unsigned int byteCount, bytesInThisPool;

    contentLength = [[headerDictionary lastStringForKey:@"content-length"] intValue];

    autoreleasePool = [[NSAutoreleasePool alloc] init];
    bytesInThisPool = 0;
    byteCount = 0;
    bytesLeft = contentLength;
    // NSLog(@"%@ readStandardBody: start: processed bytes %d of %d for dataStream %@, bytesLeft = %d", OBShortObjectDescription(self), byteCount, contentLength, OBShortObjectDescription(dataStream), bytesLeft);
    [nonretainedProcessor processedBytes:byteCount ofBytes:contentLength];

    while (bytesLeft) {
        dataStreamBytesAvailable = MIN([dataStream appendToUnderlyingBuffer:&dataStreamBuffer], bytesLeft);
        socketBytesWritten = [socketStream readBytesWithMaxLength:dataStreamBytesAvailable intoBuffer:dataStreamBuffer];
        if (!socketBytesWritten)
            break;

        byteCount += socketBytesWritten;
        bytesLeft -= socketBytesWritten;
        bytesInThisPool += socketBytesWritten;
        [nonretainedProcessor processedBytes:byteCount ofBytes:contentLength];
        // NSLog(@"%@ readStandardBody: processed bytes %d of %d for dataStream %@, bytesLeft = %d", OBShortObjectDescription(self), byteCount, contentLength, OBShortObjectDescription(dataStream), bytesLeft);
        [dataStream wroteBytesToUnderlyingBuffer:socketBytesWritten];
        // NSLog(@"%@ readStandardBody: wrote data to %@", OBShortObjectDescription(self), OBShortObjectDescription(dataStream));
        if (bytesInThisPool > 64 * 1024) {
            [autoreleasePool release];
            autoreleasePool = [[NSAutoreleasePool alloc] init];
            bytesInThisPool = 0;
        }
        if (fetchFlags.aborting)
            [NSException raise:@"User Aborted" format:NSLocalizedStringFromTableInBundle(@"User Stopped", @"OWF", [self bundle], httpsession error)];
    }
    [autoreleasePool release];
}

- (void)readClosingBodyIntoStream:(OWDataStream *)dataStream;
{
    NSAutoreleasePool *autoreleasePool = nil;
    void *dataStreamBuffer;
    unsigned int dataStreamBytesAvailable, socketBytesWritten;
    unsigned int byteCount, bytesInThisPool;

    autoreleasePool = [[NSAutoreleasePool alloc] init];
    bytesInThisPool = 0;
    byteCount = 0;
    // NSLog(@"%@ readClosingBody: start: processed bytes %d for dataStream %@", OBShortObjectDescription(self), byteCount, OBShortObjectDescription(dataStream));
    [nonretainedProcessor processedBytes:byteCount ofBytes:0];

    NS_DURING {
        while (1) {
            dataStreamBytesAvailable = [dataStream appendToUnderlyingBuffer:&dataStreamBuffer];
            socketBytesWritten = [socketStream readBytesWithMaxLength:dataStreamBytesAvailable intoBuffer:dataStreamBuffer];
            if (!socketBytesWritten)
                break;

            byteCount += socketBytesWritten;
            bytesInThisPool += socketBytesWritten;
            [nonretainedProcessor processedBytes:byteCount ofBytes:0];
            // NSLog(@"%@ readClosingBody: processed bytes %d for dataStream %@", OBShortObjectDescription(self), byteCount, OBShortObjectDescription(dataStream));
            [dataStream wroteBytesToUnderlyingBuffer:socketBytesWritten];
            // NSLog(@"%@ readClosingBody: wrote data to %@", OBShortObjectDescription(self), OBShortObjectDescription(dataStream));
            if (bytesInThisPool > 64 * 1024) {
                [autoreleasePool release];
                autoreleasePool = [[NSAutoreleasePool alloc] init];
                bytesInThisPool = 0;
            }
            if (fetchFlags.aborting)
                break;
        }        
    } NS_HANDLER {
    } NS_ENDHANDLER;

    if (fetchFlags.aborting)
        [NSException raise:@"User Aborted" format:NSLocalizedStringFromTableInBundle(@"User Stopped", @"OWF", [self bundle], httpsession error)];
    [autoreleasePool release];
}

// Exception handling

- (void)notifyProcessor:(OWHTTPProcessor *)aProcessor ofSessionException:(NSException *)sessionException;
{
    NS_DURING {
        [aProcessor handleSessionException:sessionException];
        [aProcessor processEnd];
        [aProcessor retire];
    } NS_HANDLER {
        NSLog(@"Exception trying to notify processor of session exception: sessionException = %@, localException = %@", sessionException, localException);
    } NS_ENDHANDLER;
}

@end

@implementation OWHTTPSession (OWDeprecatedAPI)

// This is just here for backwards compatibility (specifically, so the HTTPS plug-in doesn't break).

- (void)disconnect; // deprecated
{
    [self disconnectAndRequeueProcessors];
}

@end
