// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWCSSStyleSheet.h"

#import <OmniBase/OmniBase.h>
#import <Foundation/Foundation.h>
#import <OmniFoundation/OmniFoundation.h>

#import "OWAddress.h"
#import "OWContentInfo.h"
#import "OWContentType.h"
#import "OWCSSIdentifier.h"
#import "OWCSSNumber.h"
#import "OWCSSSelector.h"
#import "OWCSSSelectorGroup.h"
#import "OWCSSTokenizer.h"
#import "OWDataStream.h"
#import "OWDataStreamCharacterCursor.h"
#import "OWDataStreamScanner.h"
#import "OWCSSDeclarations.h"
#import "OWObjectStreamCursor.h"
#import "OWSGMLProcessor.h"
#import "OWSGMLTag.h"
#import "OWSimpleTarget.h"
#import "OWWebPipeline.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/CSS.subproj/OWCSSStyleSheet.m,v 1.69 2001/03/17 06:45:26 wjs Exp $")


@interface OWCSSStyleSheet (Private)
+ (void)_parseIntoDeclarations:(OWCSSDeclarations *)declarations fromTokenizer:(OWCSSTokenizer *)tokenizer;
+ (BOOL)_parseDeclarationIntoDeclarations:(OWCSSDeclarations *)declarations fromTokenizer:(OWCSSTokenizer *)tokenizer;
+ (BOOL)_atEndOfDeclarationForTokenizer:(OWCSSTokenizer *)tokenizer;
// parse generic values
+ (BOOL)_parseValueIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer allowedTokensMask:(unsigned int)allowedTokensMask;
// parse specific values
+ (BOOL)_parseBackgroundIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
+ (BOOL)_parseColorValueIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
+ (BOOL)_parseFontIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
+ (BOOL)_parseFontFamiliesIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
+ (BOOL)_parseTextDecorationsIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
//
- (void)_parseStyleSheetFromScanner:(OFCharacterScanner *)aScanner parentContentInfo:(OWContentInfo *)parentContentInfo;
//
- (void)_addSelectorGroup:(OWCSSSelectorGroup *)group;
- (void)_addDeclarations:(OWCSSDeclarations *)declarations inDictionary:(NSDictionary *)dictionary forName:(NSString *)name tag:(OWSGMLTag *)tag tagStack:(OFStaticArray *)tagStack;
-(BOOL)_checkTagStack:(OFStaticArray *)tagStack forSelectorGroup:(OWCSSSelectorGroup *)group;
@end

@interface NSObject (GettingTagFromStackItem)
- (OWSGMLTag *)tag;
@end


@implementation OWCSSStyleSheet

unsigned int OWFDebugStyleSheetsLevel = 0;

static OWContentType *cssContentType;
static BOOL weKnowIfWeRepondToCompileColorValue = NO;
static BOOL weRepondToCompileColorValue;
typedef BOOL (*parseIMPType)(id, SEL, OWCSSDeclarations *declarations, CSSDeclarationIndexes declarationIndex, OWCSSTokenizer *tokenizer); 
static struct {
    parseIMPType parseIMP;
    SEL selector;
    unsigned int allowedTokensMask;
} OWCSSPropertyParsingTable[CSSDeclarationsCount];

+ (void)initialize;
{
    unsigned int propertyIndex;
    
    OBINITIALIZE;

    cssContentType = [OWContentType contentTypeForString:@"text/css"];
    
    for (propertyIndex = 0; propertyIndex < CSSDeclarationsCount; propertyIndex++) {
        OWCSSPropertyParsingTable[propertyIndex].selector = NULL;
        OWCSSPropertyParsingTable[propertyIndex].parseIMP = NULL;
        OWCSSPropertyParsingTable[propertyIndex].allowedTokensMask = 0;
    }
    
#define OWCSSUseCustomParsingSelectorStartingWith(parseTableIndex, selectorStart) {\
    OWCSSPropertyParsingTable[parseTableIndex].selector = @selector(selectorStart##declarationIndex:fromTokenizer:); \
    OWCSSPropertyParsingTable[parseTableIndex].parseIMP = (parseIMPType)[self methodForSelector:OWCSSPropertyParsingTable[parseTableIndex].selector]; \
}

    // Background
    OWCSSUseCustomParsingSelectorStartingWith(CSSBackgroundDeclarationIndex, _parseBackgroundIntoDeclarations:);
    OWCSSUseCustomParsingSelectorStartingWith(CSSBackgroundColorDeclarationIndex, _parseColorValueIntoDeclarations:);
    OWCSSPropertyParsingTable[CSSBackgroundImageDeclarationIndex].allowedTokensMask = OWCSSTokenFunction;
    // Bottom
    OWCSSPropertyParsingTable[CSSBottomDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Clear
    OWCSSPropertyParsingTable[CSSClearDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    // Color
    OWCSSUseCustomParsingSelectorStartingWith(CSSColorDeclarationIndex, _parseColorValueIntoDeclarations:);
    // Display
    OWCSSPropertyParsingTable[CSSDisplayDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    // Float
    OWCSSPropertyParsingTable[CSSFloatDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    // Font
    OWCSSUseCustomParsingSelectorStartingWith(CSSFontDeclarationIndex, _parseFontIntoDeclarations:);
    OWCSSUseCustomParsingSelectorStartingWith(CSSFontFamilyDeclarationIndex, _parseFontFamiliesIntoDeclarations:);
    OWCSSPropertyParsingTable[CSSFontSizeDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    OWCSSPropertyParsingTable[CSSFontSizeAdjustDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    OWCSSPropertyParsingTable[CSSFontStretchDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    OWCSSPropertyParsingTable[CSSFontStyleDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    OWCSSPropertyParsingTable[CSSFontVariantDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    OWCSSPropertyParsingTable[CSSFontWeightDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Height
    OWCSSPropertyParsingTable[CSSHeightDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Left
    OWCSSPropertyParsingTable[CSSLeftDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Line-height
    OWCSSPropertyParsingTable[CSSLineHeightDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Margin
    OWCSSPropertyParsingTable[CSSMarginDeclarationIndex].allowedTokensMask = 0; // Not written yet
    OWCSSPropertyParsingTable[CSSMarginBottomDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    OWCSSPropertyParsingTable[CSSMarginLeftDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    OWCSSPropertyParsingTable[CSSMarginRightDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    OWCSSPropertyParsingTable[CSSMarginTopDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Position
    OWCSSPropertyParsingTable[CSSPositionDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    // Right
    OWCSSPropertyParsingTable[CSSRightDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Text
    OWCSSPropertyParsingTable[CSSTextAlignDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenString;
    OWCSSUseCustomParsingSelectorStartingWith(CSSTextDecorationDeclarationIndex, _parseTextDecorationsIntoDeclarations:);
    // Top
    OWCSSPropertyParsingTable[CSSTopDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Visibility
    OWCSSPropertyParsingTable[CSSVisibilityDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    // White-space
    OWCSSPropertyParsingTable[CSSWhiteSpaceDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier;
    // Width
    OWCSSPropertyParsingTable[CSSWidthDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    // Z-index
    OWCSSPropertyParsingTable[CSSZIndexDeclarationIndex].allowedTokensMask = OWCSSTokenIdentifier|OWCSSTokenNumber;
    
#undef OWCSSUseCustomParsingSelectorStartingWith
}

+ (void)parseIntoDeclarations:(OWCSSDeclarations *)declarations fromString:(NSString *)declarationsString;
{
    OFStringScanner *scanner;
    OWCSSTokenizer *tokenizer;
    
    scanner = [[OFStringScanner alloc] initWithString:declarationsString];
    tokenizer = [[OWCSSTokenizer alloc] initWithScanner:scanner];
    [scanner release];
    [self _parseIntoDeclarations:declarations fromTokenizer:tokenizer];
    [tokenizer release];
}

// Init and dealloc

- init;
{
    if (![super init])
        return nil;
    currentPosition = 0;
    return self;
}

- (void)dealloc;
{
    [lastSelectorsByTag release];
    [lastSelectorsByClass release];
    [lastSelectorsByID release];
    [lastSelectorsByPseudoclass release];
    [super dealloc];
}

// API

- (void)parseStyleSheetString:(NSString *)sheet parentContentInfo:(OWContentInfo *)parentContentInfo;
{
    OFCharacterScanner *scanner;

    scanner = [[OFStringScanner alloc] initWithString:sheet];
    [self _parseStyleSheetFromScanner:scanner parentContentInfo:parentContentInfo];
    [scanner release];
}

- (void)parseStyleSheetFromURLString:(NSString *)urlString parentContentInfo:(OWContentInfo *)parentContentInfo;
{
    OWAddress *sourceAddress;
    OWSimpleTarget *fetchTarget;
    id <OWContent> fetchedContent;
    OWDataStream *scriptDataStream;
    OWDataStreamCharacterCursor *scriptDataCursor;
    OWDataStreamScanner *dataStreamScanner;
    OWPipeline *sheetPipeline;
    OWContentInfo *sheetContentInfo;

    if (urlString == nil)
        return;

    sourceAddress = [(OWAddress *)[parentContentInfo address] addressForRelativeString:urlString];
    if (sourceAddress == nil)
        return;

    fetchTarget = [[OWSimpleTarget alloc] initWithParentContentInfo:parentContentInfo targetContentType:cssContentType];
    [fetchTarget setAcceptsAlternateContent:YES];
    sheetPipeline = [[OWWebPipeline alloc] initWithContent:sourceAddress target:fetchTarget];
    [sheetPipeline startProcessingContent];
    fetchedContent = [fetchTarget resultingContent];
    [parentContentInfo addChildFossil:fetchTarget];
    [fetchTarget release];
    if (![fetchedContent isKindOfClass:[OWDataStream class]])
        return;

    sheetContentInfo = [sheetPipeline contentInfo];
    OBASSERT(sheetContentInfo != nil);
    scriptDataStream = (OWDataStream *)fetchedContent;
    // Using an OWDataStreamCharacterCursor automatically checks for the charset header, obeys preferences, etc.
    scriptDataCursor = [[OWDataStreamCharacterCursor alloc] initForDataCursor:[scriptDataStream newCursor]];
    dataStreamScanner = [[OWDataStreamScanner alloc] initWithCursor:scriptDataCursor];
    [scriptDataCursor release];
    [self _parseStyleSheetFromScanner:dataStreamScanner parentContentInfo:sheetContentInfo];
    [dataStreamScanner release];
    [sheetPipeline release];
}


// Fill in declarations object with relevant styles

- (void)addDeclarations:(OWCSSDeclarations *)declarations cssTagName:(NSString *)name tag:(OWSGMLTag *)tag tagStack:(OFStaticArray *)tagStack;
{
    return [self _addDeclarations:declarations inDictionary:lastSelectorsByTag forName:name tag:tag tagStack:tagStack];
}

- (void)addDeclarations:(OWCSSDeclarations *)declarations cssClassName:(NSString *)name tag:(OWSGMLTag *)tag tagStack:(OFStaticArray *)tagStack;
{
    return [self _addDeclarations:declarations inDictionary:lastSelectorsByClass forName:name tag:tag tagStack:tagStack];
}

- (void)addDeclarations:(OWCSSDeclarations *)declarations cssIDName:(NSString *)name tag:(OWSGMLTag *)tag tagStack:(OFStaticArray *)tagStack;
{
    return [self _addDeclarations:declarations inDictionary:lastSelectorsByID forName:name tag:tag tagStack:tagStack];
}

- (void)addDeclarations:(OWCSSDeclarations *)declarations cssPseudoClassName:(NSString *)name tag:(OWSGMLTag *)tag tagStack:(OFStaticArray *)tagStack;
{
    return [self _addDeclarations:declarations inDictionary:lastSelectorsByPseudoclass forName:name tag:tag tagStack:tagStack];
}


@end


@implementation OWCSSStyleSheet (Private)

/* This wants to parse strings of the form:
{
    property : value value ;
    property : value
    property : any ( any ) any [ any ] any { any }
}
eg:
{	padding: 4px;
	background-color: #ff7f6b;
	border-bottom: 2px white solid;
	background: url( http://foo/bar.gif );
	background: url( "http://foo;/bar{}.gif" );
	font: 14px/14pt ";Trebuchet \"MS\"", Trebuchet, Arial, sans-serif;
	font-weight: bold
}
*/
#warning -parseDeclaration: currently ignores things like "!important"
+ (void)_parseIntoDeclarations:(OWCSSDeclarations *)declarations fromTokenizer:(OWCSSTokenizer *)tokenizer;
{
    while (1) {
        id tokenValue;

        switch ([tokenizer getNextToken:&tokenValue]) {
            case OWCSSTokenEOF:
                return;
            case OWCSSTokenWhitespace:
                continue;
            case OWCSSTokenPunctuation: {
                unichar punctuation;
                
                punctuation = [tokenValue characterAtIndex:0];
                if (punctuation == '{' || punctuation == ';')
                    continue;
                else if (punctuation == '}')
                    return;
                else
                    break;
            }
            case OWCSSTokenString:
            case OWCSSTokenIdentifier:
                [tokenizer ungetLastToken];
                if ([self _parseDeclarationIntoDeclarations:declarations fromTokenizer:tokenizer])
                    continue;
                break;
            default:
                break;
        }
        
        // Skip to the start of the next declaration if the last one didn't parse correctly
        [tokenizer skipTokensUpToAndIncludingPunctuation:@";}"];
        [tokenizer ungetLastToken];
    }
}

+ (BOOL)_parseDeclarationIntoDeclarations:(OWCSSDeclarations *)declarations fromTokenizer:(OWCSSTokenizer *)tokenizer;
{
    OWCSSTokenType tokenType;
    id tokenValue;
    OWCSSTokenType propertyTokenType;
    OWCSSIdentifier *propertyName;
    unsigned int propertyParsingTableIndex;

    // { [property] : value value ; property value }
    propertyTokenType = [tokenizer getNextToken:&propertyName]; // We know we're a string or identifier when we get here
//    if (propertyTokenType == OWCSSTokenString) propertyName = [propertyName lowercaseString];
    
    // { property[ ]: value value ; property value }
    tokenType = [tokenizer getNextToken:&tokenValue skipWhitespace:YES]; // NOTE: Strict CSS2 guidelines don't allow whitespace here anyhow, but who are we hurting?
        
    // { property [:] value value ; property value }
    if (tokenType != OWCSSTokenPunctuation || ![tokenValue isEqualToString:@":"])
        return NO;

    // { property :[ ]value value ; property value }
    tokenType = [tokenizer getNextToken:&tokenValue];
    if (tokenType != OWCSSTokenWhitespace)
        [tokenizer ungetLastToken];
    
    
    // { property :[ value value ]garbage ; property value }
    if (propertyTokenType != OWCSSTokenIdentifier) // Property wasn't a valid identifier
        return NO;
   
    propertyParsingTableIndex = [propertyName declarationIndex];     
    if (propertyParsingTableIndex == NSNotFound) // Property was an identifier, but not one we recognize as being a property name (eg, "url" would be bad)
        return NO;

    if (OWCSSPropertyParsingTable[propertyParsingTableIndex].parseIMP != NULL) {
        return OWCSSPropertyParsingTable[propertyParsingTableIndex].parseIMP(self,  OWCSSPropertyParsingTable[propertyParsingTableIndex].selector, declarations, propertyParsingTableIndex, tokenizer);
    } else if (OWCSSPropertyParsingTable[propertyParsingTableIndex].allowedTokensMask > 0) {
        return [self _parseValueIntoDeclarations:declarations declarationIndex:propertyParsingTableIndex fromTokenizer:tokenizer allowedTokensMask:OWCSSPropertyParsingTable[propertyParsingTableIndex].allowedTokensMask];
    }
    
    return NO;
}

+ (BOOL)_atEndOfDeclarationForTokenizer:(OWCSSTokenizer *)tokenizer;
{
    OWCSSTokenType tokenType;
    id tokenValue;

    tokenType = [tokenizer getNextToken:&tokenValue];
    [tokenizer ungetLastToken];
    if (tokenType == OWCSSTokenPunctuation) {
        unichar character;
        
        character = [tokenValue characterAtIndex:0];
        return (character == ';' || character == '}');
    } else
        return NO;
}

// parse generic values

+ (BOOL)_parseValueIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer allowedTokensMask:(unsigned int)allowedTokensMask;
{
    id value;
    OWCSSTokenType tokenType;
        
    tokenType = [tokenizer getNextToken:&value];

    if (tokenType & allowedTokensMask) {
        [declarations setObject:value atIndex:declarationIndex];
        return [self _atEndOfDeclarationForTokenizer:tokenizer];
    } else {
        [tokenizer ungetLastToken];
        return NO;
    }
}


// parse specific values

// background: [<'background-color'> || <'background-image'> || <'background-repeat'> || <'background-attachment'> || <'background-position'>]+ | inherit
// background: red
// background: url("chess.png") gray 50% repeat fixed
+ (BOOL)_parseBackgroundIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
{
    OWCSSTokenType tokenType;
    id backgroundValue;

    while (1) {
        tokenType = [tokenizer getNextToken:&backgroundValue excludingIdentifiers:NO excludingNumbers:NO];

        switch (tokenType) {
            case OWCSSTokenFunction:
                [declarations setObject:backgroundValue atIndex:CSSBackgroundImageDeclarationIndex];
                break;
            case OWCSSTokenString:
            case OWCSSTokenIdentifier: // Warning WJS 2/15/2001: When we extend what background attributes we parse, we need to make this more complex, because right now if the user says something like the second example above, the word "fixed" will be stored in our background color attribute.
                [tokenizer ungetLastToken];
                if ([self _parseColorValueIntoDeclarations:declarations declarationIndex:CSSBackgroundColorDeclarationIndex fromTokenizer:tokenizer])
                    return YES;
                break;
            default:
                [tokenizer ungetLastToken];
                return [self _atEndOfDeclarationForTokenizer:tokenizer];
        }
    }
    return YES; // NOTREACHED
}

// color: black
// color: #f00              /* #rgb */
// color: #ff0000           /* #rrggbb */
// color: ff0000		/* not-up-to-spec format used on excite.com */
// color: rgb(255,0,0)      /* integer range 0 - 255 */
// color: rgb(100%, 0%, 0%) /* float range 0.0% - 100.0% */
+ (BOOL)_parseColorValueIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
{
    OWCSSTokenType tokenType;
    id colorValue;

    if (OWFDebugStyleSheetsLevel >= 2) NSLog(@"_readIntoColorValue:fromTokenizer:");

    tokenType = [tokenizer getNextToken:&colorValue excludingIdentifiers:NO excludingNumbers:YES];
    if (OWFDebugStyleSheetsLevel >= 1) NSLog(@"\tgot color value:%@", colorValue);

    if (!weKnowIfWeRepondToCompileColorValue) {
        // multi-thread race condition actually has no ill effects if you do statements in this order:
        weRepondToCompileColorValue = [self respondsToSelector:@selector(compileColorValue:)];
        weKnowIfWeRepondToCompileColorValue = YES;
    }
    if (weRepondToCompileColorValue) {
        colorValue = [self compileColorValue:colorValue];
        if (OWFDebugStyleSheetsLevel >= 2) NSLog(@"\tcompiled color value to:%@", colorValue);
    }
    
    [declarations setObject:colorValue atIndex:declarationIndex];
    return [self _atEndOfDeclarationForTokenizer:tokenizer];
}

// font: [ [ <font-style> || <font-variant> || <font-weight> ]? <font-size> [ / <line-height> ]? <font-family> ] | caption | icon | menu | message-box | small-caption | status-bar | inherit
+ (BOOL)_parseFontIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
{
    OWCSSTokenType tokenType;
    id tokenValue, fontSizeValue;
    BOOL stylesDone = NO;
    
    tokenType = [tokenizer getNextToken:&tokenValue];
    // caption | icon | menu | message-box | small-caption | status-bar | inherit
    if (tokenValue == OWCSSCaptionIdentifier || tokenValue == OWCSSIconIdentifier || tokenValue == OWCSSMenuIdentifier || tokenValue == OWCSSMessageBoxIdentifier || tokenValue == OWCSSSmallCaptionIdentifier || tokenValue == OWCSSStatusBarIdentifier || tokenValue == OWCSSInheritIdentifier) {
        [declarations setObject:tokenValue atIndex:declarationIndex];
        if (OWFDebugStyleSheetsLevel) NSLog(@"FONT: ident %@", tokenValue);
        return [self _atEndOfDeclarationForTokenizer:tokenizer];
    }

    do { // [ <font-style> || <font-variant> || <font-weight> ]?
        switch (tokenType) {
            case OWCSSTokenIdentifier:
                if ([tokenValue isFontStyleIdentifier]) // font-style: normal | italic | oblique | inherit
                    [declarations setObject:tokenValue atIndex:CSSFontStyleDeclarationIndex];
                else if ([tokenValue isFontVariantIdentifier]) // font-variant: normal | small-caps | inherit
                    [declarations setObject:tokenValue atIndex:CSSFontVariantDeclarationIndex];
                else if ([tokenValue isFontWeightIdentifier]) // font-weight: normal | bold | bolder | lighter | 100 | 200 | 300 | 400 | 500 | 600 | 700 | 800 | 900 | inherit 
                    [declarations setObject:tokenValue atIndex:CSSFontWeightDeclarationIndex];
                else
                    stylesDone = YES;
                break;
            case OWCSSTokenNumber:
                if ([tokenValue unitsIdentifier] == nil)
                    [declarations setObject:tokenValue atIndex:CSSFontWeightDeclarationIndex];
                else
                    stylesDone = YES;
                break;
            case OWCSSTokenWhitespace:
                break;
            case OWCSSTokenPunctuation: {
                unichar character;
                
                character = [tokenValue characterAtIndex:0];
                [tokenizer ungetLastToken];
                return (character == ';' || character == '}');
            }
            case OWCSSTokenEOF:
                return YES;
            default:
                stylesDone = YES;
                break;
        }
        if (OWFDebugStyleSheetsLevel) NSLog(@"FONT: maybe style %@, isStyle %d", tokenValue, !stylesDone);
        if (!stylesDone)
            tokenType = [tokenizer getNextToken:&tokenValue];
    } while (!stylesDone);
    
    // <font-size>
    fontSizeValue = tokenValue;
    if (tokenType == OWCSSTokenIdentifier && [fontSizeValue isFontSizeIdentifier]) // font-size: [xx-small | x-small | small | medium | large | x-large | xx-large ] | [ larger | smaller ] | <length> | <percentage> | inherit 
        [declarations setObject:fontSizeValue atIndex:CSSFontSizeDeclarationIndex];
    else if (tokenType == OWCSSTokenNumber && [fontSizeValue unitsIdentifier] != nil)
        [declarations setObject:fontSizeValue atIndex:CSSFontSizeDeclarationIndex];
    else
        return NO;
    if (OWFDebugStyleSheetsLevel) NSLog(@"FONT: font-size %@", fontSizeValue);
        
    // [ / <line-height> ]? 
    tokenType = [tokenizer getNextToken:&tokenValue skipWhitespace:YES];
    if (tokenType == OWCSSTokenPunctuation) {
        unichar character;
            
        character = [tokenValue characterAtIndex:0];
        if (character == ';' || character == '}') {
            [tokenizer ungetLastToken];
            return YES;
        }
        if (character == '/') {
            tokenType = [tokenizer getNextToken:&tokenValue skipWhitespace:YES];
            if (OWFDebugStyleSheetsLevel) NSLog(@"FONT: supposed line-height %@", tokenValue);
            if (tokenValue == OWCSSNormalIdentifier || tokenValue == OWCSSInheritIdentifier)
                [declarations setObject:tokenValue atIndex:CSSLineHeightDeclarationIndex];
            else if (tokenType == OWCSSTokenNumber) {
                if ([tokenValue unitsIdentifier] != nil)
                    [declarations setObject:tokenValue atIndex:CSSLineHeightDeclarationIndex];
                else
                    [declarations setObject:[OWCSSNumber numberWithFloatValue:[tokenValue floatValue] unitsIdentifier:[fontSizeValue unitsIdentifier]] atIndex:CSSLineHeightDeclarationIndex];
            } else
                [tokenizer ungetLastToken];
        } else
            [tokenizer ungetLastToken];
    } else
        [tokenizer ungetLastToken];
    
    // <font-family>
    return [self _parseFontFamiliesIntoDeclarations:declarations declarationIndex:CSSFontFamilyDeclarationIndex fromTokenizer:tokenizer];
}

// font-family: [[ <family-name> | <generic-family> ],]* [ <family-name> | <generic-family> ] | inherit
/*
<family-name> 
The name of a font family of choice. In the previous example, "Baskerville", "Heisi Mincho W3", and "Symbol" are font families. Font family names containing whitespace should be quoted. If quoting is omitted, any whitespace characters before and after the font name are ignored and any sequence of whitespace characters inside the font name is converted to a single space. 
<generic-family> 
The following generic families are defined: 'serif', 'sans-serif', 'cursive', 'fantasy', and 'monospace'. Please see the section on generic font families for descriptions of these families. Generic font family names are keywords, and therefore must not be quoted. 
*/
+ (BOOL)_parseFontFamiliesIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
{
    OWCSSTokenType tokenType;
    NSMutableArray *mutableFamilyNames;

    mutableFamilyNames = [[[NSMutableArray alloc] init] autorelease];

    if (OWFDebugStyleSheetsLevel >= 2) NSLog(@"_readIntoFontFamilies:fromTokenizer:");

    while (1) {
        id familyName;
        
        tokenType = [tokenizer getNextToken:&familyName];
        if (OWFDebugStyleSheetsLevel >= 1) NSLog(@"\tgot family name:%@", familyName);

        if (tokenType == OWCSSTokenString || tokenType == OWCSSTokenIdentifier) {
            while (1) {
                OWCSSTokenType furtherTokenType;
                id furtherTokenValue;
            
                furtherTokenType = [tokenizer getNextToken:&furtherTokenValue excludingIdentifiers:YES excludingNumbers:YES];
                if (OWFDebugStyleSheetsLevel >= 1) NSLog(@"\tgot further token:%@", furtherTokenValue);
    
                if (furtherTokenType == OWCSSTokenString && tokenType == OWCSSTokenString) {
                    familyName = [[familyName stringByAppendingString:@" "] stringByAppendingString:furtherTokenValue];
                    continue;
                } else if (furtherTokenType == OWCSSTokenPunctuation) {
                    unichar character;
                    
                    character = [furtherTokenValue characterAtIndex:0];
                    if (character == ',') {
                        [mutableFamilyNames addObject:familyName];
                        [declarations setObject:mutableFamilyNames atIndex:declarationIndex];
                        break;
                    } else if (character == ';' || character == '}') {
                        [mutableFamilyNames addObject:familyName];
                        [declarations setObject:mutableFamilyNames atIndex:declarationIndex];
                        [tokenizer ungetLastToken];
                        return YES;
                    } else {
                        [tokenizer ungetLastToken];
                        return NO;
                    }
                } else if (furtherTokenType != OWCSSTokenWhitespace) {
                    [tokenizer ungetLastToken];
                    return NO;
                }
            }
            
        } else if (tokenType != OWCSSTokenWhitespace) {
            [tokenizer ungetLastToken];
            return NO;
        }
    }
    
    return YES; // NOT REACHED
}

// text-decoration: none | [ underline || overline || line-through || blink ] | inherit
+ (BOOL)_parseTextDecorationsIntoDeclarations:(OWCSSDeclarations *)declarations declarationIndex:(CSSDeclarationIndexes)declarationIndex fromTokenizer:(OWCSSTokenizer *)tokenizer;
{
    OWCSSTokenType tokenType;
    id returnObject = nil;
    NSMutableArray *mutableDecorations = nil;
    BOOL returnValue = NO;

    while (1) {
        id nextToken;
        unichar character;
        
        tokenType = [tokenizer getNextToken:&nextToken];

        switch (tokenType) {
            case OWCSSTokenIdentifier:
                if (returnObject == nil) {
                    returnObject = nextToken;
                } else {
                    if (mutableDecorations == nil) {
                        mutableDecorations = [[[NSMutableArray alloc] init] autorelease];
                        [mutableDecorations addObject:returnObject];
                        returnObject = mutableDecorations;
                    }
                    [mutableDecorations addObject:nextToken];
                }
                break;
                
            case OWCSSTokenWhitespace:
                break;
                
            case OWCSSTokenPunctuation:
                character = [nextToken characterAtIndex:0];
                if (character == ';' || character == '}')
                    returnValue = YES;
                // fall through
            default:
                [tokenizer ungetLastToken];
                [declarations setObject:returnObject atIndex:declarationIndex];
                return returnValue;
        }
    }
    
    return YES; // NOT REACHED
}


//

- (void)_parseStyleSheetFromScanner:(OFCharacterScanner *)aScanner parentContentInfo:(OWContentInfo *)parentContentInfo;
{
    OWCSSTokenizer *tokenizer;
    OWCSSTokenType tokenType;
    id tokenValue;
    BOOL importsAllowed = YES;

    if (aScanner == nil)
        return;

    tokenizer = [[OWCSSTokenizer alloc] initWithScanner:aScanner];
    [tokenizer autorelease];
        
    if (OWFDebugStyleSheetsLevel >= 4) {
        while ((tokenType = [tokenizer getNextToken:&tokenValue]) != OWCSSTokenEOF) {
            switch (tokenType) {
                case OWCSSTokenString:
                    NSLog(@"string <%@>", tokenValue);
                    break;
                case OWCSSTokenPunctuation:
                    NSLog(@"punctuation '%@'", tokenValue);
                    break;
                case OWCSSTokenWhitespace:
                    NSLog(@"whitespace", tokenValue);
                    break;
                case OWCSSTokenNumber:
                    NSLog(@"number %.1f%@", [tokenValue floatValue], [tokenValue unitsIdentifier] ? [tokenValue unitsIdentifier] : (OWCSSIdentifier *)@"");
                    break;
                case OWCSSTokenFunction:
                    NSLog(@"function %@%@", [tokenValue objectAtIndex:0], [tokenValue subarrayWithRange:NSMakeRange(1, [tokenValue count]-1)]);
                    break;
                case OWCSSTokenIdentifier:
                    NSLog(@"%@", [tokenValue description]);
                    break;
                case OWCSSTokenEOF:
                    // NOTREACHED
                    break;
            }
        }
        return;
    }

    while (((tokenType = [tokenizer getNextToken:&tokenValue]) != OWCSSTokenEOF)) {
    
        if (tokenType == OWCSSTokenWhitespace)
            continue;
    
        // at-rule, including @import
        if (tokenType == OWCSSTokenPunctuation && [tokenValue isEqualToString:@"@"]) {

            tokenType = [tokenizer getNextToken:&tokenValue];
            if (tokenType == OWCSSTokenString && [tokenValue caseInsensitiveCompare:@"import"] == NSOrderedSame) {

                if (OWFDebugStyleSheetsLevel) NSLog(@"Hey, got an import at-rule");
                
                if (importsAllowed) {
                    NSString *urlString = nil;
                    
                    tokenType = [tokenizer getNextToken:&tokenValue];
                    if (tokenType == OWCSSTokenWhitespace) // There will normally be a space after the @import statement...
                        tokenType = [tokenizer getNextToken:&tokenValue];
                    
                    if (tokenType == OWCSSTokenFunction) {
                        if ([tokenValue objectAtIndex:0] == OWCSSURLIdentifier)
                            urlString = [tokenValue objectAtIndex:1];
                    } else if (tokenType == OWCSSTokenString)
                        urlString = tokenValue;

                    if (OWFDebugStyleSheetsLevel) NSLog(@"urlString = '%@', go get it.", urlString);

                    [self parseStyleSheetFromURLString:urlString parentContentInfo:parentContentInfo];
                    
                } else { // Imports are NO LONGER allowed, because we've seen some declarations
                    if (OWFDebugStyleSheetsLevel) NSLog(@"Too bad imports aren't allowed now.");
                }
                
            } else {
                if (OWFDebugStyleSheetsLevel) NSLog(@"Hey, got some unknown ATKEYWORD: %@\nI'll just blow it away'", tokenValue);
            }

            [tokenizer skipTokensUpToAndIncludingPunctuation:@";"];
            continue;
            
        // declaration
        } else {
            OWCSSDeclarations *declarations;
            NSMutableArray *newGroups;

            [tokenizer ungetLastToken];
            newGroups = [NSMutableArray array];
            while (1) {
                OWCSSSelectorGroup *group;
                
                group = [[OWCSSSelectorGroup alloc] initWithTokenizer:tokenizer withPosition:currentPosition++];
                if (group == nil)
                    break;
                [self _addSelectorGroup:group];
                [newGroups addObject:group];
                [group release];
                
                tokenType = [tokenizer getNextToken:&tokenValue skipWhitespace:YES];
                if (OWFDebugStyleSheetsLevel) NSLog(@"Found token value %@ after selector group ", tokenValue);
                if (tokenType == OWCSSTokenPunctuation) {
                    unichar punctuation;
                    
                    punctuation = [tokenValue characterAtIndex:0];
                    if (punctuation == ',') {
                        continue;
                    } else if (punctuation == '{') {
                        [tokenizer ungetLastToken]; // Put bracket back
                        break;
                    }
                }
                    
                [tokenizer skipTokensUpToAndIncludingPunctuation:@"{"];
                [tokenizer ungetLastToken]; // Put bracket back
                break;
            }
            
            declarations = [[OWCSSDeclarations alloc] init];
            [isa _parseIntoDeclarations:declarations fromTokenizer:tokenizer];
            if (OWFDebugStyleSheetsLevel) NSLog(@"declaration dictionary: '%@'", declarations);
            
            importsAllowed = NO;
            
            // set declarations on groups 
            [newGroups makeObjectsPerformSelector:@selector(setDeclarations:) withObject:declarations];
            [declarations release];
        }
    }
}


 // Tags and pseudoclasses are NOT case sensitive, classes and IDs are.
- (void)_addSelectorGroup:(OWCSSSelectorGroup *)group;
{
    OWCSSSelector *lastSelector = [group lastSelector];
    NSString *key;
    NSMutableArray *list;
    NSMutableDictionary *selectorsDictionary;
    
    if (OWFDebugStyleSheetsLevel) NSLog(@"group = '%@'", group);
    if ((key = [lastSelector pseudoClassName])) {
        if (OWFDebugStyleSheetsLevel) NSLog(@"adding '%@' to the pseudoclass list", key);

        if (lastSelectorsByPseudoclass == nil)
            lastSelectorsByPseudoclass = OFCreateCaseInsensitiveKeyMutableDictionary();

        selectorsDictionary = lastSelectorsByPseudoclass;
    } else if ((key = [lastSelector tagName])) {
        if (OWFDebugStyleSheetsLevel) NSLog(@"adding '%@' to the tag list", key);

        if (lastSelectorsByTag == nil)
            lastSelectorsByTag = OFCreateCaseInsensitiveKeyMutableDictionary();

        selectorsDictionary = lastSelectorsByTag;
    } else if ((key = [lastSelector className])) {
        if (OWFDebugStyleSheetsLevel) NSLog(@"adding '%@' to the class list", key);

        if (lastSelectorsByClass == nil)
            lastSelectorsByClass = [[NSMutableDictionary alloc] init];

        selectorsDictionary = lastSelectorsByClass;
    } else if ((key = [lastSelector idName])) {
        if (OWFDebugStyleSheetsLevel) NSLog(@"adding '%@' to the ID list", key);

        if (lastSelectorsByID == nil)
            lastSelectorsByID = [[NSMutableDictionary alloc] init];

        selectorsDictionary = lastSelectorsByID;
    } else {
        if (OWFDebugStyleSheetsLevel) NSLog(@"I see an empty selector. Discarding.");
        return;
    }

    list = [selectorsDictionary objectForKey:key];
    if (list == nil) {
        list = [NSMutableArray array];
        [selectorsDictionary setObject:list forKey:key];
    }
    [list addObject:group];
    [list sortUsingSelector:@selector(compare:)];
    
    if (OWFDebugStyleSheetsLevel) NSLog(@"list = '%@'", list);
}

- (void)_addDeclarations:(OWCSSDeclarations *)declarations inDictionary:(NSDictionary *)dictionary forName:(NSString *)name tag:(OWSGMLTag *)tag tagStack:(OFStaticArray *)tagStack;
{
    NSArray *matchingGroups;
    unsigned int groupIndex, groupCount;

    matchingGroups = [dictionary objectForKey:name];
    if (OWFDebugStyleSheetsLevel >= 3) {
        NSMutableArray *tagNames;
        unsigned int tagIndex, tagCount;

        tagCount = [tagStack count];
        tagNames = [[NSMutableArray alloc] initWithCapacity:[tagStack count]];
        for (tagIndex = 0; tagIndex < tagCount; tagIndex++) {
            [tagNames addObject:[(OWSGMLTag *)[[tagStack objectAtIndex:tagIndex] tag] name]];
        }
        NSLog(@"-_addDeclarations:inDictionary:%@ forName:'%@' tag:<%@> tagStack:%@\nfound matching groups %@", [dictionary allKeys], name, [tag name], tagNames, matchingGroups);
        [tagNames release];
    } else if (OWFDebugStyleSheetsLevel >= 2 && matchingGroups)
        NSLog(@"-_addDeclarations:inDictionary:[skip] forName:'%@' tag:<%@> tagStack:[skip]\nfound matching groups %@", name, [tag name], matchingGroups);

    if (!matchingGroups)
        return;
        
    groupCount = [matchingGroups count];
    for (groupIndex = 0; groupIndex < groupCount; groupIndex++) {
        OWCSSSelectorGroup *group;
        OWCSSSelector *lastSelector;

        group = [matchingGroups objectAtIndex:groupIndex];
        lastSelector = [group lastSelector];

        if ([lastSelector matchesTag:tag] && (([group count] == 0) || [self _checkTagStack:tagStack forSelectorGroup:group])) {
            [declarations addEntriesFromDeclarations:[group declarations]];
        }
    }
}

-(BOOL)_checkTagStack:(OFStaticArray *)tagStack forSelectorGroup:(OWCSSSelectorGroup *)group;
{
    NSArray *selectorList = [group selectorList];
    unsigned int selectorIndex, tagIndex;
    
    // Skip the first one, since if we're here it matches the last tag.
    selectorIndex = [selectorList count];
    if (selectorIndex-- == 0)
        return NO;
    tagIndex = [tagStack count];
    
    while (selectorIndex--) {
        OWCSSSelector *selector;
        NSString *tagName;
        NSString *className;
        NSString *idName;
        BOOL stillChecking = YES;
        
        selector = [selectorList objectAtIndex:selectorIndex];
        tagName = [selector tagName];
        className = [selector className];
        idName = [selector idName];
        
        while (stillChecking && tagIndex--) {
            OWSGMLTag *tag;
            
            tag = [[tagStack objectAtIndex:tagIndex] tag];
            
            if (tagName) {
                stillChecking = ! [tagName isEqualToString:[tag name]];

                if (OWFDebugStyleSheetsLevel) NSLog(@"looking for '%@', got '%@'", tagName, [tag name]);
            }
            if (className) {
                stillChecking = ! [className isEqualToString:[tag valueForAttribute:@"class"]];
            }
            if (idName) {
                stillChecking = ! [idName isEqualToString:[tag valueForAttribute:@"id"]];
            }
        }
        if (stillChecking)
            return NO;
    }
    return YES;
}

@end
