// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.
//
// $Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/Pipelines.subproj/OWContentCache.h,v 1.16 2001/02/25 20:32:57 wiml Exp $

#import <OmniFoundation/OFObject.h>

@class NSArray, NSDate, NSRecursiveLock, NSMutableSet, NSSet;
@class OFZone;
@class OWCookie;

#import <OWF/OWAddressProtocol.h>
#import <OWF/OWContentProtocol.h>

#import <Foundation/NSDate.h> // For NSTimeInterval

@interface OWContentCache : OFObject
{
    id <OWAddress> address;
    NSRecursiveLock *cacheLock;
    NSMutableSet *contentTypes;

    // For each contentType, this contains the content that's in the cache
    NSMutableDictionary *contentDictionary;
    // Non-sharable content that is "checked out" is in this dictionary
    NSMutableDictionary *recyclableContentDictionary;
    NSMutableDictionary *expireContentEventDictionary;

    struct {
        unsigned int contentIsError:1;
    } flags;
}

+ (OWContentCache *)contentCacheForAddress:(id <OWAddress>)anAddress;
+ (OWContentCache *)lookupContentCacheForAddress:(id <OWAddress>)anAddress;
+ (void)flushCachedContent;
+ (void)flushCachedContentMatchingCookie:(OWCookie *)aCookie;
+ (void)updateMinumumExpirationTimeIntervalFromDefaults;

// Putting content in the cache
- (void)addContent:(id <OWContent>)content;
    // This adds some new content to the cache.
- (void)registerContent:(id <OWContent>)content;
    // Registers content with the cache without giving up ownership.  If the content is shareable, this simply calls -addContent:.  Otherwise, it marks the content as potentially recyclable, and starts the expiration countdown.
- (void)recycleContent:(id <OWContent>)content;
    // This returns some unshareable content to the cache.  The content must have been previously retrieved from the cache via -contentOfType: or registered via -registerContent: via it will simply be ignored.
- (OFZone *)contentZone;
    // This returns a zone for this cache key, which can be used to improve locality if you feel like it

- (BOOL)contentIsError;
    // Returns whether the cached content results from an error (e.g., a bad HTTP fetch).
- (void)setContentIsError:(BOOL)newContentIsError;
    // Sets whether the cached content results from an error.

// Retrieving content from the cache
- (id <OWAddress>)address;
    // Returns the cache address.
- (NSSet *)contentTypes;
    // Returns the types of content currently available in the cache.
- (id <OWContent>)peekAtContentOfType:(OWContentType *)contentType;
    // Doesn't restart expiration or worry about sharability.  Use only for status displays. NB that -peek...: will return non-shareable content even if it's already in use somewhere.
- (id <OWContent>)contentOfType:(OWContentType *)aContentType;
    // If the content returned is unshareable, it will become unavailable until it is recycled via -recycleContent:.

// Removing content from the cache
- (void)flushCachedContent;
    // This flushes all cached content
- (void)flushCachedContentExceptTypes:(NSSet *)keepTheseTypes;
    // This flushes all content except for OWContentTypes in the specified set
- (void)flushCachedErrorContent;
    // If the cached content resulted from an error (see -setContentIsError:), this flushes all cached content.
- (void)flushContentOfType:(OWContentType *)aContentType;
    // This flushes content of a particular type
- (void)expireAtDate:(NSDate *)expireDate;
    // This calls flushCachedContent at the specified date.
- (void)expireAfterTimeInterval:(NSTimeInterval)expireInterval;
    // This calls flushCachedContent after the specified time interval.
- (void)expireErrorContentAfterTimeInterval:(NSTimeInterval)expireInterval;
    // This calls flushCachedErrorContent after the specified time interval.
- (NSDate *)expireDateForContentOfType:(OWContentType *)contentType;
    // Returns the date for when a content type is naturally scheduled to expire.

@end

#import <OWF/FrameworkDefines.h>

OWF_EXTERN NSString *OWContentCacheFlushedCacheNotification;
