// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OWF/OWDataStream.h>

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OmniBase/system.h>

#import <OWF/OWContentCache.h>
#import <OWF/OWContentType.h>
#import <OWF/OWDataStreamCharacterProcessor.h>
#import <OWF/OWDataStreamCursor.h>
#import <OWF/OWParameterizedContentType.h>
#import <OWF/OWUnknownDataStreamProcessor.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OWF/Content.subproj/OWDataStream.m,v 1.31 2001/04/28 00:59:39 kc Exp $")

@interface OWDataStream (Private)
- (void)flushContentsToFile;
- (void)flushAndCloseSaveFile;
- (void)_noMoreData;
@end

@implementation OWDataStream

unsigned int OWDataStreamUnknownLength = NSNotFound;

static OWContentType *unknownContentType;
static OWContentType *unencodedContentEncoding;

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
	return;
    initialized = YES;

    unknownContentType = [OWUnknownDataStreamProcessor unknownContentType];
    unencodedContentEncoding = [OWContentType contentTypeForString:@"encoding/none"];
}

- initWithLength:(unsigned int)newLength;
{
    if (![super init])
	return nil;

    dataLength = newLength;
    
    dataAvailableCondition = [[OFCondition alloc] init];
    endOfDataCondition = [[OFCondition alloc] init];

    first = NSAllocateMemoryPages(sizeof(OWDataStreamBuffer));
    last = first;
    last->next = NULL;
    readLength = 0;
    dataInBuffer = 0;
    
    stringEncoding = kCFStringEncodingInvalidId;

    flags.endOfData = NO;
    flags.hasThrownAwayData = NO;
    flags.hasIssuedCursor = NO;
    flags.hasResetContentTypeAndEncoding = NO;
    
    saveFilename = nil;
    saveFileHandle = nil;

    return self;
}

- init;
{
    return [self initWithLength:OWDataStreamUnknownLength];
}

// OWAbstractContent subclass (to force inspectors to guess what we are)

- initWithName:(NSString *)name;
{
    // Normally, abstractContent gets initWithName:@"DataStream", because the init method takes the class name and creates a guess with that.  However, in this case, we don't want the guess, because we'd rather have OWPipeline's -rebuildCompositeTypeString method take a guess what to call us than to show the user the word "DataStream", which really means nothing to her.
    return [super initWithName:nil];
}

- (void)dealloc;
{
    OBASSERT(saveFileHandle == nil);
    while (first) {
        last = first->next;
        NSDeallocateMemoryPages(first, sizeof(OWDataStreamBuffer));
        first = last;
    }
    [dataAvailableCondition release];
    [endOfDataCondition release];
    [saveFilename release];
    [super dealloc];
}

- (id)newCursor;
{
    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    flags.hasIssuedCursor = YES;
    return [[[OWDataStreamCursor alloc] initForDataStream:self] autorelease];
}

- (NSData *)bufferedData;
{
    unsigned int length, totalLength;
    OWDataStreamBuffer *buffer;
    NSMutableData *result;

    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    length = readLength;
    if (length <= OWDataStreamBuffer_BufferedDataLength)
        return [NSData dataWithBytes:first->data length:length];
    result = [[NSMutableData alloc] initWithCapacity:length];
    totalLength = readLength;
    for (buffer = first; totalLength > 0; buffer = buffer->next) {
        length = MIN(totalLength, OWDataStreamBuffer_BufferedDataLength);
        [result appendBytes:buffer->data length:length];
        totalLength -= length;
    }
    return [result autorelease];
}

- (unsigned int)bufferedDataLength;
{
    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    return readLength;
}

- (unsigned int)accessUnderlyingBuffer:(void **)returnedBufferPtr startingAtLocation:(unsigned int)dataOffset;
{
    OWDataStreamBuffer *dsBuffer;
    unsigned int remainingOffset;

    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];
    if (readLength <= dataOffset)
        return 0;
    
    dsBuffer = first;
    remainingOffset = dataOffset;
    while (remainingOffset >= OWDataStreamBuffer_BufferedDataLength) {
        dsBuffer = dsBuffer->next;
        remainingOffset -= OWDataStreamBuffer_BufferedDataLength;
    }
    *returnedBufferPtr = dsBuffer->data + remainingOffset;
    return MIN(OWDataStreamBuffer_BufferedDataLength - remainingOffset, readLength - dataOffset);
}

- (unsigned int)dataLength;
{
    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    if (![self knowsDataLength])
        [endOfDataCondition waitForCondition];
    return dataLength;
}

- (BOOL)knowsDataLength;
{
    return dataLength != OWDataStreamUnknownLength;
}

- (BOOL)getBytes:(void *)buffer range:(NSRange)range;
{
    OWDataStreamBuffer *dsBuffer;
    unsigned int length;

    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    if (![self waitForBufferedDataLength:NSMaxRange(range)])
        return NO;

    dsBuffer = first;
    while (range.location >= OWDataStreamBuffer_BufferedDataLength) {
        dsBuffer = dsBuffer->next;
        range.location -= OWDataStreamBuffer_BufferedDataLength;
    }
    while ((length = MIN(range.length, OWDataStreamBuffer_BufferedDataLength - range.location))) {
        bcopy(dsBuffer->data + range.location, buffer, length);
        range.location = 0;
        range.length -= length;
        dsBuffer = dsBuffer->next;
        buffer += length;
    }
    return YES;
}

- (NSData *)dataWithRange:(NSRange)range;
{
    NSMutableData *subdata;
    OWDataStreamBuffer *dsBuffer;
    unsigned int length;

    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    if (![self waitForBufferedDataLength:NSMaxRange(range)])
        return nil;

    dsBuffer = first;
    while (range.location >= OWDataStreamBuffer_BufferedDataLength) {
        dsBuffer = dsBuffer->next;
        range.location -= OWDataStreamBuffer_BufferedDataLength;
    }
    if (range.location + range.length <= OWDataStreamBuffer_BufferedDataLength)
        return [NSData dataWithBytes:dsBuffer->data + range.location length:range.length];
    
    subdata = [[NSMutableData alloc] initWithCapacity:range.length];
    while ((length = MIN(range.length, OWDataStreamBuffer_BufferedDataLength - range.location))) {
        [subdata appendBytes:dsBuffer->data + range.location length:length];
        range.location = 0;
        range.length -= length;
        dsBuffer = dsBuffer->next;
    }
    return [subdata autorelease];
}

- (BOOL)waitForMoreData;
{
    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    if (flags.endOfData)
	return NO;
    [dataAvailableCondition waitForCondition];
    return YES;
}

- (BOOL)waitForBufferedDataLength:(unsigned int)length;
{
    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    while (readLength < length)
        if (![self waitForMoreData])
            return NO;
    return YES;
}

- (void)writeData:(NSData *)newData;
{
    NSRange range;
    unsigned int length, lengthLeft;
    OWDataStreamBuffer *new;
    
    length = [newData length];
    if (length == 0)
        return;
    lengthLeft = length;
    range.location = 0;
    range.length = MIN(lengthLeft, OWDataStreamBuffer_BufferedDataLength - dataInBuffer);
    while (lengthLeft) {
        [newData getBytes:last->data + dataInBuffer range:range];
        lengthLeft -= range.length;
        dataInBuffer += range.length;
        if (dataInBuffer == OWDataStreamBuffer_BufferedDataLength) {
            new = NSAllocateMemoryPages(sizeof(OWDataStreamBuffer));
            new->next = NULL;
            last->next = new;
            last = new;
            dataInBuffer = 0;
            range.location += range.length;
            range.length = MIN(lengthLeft, OWDataStreamBuffer_BufferedDataLength - dataInBuffer);
        }
    }
    readLength += length;
    [dataAvailableCondition broadcastCondition];
    if (saveFilename)
        [self flushContentsToFile];
}

- (void)writeString:(NSString *)string;
{
    CFStringEncoding writeEncoding = stringEncoding;
    CFDataRef bytes;
    
    if (string == nil)
	return;
    
    if (writeEncoding == kCFStringEncodingInvalidId)
        writeEncoding = CFStringGetSystemEncoding();  // ??? Maybe this should always be Latin-1? TODO
        
    bytes = CFStringCreateExternalRepresentation(kCFAllocatorDefault, (CFStringRef)string, writeEncoding, 1);
    
    [self writeData:(NSData *)bytes];
    
    CFRelease(bytes);
}

- (void)writeFormat:(NSString *)formatString, ...;
{
    NSString *string;
    va_list argList;

    va_start(argList, formatString);
    string = [[NSString alloc] initWithFormat:formatString arguments:argList];
    va_end(argList);
    [self writeString:string];
    [string release];
}

- (unsigned int)appendToUnderlyingBuffer:(void **)returnedBufferPtr;
{
    OBINVARIANT(dataInBuffer <= OWDataStreamBuffer_BufferedDataLength);
    if (dataInBuffer == OWDataStreamBuffer_BufferedDataLength) {
        OWDataStreamBuffer *new;

        new = NSAllocateMemoryPages(sizeof(OWDataStreamBuffer));
        new->next = NULL;
        last->next = new;
        last = new;
        dataInBuffer = 0;
    }	
    *returnedBufferPtr = last->data + dataInBuffer;
    return OWDataStreamBuffer_BufferedDataLength - dataInBuffer;
}

- (void)wroteBytesToUnderlyingBuffer:(unsigned int)count;    
{
    dataInBuffer += count;
    OBINVARIANT(dataInBuffer <= OWDataStreamBuffer_BufferedDataLength);
    readLength += count;
    [dataAvailableCondition broadcastCondition];
    if (saveFilename)
        [self flushContentsToFile];    
}

- (CFStringEncoding)stringEncoding;
{
//    NSLog(@"Have ct=[%@], returning encoding=%d", [fullContentType contentTypeString], stringEncoding);
    return stringEncoding;
}

- (void)setCFStringEncoding:(CFStringEncoding)aStringEncoding;
{
    NSString *encodingName;
    OWParameterizedContentType *parameterizedContentType;

    stringEncoding = aStringEncoding;
    encodingName = [OWDataStreamCharacterProcessor charsetForCFEncoding:stringEncoding];
    parameterizedContentType = [self fullContentType];
    if (![[parameterizedContentType objectForKey:@"charset"] isEqual:encodingName]) {
        [parameterizedContentType setObject:encodingName forKey:@"charset"];
    }
}

- (OWContentType *)contentEncoding;
{
    return contentEncoding;
}

- (OWContentType *)encodedContentType;
{
    return [super contentType];
}

- (void)setContentEncoding:(OWContentType *)aContentEncoding;
{
    if (aContentEncoding == unencodedContentEncoding)
	aContentEncoding = nil;
    contentEncoding = aContentEncoding;
}

- (void)setContentTypeAndEncodingFromFilename:(NSString *)aFilename isLocalFile:(BOOL)isLocalFile;
{
    OWContentType *type;

    // TODO: Read the file's type/creator code using PBGetCatInfoSync() or the like

    type = [OWContentType contentTypeForFilename:aFilename isLocalFile:isLocalFile];
    if ([type isEncoding]) {
	[self setContentEncoding:type];
	type = [OWContentType contentTypeForFilename:[aFilename stringByDeletingPathExtension] isLocalFile:NO];
    }
    [self setContentType:type];
}

- (NSString *)pathExtensionForContentTypeAndEncoding;
{
    NSString *typeExtension;
    NSString *encodingExtension;
    
    typeExtension = [[self encodedContentType] primaryExtension];
    encodingExtension = [contentEncoding primaryExtension];
    if (encodingExtension == nil)
	return typeExtension;
    else if (typeExtension == nil)
	return encodingExtension;
    else
	return [typeExtension stringByAppendingPathExtension:encodingExtension];
}

- (BOOL)resetContentTypeAndEncoding;
{
    if (flags.hasResetContentTypeAndEncoding)
        return NO;
    flags.hasResetContentTypeAndEncoding = YES;
    [self setContentType:unknownContentType];
    [self setContentEncoding:unencodedContentEncoding];
    return YES;
}

//

- (BOOL)pipeToFilename:(NSString *)aFilename contentCache:(OWContentCache *)cache;
{
    BOOL fileCreated;
    
    if (saveFilename)
	return NO;

    if (flags.hasThrownAwayData)
	[NSException raise:OWDataStreamNoLongerValidException format:@"Data stream no longer contains valid data"];

    fileCreated = [[NSFileManager defaultManager] createFileAtPath:aFilename contents:[NSData data] attributes:nil];
    if (!fileCreated)
        [NSException raise:@"Can't save" format:NSLocalizedStringFromTableInBundle(@"Can't create file at path %@: %s", @"OWF", [self bundle], datastream error), aFilename, strerror(OMNI_ERRNO())];
    if ([[self contentType] hfsType] != 0)
        [[NSFileManager defaultManager] setType:[[self contentType] hfsType] andCreator:[[self contentType] hfsCreator] forPath:aFilename];

    saveFileHandle = [[NSFileHandle fileHandleForWritingAtPath:aFilename] retain];
    if (!saveFileHandle)
	[NSException raise:@"Can't save" format:NSLocalizedStringFromTableInBundle(@"Can't open file %@ for writing: %s", @"OWF", [self bundle], datastream error), aFilename, strerror(OMNI_ERRNO())];
    savedBuffer = first;
    savedInBuffer = 0;

    saveFilename = [aFilename retain];

    // If end of data happened before we set saveFilename, we need to flush out everything ourselves
    if (flags.endOfData)
        [self flushAndCloseSaveFile];

    if (!flags.hasIssuedCursor) {
        [cache flushContentOfType:[self contentType]];
        flags.hasThrownAwayData = YES;
    }

    return YES;
}

- (NSString *)filename;
{
    return saveFilename;
}

- (BOOL)hasThrownAwayData;
{
    return flags.hasThrownAwayData;
}

- (unsigned int)bytesWrittenToFile;
{
    return readLength;
}

// OWStream subclass

- (void)setFullContentType:(OWParameterizedContentType *)aType
{
    CFStringEncoding ctEncoding;
    
    [super setFullContentType:aType];

    ctEncoding = [OWDataStreamCharacterProcessor stringEncodingForContentType:aType];
    if (ctEncoding != kCFStringEncodingInvalidId && ctEncoding != stringEncoding) {
        [self setCFStringEncoding:ctEncoding];
    }
}

- (void)dataEnd;
{
    dataLength = readLength;
    [self _noMoreData];
}

- (void)dataAbort;
{
    flags.hasThrownAwayData = YES;

    [self _noMoreData];

    if (saveFilename) {
        NSString *oldFilename;

        oldFilename = saveFilename;
        saveFilename = nil;
        [[NSFileManager defaultManager] removeFileAtPath:oldFilename handler:nil];
        [oldFilename release];
    }
}

- (void)waitForDataEnd;
{
    if (!flags.endOfData)
        [endOfDataCondition waitForCondition];
}

- (BOOL)endOfData;
{
    return flags.endOfData;
}

// OWContent protocol

- (OWContentType *)contentType;
{
    return contentEncoding != nil ? contentEncoding : [self encodedContentType];
}

- (OWParameterizedContentType *)fullContentType;
{
    if (contentEncoding != nil) {
        OWParameterizedContentType *parameterizedContentEncoding;
        
        parameterizedContentEncoding = [[OWParameterizedContentType alloc] initWithContentType:contentEncoding];
        return [parameterizedContentEncoding autorelease];
    }
    
    return [super fullContentType];
}

- (unsigned long int)cacheSize;
{
    return readLength;
}

- (BOOL)contentIsValid;
{
    return ![self hasThrownAwayData];
}

// OBObject subclass (Debugging)

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    [debugDictionary setObject:flags.endOfData ? @"YES" : @"NO" forKey:@"flags.endOfData"];
    [debugDictionary setObject:flags.hasThrownAwayData ? @"YES" : @"NO" forKey:@"flags.hasThrownAwayData"];
    [debugDictionary setObject:flags.hasIssuedCursor ? @"YES" : @"NO" forKey:@"flags.hasIssuedCursor"];
    [debugDictionary setObject:flags.hasResetContentTypeAndEncoding ? @"YES" : @"NO" forKey:@"flags.hasResetContentTypeAndEncoding"];
    [debugDictionary setObject:[NSNumber numberWithInt:readLength] forKey:@"readLength"];
    return debugDictionary;
}

@end


@implementation OWDataStream (Private)

// This always happens in writer's thread, or after writer is done.
- (void)flushContentsToFile;
{
    const OFByte *bytes;
    unsigned int bytesRemaining;

    do {
        NSData *data;
        
        if (savedBuffer == last)
            bytesRemaining = dataInBuffer - savedInBuffer;
        else
            bytesRemaining = OWDataStreamBuffer_BufferedDataLength - savedInBuffer;
        bytes = savedBuffer->data + savedInBuffer;

        data = [[NSData alloc] initWithBytes:bytes length:bytesRemaining];
        [saveFileHandle writeData:data];
        [data release];

        savedInBuffer = 0;
        savedBuffer = savedBuffer->next;
    } while (savedBuffer);
    savedBuffer = last;
    savedInBuffer = dataInBuffer;

    // throw away anything no longer needed
    if (flags.hasThrownAwayData && !flags.hasIssuedCursor) {
        while (first != savedBuffer) {
            last = first->next;
            NSDeallocateMemoryPages(first, sizeof(OWDataStreamBuffer));
            first = last;
        }
    }
}

- (void)flushAndCloseSaveFile;
{
    [self flushContentsToFile];
    [saveFileHandle release];
    saveFileHandle = nil;
}

- (void)_noMoreData;
{
    if (saveFilename)
        [self flushAndCloseSaveFile];
    
    flags.endOfData = YES;
    [dataAvailableCondition clearCondition];
    [endOfDataCondition clearCondition];
}

@end

NSString *OWDataStreamNoLongerValidException = @"Stream invalid";
