// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OWF/OWDocumentTitle.h>

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OWF/OWContentCache.h>
#import <OWF/OWContentType.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OWF/Content.subproj/OWDocumentTitle.m,v 1.7 2001/02/15 15:11:58 kc Exp $")

@implementation OWDocumentTitle

static OWContentType *documentTitleGuessType = nil;
static OWContentType *documentTitleRealType = nil;
static NSNotificationCenter *notificationCenter;

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
	return;
    initialized = YES;

    documentTitleGuessType = [OWContentType contentTypeForString:@"DocumentTitle/Guess"];
    documentTitleRealType = [OWContentType contentTypeForString:@"DocumentTitle/Real"];
    notificationCenter = [[NSNotificationCenter alloc] init];
}

+ (NSString *)titleForAddress:(id <OWAddress>)address;
{
    OWContentCache *contentCache;
    OWDocumentTitle *documentTitle;

    contentCache = [OWContentCache lookupContentCacheForAddress:address];
    if (!contentCache)
	return nil;
	
    documentTitle = [contentCache peekAtContentOfType:documentTitleRealType];
    if (documentTitle)
	return [documentTitle title];
    documentTitle = [contentCache peekAtContentOfType:documentTitleGuessType];
    if (documentTitle)
	return [documentTitle title];
    return nil;
}

static inline void
cacheTitleOfTypeInCache(NSString *title, OWContentType *contentType, OWContentCache *contentCache)
{
    OWDocumentTitle *documentTitle;

    if (!title || ![title length]) {
	[contentCache flushContentOfType:contentType];
	return;
    }
	
    documentTitle = [contentCache peekAtContentOfType:contentType];
    if ([[documentTitle title] isEqualToString:title])
	return;

    documentTitle = [[OWDocumentTitle alloc] initWithTitle:title contentType:contentType];
    [contentCache addContent:documentTitle];
    [documentTitle release];
}

+ (void)cacheRealTitle:(NSString *)aTitle forAddress:(id <OWAddress>)anAddress;
{
    if (!anAddress)
	return;
    
    cacheTitleOfTypeInCache(aTitle, documentTitleRealType, [OWContentCache contentCacheForAddress:anAddress]);

    // Give the UI a chance to respond
    [notificationCenter postNotificationName:[anAddress cacheKey] object:anAddress];
}

+ (void)cacheGuessTitle:(NSString *)aTitle forAddress:(id <OWAddress>)anAddress;
{
    OWContentCache *contentCache;
    
    if (!anAddress || !aTitle || [aTitle length] == 0)
	return;
    
    contentCache = [OWContentCache contentCacheForAddress:anAddress];
    // Don't cache a guess if we have the real thing
    if ([contentCache peekAtContentOfType:documentTitleRealType])
        return;

    cacheTitleOfTypeInCache(aTitle, documentTitleGuessType, contentCache);
}

+ (void)addObserver:(id)anObserver selector:(SEL)aSelector address:(id <OWAddress>)anAddress;
{
    [notificationCenter addObserver:anObserver selector:aSelector name:[anAddress cacheKey] object:nil];
}

+ (void)removeObserver:(id)anObserver address:(id <OWAddress>)anAddress;
{
    [notificationCenter removeObserver:anObserver name:[anAddress cacheKey] object:nil];
}

+ (void)removeObserver:(id)anObserver;
{
    [notificationCenter removeObserver:anObserver];
}

//

- initWithTitle:(NSString *)aTitle contentType:(OWContentType *)aType;
{
    [super init];
    title = [aTitle retain];
    type = aType;
    return self;
}

- (void)dealloc
{
    [title release];
    [super dealloc];
}

- (NSString *)title;
{
    return title;
}

// OWContent protocol

- (OWContentType *)contentType;
{
    return type;
}

- (unsigned long int)cacheSize;
{
    return sizeof(*self) + sizeof(*title) + [title length];
}

- (BOOL)shareable;
{
    return YES;
}

// OBObject subclass

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    if (title)
        [debugDictionary setObject:title forKey:@"title"];
    if (type)
        [debugDictionary setObject:type forKey:@"type"];
    return debugDictionary;
}

@end
