// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OWF/OWWebPipeline.h>

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OWF/NSDate-OWExtensions.h>
#import <OWF/OWAddress.h>
#import <OWF/OWContentCache.h>
#import <OWF/OWContentInfo.h>
#import <OWF/OWCookieDomain.h>
#import <OWF/OWCookie.h>
#import <OWF/OWHeaderDictionary.h>
#import <OWF/OWURL.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OWF/Pipelines.subproj/OWWebPipeline.m,v 1.28 2001/04/20 20:53:58 kc Exp $")

@interface OWWebPipeline (Private)
- (void)setRefreshEvent:(OFScheduledEvent *) aRefreshEvent;
- (void)postProcessHeaders;
- (void)processRefreshHeader:(NSString *)refresh;
- (void)processExpiresHeader:(NSString *)expires;
@end

@implementation OWWebPipeline

static NSCharacterSet *TokenSet;
static OFScheduler *refreshScheduler;

+ (void)initialize;
{
    OBINITIALIZE;

    refreshScheduler = [[[OFScheduler mainScheduler] subscheduler] retain];
    TokenSet = [[[NSCharacterSet characterSetWithCharactersInString:@";"] invertedSet] retain];
}

- initWithContent:(id <OWContent>)aContent target:(id <OWTarget, OFWeakRetain, NSObject>)aTarget useCachedErrorContent:(BOOL)useError;
{
    if (![super initWithContent:aContent target:aTarget useCachedErrorContent:useError])
        return nil;

    webPipelineFlags.proxyCacheDisabled = NO;
    historyAction = OWWebPipelineForwardHistoryAction;
    if (parentContentInfo)
        [self setReferringContentInfo:parentContentInfo];
    
    return self;
}

- (void)dealloc;
{
    [referringContentInfo release];
    [referringAddress release];
    [refreshEvent release];
    [super dealloc];
}

- (id <OWAddress>)referringAddress;
{
    return referringAddress;
}

- (void)setReferringAddress:(id <OWAddress>)anAddress;
{
    id <OWAddress> oldReferringAddress;

    if (referringAddress == anAddress)
        return;

    oldReferringAddress = referringAddress;
    referringAddress = [anAddress retain];
    [oldReferringAddress release];
}

- (OWContentInfo *)referringContentInfo
{
    return referringContentInfo;
}

- (void)setReferringContentInfo:(OWContentInfo *)anInfo
{
    if (anInfo != referringContentInfo) {
        [anInfo retain];
        [referringContentInfo release];
        referringContentInfo = anInfo;
    }

    [self setReferringAddress:[referringContentInfo address]];
}

- (OWWebPipelineHistoryAction)historyAction;
{
    return historyAction;
}

- (void)setHistoryAction:(OWWebPipelineHistoryAction)newHistoryAction;
{
    historyAction = newHistoryAction;
}

- (BOOL)proxyCacheDisabled;
{
    return webPipelineFlags.proxyCacheDisabled || (lastAddress && [lastAddress isAlwaysUnique]);
}

- (void)setProxyCacheDisabled:(BOOL)newDisabled;
{
    webPipelineFlags.proxyCacheDisabled = newDisabled;
}

// OWPipeline subclass

- (void)startProcessingContent;
{
    if (state == PipelineInit && webPipelineFlags.proxyCacheDisabled)
        [[self contentCacheForLastAddress] flushCachedContent];
    [super startProcessingContent];
}

- (void)refetchContentFromLastAddress;
{
    historyAction = OWWebPipelineReloadHistoryAction;
    [self setProxyCacheDisabled:YES];
    [super refetchContentFromLastAddress];
}

- (void)invalidate;
{
    [refreshScheduler abortEvent:refreshEvent];
    [refreshEvent release];
    refreshEvent = nil;
    [super invalidate];
}

- (void)addHeader:(NSString *)headerName value:(NSString *)headerValue;
{
    [super addHeader:headerName value:headerValue];
    
    if ([headerName compare:OWSetCookieHeader options:NSCaseInsensitiveSearch] == NSOrderedSame) {
        [OWCookieDomain registerCookiesFromPipeline:self headerValue:headerValue];
    }
}

// OWPipeline subclass (SubclassesOnly)

- (void)deactivate;
{
    [self postProcessHeaders];
    [super deactivate];
}

// NSCopying protocol

- copyWithZone:(NSZone *)zone;
{
    OWWebPipeline *newPipeline;

    newPipeline = [super copyWithZone:zone];
    [newPipeline setReferringContentInfo:referringContentInfo];
    [newPipeline setReferringAddress:referringAddress];
    [newPipeline setHistoryAction:OWWebPipelineReloadHistoryAction];
    return newPipeline;
}

@end

@implementation OWWebPipeline (Private)

- (void)setRefreshEvent:(OFScheduledEvent *) aRefreshEvent;
{
    OBPRECONDITION(!refreshEvent);
    refreshEvent = [aRefreshEvent retain];
}

- (void)postProcessHeaders;
{
    NSString *headerValue;

    if ((headerValue = [headerDictionary lastStringForKey:@"refresh"]))
	[self processRefreshHeader:headerValue];
    if ((headerValue = [headerDictionary lastStringForKey:@"expires"]))
	[self processExpiresHeader:headerValue];
        
    // DON'T register cookies here.  They've already been registered in OWHTTPSession.
    // We have to register the cookies at the beginning of the document so that they
    // are around for pipelines that might get spawned off due to the documents content
    
#warning Should process other headers, e.g. Cache-Control
}

- (void)processRefreshHeader:(NSString *)refresh;
{
    NSString *refreshTimeString, *urlString;
    NSTimeInterval refreshTimeInterval;
    NSCalendarDate *refreshDate;
    OWURL *refreshURL, *referringURL;
    OWAddress *refreshAddress;
    OWWebPipeline *refreshPipeline;

    if ([refresh containsString:@";"]) {
        NSScanner *scanner;

        refreshTimeString = nil;
        urlString = nil;
        scanner = [[NSScanner alloc] initWithString:refresh];
        [scanner scanCharactersFromSet:TokenSet intoString:&refreshTimeString];
        while ([scanner scanString:@";" intoString:NULL])
            if ([scanner scanString:@"url=" intoString:NULL]) {
                [scanner scanCharactersFromSet:TokenSet intoString:&urlString];
                urlString = [OWURL cleanURLString:urlString];
            }
        [scanner release];
    } else {
        refreshTimeString = refresh;
        urlString = nil;
    }
    if (!refreshTimeString)
        return;
    referringURL = [(OWAddress *)lastAddress url];
    refreshURL = referringURL;
    if (urlString) {
        if (refreshURL)
            refreshURL = [refreshURL urlFromRelativeString:urlString];
        else
            refreshURL = [OWURL urlFromString:urlString];
    }
    refreshAddress = [OWAddress addressWithURL:refreshURL];
    if (![refreshAddress isSameDocumentAsAddress:(OWAddress *)lastAddress]) {
        // If we've been asked to redirect to another page, we need to make sure we redirect on schedule the next time we load this page.
        // TODO: Rather than flushing our content from the cache, it would be much better to cache the HTTP headers so that the next pipeline gets the cached headers and content rather than having to start again from scratch.
        [[self contentCacheForLastAddress] flushCachedContent];
    }
    refreshTimeInterval = [refreshTimeString floatValue];
    refreshPipeline = [[[self class] alloc] initWithContent:refreshAddress target:[self target]];
    [refreshPipeline setProxyCacheDisabled:YES];
    refreshDate = [[NSCalendarDate alloc] initWithTimeIntervalSinceNow:refreshTimeInterval];
    [refreshDate setCalendarFormat:NSLocalizedStringFromTableInBundle(@"%b %d %H:%M:%S", @"OWF", [self bundle], webpipeline timed refresh NSCalendarDate format)];
    [refreshPipeline setContextObject:[NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Timed Refresh at %@", @"OWF", [self bundle], webpipeline timed refresh message), refreshDate] forKey:@"Status"];
    if (referringURL)
        [refreshPipeline setReferringAddress:[OWAddress addressWithURL:referringURL]];
    if (refreshTimeInterval <= 1.0) {
        [refreshPipeline startProcessingContent];
    } else {
        OFScheduledEvent *event;
        
        event = [refreshScheduler scheduleSelector:@selector(startProcessingContent) onObject:refreshPipeline withObject:nil atDate:refreshDate];
        [refreshPipeline setRefreshEvent:event];
    }
    [refreshDate release];
    [refreshPipeline release];
}

- (void)processExpiresHeader:(NSString *)expires;
{
    NSDate *expireDate;

    expireDate = [NSDate dateWithHTTPDateString:expires];
    if (!expireDate)
	expireDate = [NSDate dateWithTimeIntervalSinceNow:0];
    [contentCacheForLastAddress expireAtDate:expireDate];
}

@end
