// Copyright 2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWAuthorization-KeychainFunctions.h"

#import <Foundation/Foundation.h>
#import <CoreServices/CoreServices.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OWF/Processors.subproj/Protocols.subproj/OWAuthorization-KeychainFunctions.m,v 1.3.6.1 2001/08/22 03:56:31 rick Exp $");

/* This'll be big enough for practically everything. On the off chance it's too small, we try again. */
/* NOTE: KCGetAttribute() will do invisible silent *format conversions* on some attributes of you say you have an 8-byte buffer! What was Apple smoking? */
#define INITIAL_TRY_LENGTH 9  /* change to 128 or so after debug */

NSData *OWKCGetItemAttribute(KCItemRef item, KCItemAttr attrTag)
{
    SecKeychainAttribute attr;
    OSStatus ok;
    UInt32 actualLength;
    void *freeMe = NULL;
    
    attr.tag = attrTag;
    actualLength = INITIAL_TRY_LENGTH;
    attr.length = actualLength;  /* KCGetAttribute() doesn't appear to write this field, at least in Cheetah4K29, but it may read it */
    attr.data = alloca(actualLength);
        
    ok = KCGetAttribute(item, &attr, &actualLength);
    if (ok == errKCBufferTooSmall) {
        /* the attribute length will have been placed into actualLength */
        freeMe = NSZoneMalloc(NULL, actualLength);
        attr.length = actualLength;
        attr.data = freeMe;
        ok = KCGetAttribute(item, &attr, &actualLength);
    }
    if (ok == noErr) {
        NSData *retval = [NSData dataWithBytes:attr.data length:actualLength];
	if (freeMe != NULL)
            NSZoneFree(NULL, freeMe);
        // NSLog(@"attr '%c%c%c%c' value %@", ((char *)&attrTag)[0], ((char *)&attrTag)[1], ((char *)&attrTag)[2], ((char *)&attrTag)[3], retval);
        return retval;
    }
    
    if (freeMe != NULL)
        NSZoneFree(NULL, freeMe);

    if (ok == errKCNoSuchAttr) {
        /* An expected error. Return nil for nonexistent attributes. */
        return nil;
    }
    
    /* An unexpected error, probably indicating a real problem. Raise an exception. */
    [NSException raise:@"Keychain error" format:@"Error number %d occurred while trying to fetch an item attribute, and Apple's too stingy to include a strerror() equivalent.", ok];
    
    return nil;  // appease the dread compiler warning gods
}

#if 0

#warning using private API
// KCGetData() is in Carbon, which we don't want to link against in OWF. It has a non-UI counterpart in CoreServices called KCGetDataNoUI(), but it's not declared.
extern OSStatus 
KCGetDataNoUI(
  KCItemRef   item,
  UInt32      maxLength,
  void *      data,
  UInt32 *    actualLength);

#else

// KCGetDataNoUI() disappeared in a recent (post-10.0.4) build at Apple, so we replace it with a stub function which always fails. OmniWeb overrides this function, so this won't affect OmniWeb. But it does mean that other users of OWF will be unable to perform authorization (unless they also override the function)

static OSStatus getDataNoUIStub(
  KCItemRef   item,
  UInt32      maxLength,
  void *      data,
  UInt32 *    actualLength)
{
    if (actualLength)
        *actualLength = 0;
    
    NSLog(@"warning: KCGetDataNoUI() no longer available; noninteractive apps can't use keychain");
    
    return errKCInteractionNotAllowed;
}

#define KCGetDataNoUI getDataNoUIStub

#endif


OSStatus OWKCExtractKeyData(KCItemRef item, NSData **password, void *funcOverride)
{
    OSStatus ok;
    UInt32 actualLength;
    char shortBuf[INITIAL_TRY_LENGTH];
    void *freeMe, *buffer;
    OSStatus (*getDataFunc)(KCItemRef, UInt32, void *, UInt32 *);
    
    // This bit of silliness is to allow OWF to remain innocent of all GUI functionality (such as the unlock / password dialogues popped up by KCGetData()) but to allow higher-level frameworks to plug in a more interactive routine.
    if (funcOverride)
        getDataFunc = funcOverride;
    else
        getDataFunc = KCGetDataNoUI;

    freeMe = NULL;
    actualLength = INITIAL_TRY_LENGTH;
    buffer = shortBuf;
    ok = getDataFunc(item, actualLength, buffer, &actualLength);
    if (ok == errKCBufferTooSmall) {
        freeMe = NSZoneMalloc(NULL, actualLength);
        buffer = freeMe;
        ok = getDataFunc(item, actualLength, buffer, &actualLength);
    }
    if (ok == noErr) {
        *password = [NSData dataWithBytes:buffer length:actualLength];
    }
    if (freeMe)
        NSZoneFree(NULL, freeMe);
    return ok;
}

static inline NSString *parseKeychainString(NSData *strbytes) {
#warning encoding breakage
    // Once Keychain Access supports multiple encodings, it will probably stash a ScriptCode into the keychain item. We'll be able to use that ScriptCode to convert item attributes correctly.
    // Trim trailing nulls here, since some strings have 'em and some don't. This could break multibyte encodings, so when we fix those (see above) we'll have to change this as well.
    if ([strbytes length] && (((char *)[strbytes bytes])[[strbytes length]-1] == 0))
        strbytes = [strbytes subdataWithRange:NSMakeRange(0, [strbytes length]-1)];
    return [[[NSString alloc] initWithData:strbytes encoding:[NSString defaultCStringEncoding]] autorelease];
}

#if 0
static NSDate *parseKeychainDate(NSData *date)
{
    NSString *text;
    NSDate *retval;
    
    /* The documentation states that keychain creation and mod-time dates are UInt32, but in fact they appear to be ASCII strings in a packed calendar format. */
    if ([date length] <= 8) {
        /* Are these Unix-style seconds-since-1970, or Mac-style seconds-since-1900 ? Nobody knows. */
        [NSException raise:@"Keychain error" format:@"Unexpected timestamp format in keychain item."];
    }
    
    text = parseKeychainString(date);
    retval = [NSCalendarDate dateWithString:text calendarFormat:@"%Y%m%d%H%M%SZ"];
    if (!retval) {
        NSLog(@"Couldn't convert date: %@", text);
        return [NSDate distantPast];
    }
    return retval;
}
#endif

static NSNumber *parseKeychainInteger(NSData *value) {
    // Endianness? Portability? Bah! We don't need no portability! Everything's a Macintosh now!
    UInt32 int4 = 0;
    UInt16 int2 = 0;
    UInt8 int1 = 0;
    
    switch([value length]) {
        case 4:
            [value getBytes:&int4];
            break;
        case 2:
            [value getBytes:&int2];
            int4 = int2;
            break;
        case 1:
            [value getBytes:&int1];
            int4 = int1;
            break;
        default:
            [NSException raise:@"Keychain error" format:@"Unexpected integer format in keychain item."];
    }
    
    return [NSNumber numberWithUnsignedInt:int4];
}

NSMutableDictionary *OWKCExtractKeyAttributes(KCItemRef itemRef)
{
    NSMutableDictionary *dict = [[[NSMutableDictionary alloc] initWithCapacity:13] autorelease];
    NSData *val;
    
    val = OWKCGetItemAttribute(itemRef, kClassKCItemAttr);
    if (val) [dict setObject:val forKey:@"class"];

    val = OWKCGetItemAttribute(itemRef, kCreationDateKCItemAttr);
    if (val) [dict setObject:val forKey:@"creationDate"];
    
    val = OWKCGetItemAttribute(itemRef, kModDateKCItemAttr);
    if (val) [dict setObject:val forKey:@"modDate"];
    
    val = OWKCGetItemAttribute(itemRef, kDescriptionKCItemAttr);
    if (val && [val length]) [dict setObject:parseKeychainString(val) forKey:@"description"];
    
    val = OWKCGetItemAttribute(itemRef, kCommentKCItemAttr);
    if (val && [val length]) [dict setObject:parseKeychainString(val) forKey:@"comment"];
    
    val = OWKCGetItemAttribute(itemRef, kLabelKCItemAttr);
    if (val && [val length]) [dict setObject:parseKeychainString(val) forKey:@"label"];
    
    val = OWKCGetItemAttribute(itemRef, kAccountKCItemAttr);
    if (val) [dict setObject:parseKeychainString(val) forKey:@"account"];

    val = OWKCGetItemAttribute(itemRef, kServiceKCItemAttr);
    if (val) [dict setObject:parseKeychainString(val) forKey:@"service"];

    val = OWKCGetItemAttribute(itemRef, kSecurityDomainKCItemAttr);
    if (val && [val length]) [dict setObject:parseKeychainString(val) forKey:@"securityDomain"];

    val = OWKCGetItemAttribute(itemRef, kServerKCItemAttr);
    if (val) [dict setObject:parseKeychainString(val) forKey:@"server"];

    val = OWKCGetItemAttribute(itemRef, kAuthTypeKCItemAttr);
    if (val) [dict setObject:val forKey:@"authType"];

    val = OWKCGetItemAttribute(itemRef, kProtocolKCItemAttr);
    if (val) [dict setObject:val forKey:@"protocol"];

    val = OWKCGetItemAttribute(itemRef, kPortKCItemAttr);
    if (val && [val length]) [dict setObject:parseKeychainInteger(val) forKey:@"port"];

    val = OWKCGetItemAttribute(itemRef, kPathKCItemAttr);
    if (val && [val length]) [dict setObject:parseKeychainString(val) forKey:@"path"];

    return dict;
}

static NSData *formatKeychain4CC(id value)
{
    if ([value isKindOfClass:[NSData class]])
        return value;
    if ([value isKindOfClass:[NSNumber class]]) {
        UInt32 aCode = [value unsignedIntValue];
        return [NSData dataWithBytes:&aCode length:sizeof(aCode)];
    }
    
    return [value dataUsingEncoding:[NSString defaultCStringEncoding]];
}

static NSData *formatKeychainString(id value)
{
    return [value dataUsingEncoding:[NSString defaultCStringEncoding]];
}

static NSData *formatKeychainInteger(id value)
{
    UInt32 int4;
    
    int4 = [value intValue];
    return [NSData dataWithBytes:&int4 length:sizeof(int4)];
}

OSStatus OWKCBeginKeychainSearch(KCRef chain, NSDictionary *params, KCSearchRef *grepstate, KCItemRef *firstitem)
{
    SecKeychainAttribute *attributes;
    SecKeychainAttributeList attrList;
    unsigned attrCount, attrIndex;
    NSEnumerator *paramNameEnumerator;
    NSString *paramName;
    OSStatus ok;
    
    if (!params) {
	return KCFindFirstItem(chain, NULL, grepstate, firstitem);
    }
    
    attrCount = [params count];
    attributes = alloca(sizeof(*attributes) * attrCount);
    attrIndex = 0;
    
    paramNameEnumerator = [params keyEnumerator];
    while( (paramName = [paramNameEnumerator nextObject]) != nil) {
        id paramValue = [params objectForKey:paramName];
        NSData *data;
        
        if ([paramName isEqualToString:@"class"]) {
            attributes[attrIndex].tag = kClassKCItemAttr;
            data = formatKeychain4CC(paramValue);
        } else if ([paramName isEqualToString:@"description"]) {
            attributes[attrIndex].tag = kDescriptionKCItemAttr;
            data = formatKeychainString(paramValue);
        } else if ([paramName isEqualToString:@"account"]) {
            attributes[attrIndex].tag = kAccountKCItemAttr;
            data = formatKeychainString(paramValue);
        } else if ([paramName isEqualToString:@"service"]) {
            attributes[attrIndex].tag = kServiceKCItemAttr;
            data = formatKeychainString(paramValue);
        } else if ([paramName isEqualToString:@"securityDomain"]) {
            attributes[attrIndex].tag = kSecurityDomainKCItemAttr;
            data = formatKeychainString(paramValue);
        } else if ([paramName isEqualToString:@"server"]) {
            attributes[attrIndex].tag = kServerKCItemAttr;
            data = formatKeychainString(paramValue);
        } else if ([paramName isEqualToString:@"authType"]) {
            attributes[attrIndex].tag = kAuthTypeKCItemAttr;
            data = formatKeychain4CC(paramValue);
        } else if ([paramName isEqualToString:@"port"]) {
            attributes[attrIndex].tag = kPortKCItemAttr;
            data = formatKeychainInteger(paramValue);
        } else if ([paramName isEqualToString:@"path"]) {
            attributes[attrIndex].tag = kPathKCItemAttr;
            data = formatKeychainString(paramValue);
        } else if ([paramName isEqualToString:@"protocol"]) {
            attributes[attrIndex].tag = kProtocolKCItemAttr;
            data = formatKeychain4CC(paramValue);
        } else {
            // this shouldn't happen.
            continue;
        }

        attributes[attrIndex].length = [data length];
        attributes[attrIndex].data = (char *)[data bytes];
        attrIndex ++;
    }
    
    attrList.count = attrIndex;
    attrList.attr = attributes;
    
    ok = KCFindFirstItem(chain, &attrList, grepstate, firstitem);
    return ok; 
}

