// Copyright 2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWAuthSchemeHTTPDigest.h"

#import <OWF/OWHTTPProcessor.h>
#import <OWF/OWAddress.h>
#import <OWF/OWURL.h>
#import <OWF/OWPipeline.h>

#import <OmniFoundation/OmniFoundation.h>
#import <Foundation/Foundation.h>
#import <OmniBase/rcsid.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/Processors.subproj/Protocols.subproj/HTTP.subproj/OWAuthSchemeHTTPDigest.m,v 1.3 2001/10/08 23:04:08 wiml Exp $");

@implementation OWAuthSchemeHTTPDigest

// Init and dealloc

- init;
{
    if (![super init])
        return nil;

    return self;
}

- initAsCopyOf:otherInstance
{
    
    if (!(self = [super initAsCopyOf:otherInstance]))
        return nil;
        
    if ([otherInstance isKindOfClass:[OWAuthorizationPassword class]]) {
        OWAuthSchemeHTTPDigest *other = otherInstance;
        nonce = [other->nonce copy];
        opaque = [other->opaque copy];
    } else {
        nonce = nil;
        opaque = nil;
    }

    return self;
}

- (void)dealloc;
{
    [nonce release];
    [opaque release];
    [super dealloc];
}


- (void)setNonce:(NSString *)newNonce opaque:(NSString *)newOpaque
{
    [newNonce retain];
    [nonce release];
    nonce = newNonce;
    
    [newOpaque retain];
    [opaque release];
    opaque = newOpaque;
}

- (int)compareToNewCredential:(OWAuthorizationCredential *)other
{
    int compare = [super compareToNewCredential:other];
    // NB: super will also check that 'other' is of the same class we are, which ensures that the casts (below) are valid
    
    if (compare == OWCredentialIsEquivalent) {
        NSString *otherOpaque;
        if (![nonce isEqual:(((OWAuthSchemeHTTPDigest *)other)->nonce)])
            compare = OWCredentialWouldReplace;
        otherOpaque = (((OWAuthSchemeHTTPDigest *)other)->opaque);
        if ((opaque == nil && otherOpaque != nil) ||
            (opaque != nil && otherOpaque == nil) ||
            (opaque != otherOpaque && ![opaque isEqual:otherOpaque]))
            compare = OWCredentialWouldReplace;
    }
    
    return compare;
}

static NSString *computeDigest(NSString *username,
                               NSString *password,
                               NSString *nonce,
                               NSString *realmname,
                               NSString *method,
                               NSString *fetchPath)
{
    NSMutableString *Ax;
    NSString *A1hash, *A2hash;
    NSString *response;
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];

    // buffer for string manipulation
    Ax = [[[NSMutableString alloc] init] autorelease];

    // compute A1 and its MD5-hash
    [Ax appendStrings:username, @":", realmname, @":", password, nil];
    A1hash = [[[Ax dataUsingEncoding:NSISOLatin1StringEncoding] md5Signature] unadornedLowercaseHexString];
    [Ax deleteCharactersInRange:NSMakeRange(0, [Ax length])];
    
    // compute A2 and its MD5-hash
    [Ax appendStrings:method, @":", fetchPath, nil];
    A2hash = [[[Ax dataUsingEncoding:NSISOLatin1StringEncoding] md5Signature] unadornedLowercaseHexString];
    [Ax deleteCharactersInRange:NSMakeRange(0, [Ax length])];

    // compute the final digest
    [Ax appendStrings:A1hash, @":", nonce, @":", A2hash, nil];
    response = [[[Ax dataUsingEncoding:NSISOLatin1StringEncoding] md5Signature] unadornedLowercaseHexString];
    
    [response retain];
    [pool release];
    
    return [response autorelease];
}

// API

- (NSString *)httpHeaderStringForProcessor:(OWProcessor *)aProcessor
{
    NSString *response;
    OWAddress *uri;
    NSString *fetchPath, *fetchMethod;
    NSString *headerName;

    // TODO: How should we handle proxies who use Digest authentication?
    if (aProcessor == nil)
        return nil;

    uri = [[aProcessor pipeline] lastAddress];
    OBASSERT(uri != nil);
    fetchPath = [[uri url] fetchPath];
    fetchMethod = [uri methodString];
    OBASSERT(fetchPath != nil);
    OBASSERT(fetchMethod != nil);

    if (type == OWAuth_HTTP)
        headerName = @"Authorization";
    else if (type == OWAuth_HTTP_Proxy)
        headerName = @"Proxy-Authorization";
    else
        headerName = @"X-Bogus-Header"; // TODO        
    
    // TODO: quote correctly
    response = [NSString stringWithFormat:@"%@: Digest username=\"%@\" realm=\"%@\" nonce=\"%@\" uri=\"%@\" response=\"%@\" algorithm=\"MD5\"%@",
        headerName,
        username,
        realm,
        nonce,
        fetchPath,
        computeDigest(username, password, nonce, realm, fetchMethod, fetchPath),
        opaque ? [NSString stringWithStrings:@" opaque=\"", opaque, @"\"", nil] : @""];

    /* TODO: entity-digest headers */
    /* opaque is required for conformance; entity-digest is a transport integrity check kinda thing */

    return response;
}

- (BOOL)appliesToHTTPChallenge:(NSDictionary *)challenge
{
    // Correct scheme?
    if ([[challenge objectForKey:@"scheme"] caseInsensitiveCompare:@"digest"] != NSOrderedSame)
        return NO;
    
    // Correct realm?
    if (realm && [realm caseInsensitiveCompare:[challenge objectForKey:@"realm"]] != NSOrderedSame)
        return NO;
    
    // It's OK to use an old nonce. The server will tell us if the nonce is too old and give us a chance to try again.
    
    return YES;
}

- (void)authorizationSucceeded:(BOOL)success response:(OWHeaderDictionary *)response;
{
    /* If we've failed, but we succeeded some time in the past, check to see if we can just try again with a new nonce instead of prompting the user again. */
    if (!success && lastSucceededTimeInterval > OWAuthDistantPast) {
        NSArray *challenges = [OWAuthorizationRequest findParametersOfType:type headers:response];
        int challengeIndex;
        
        for(challengeIndex = 0; challengeIndex < [challenges count]; challengeIndex ++) {
            NSDictionary *challenge = [challenges objectAtIndex:challengeIndex];
            if ([self appliesToHTTPChallenge:challenge]) {
                NSString *stale = [challenge objectForKey:@"stale"];
                NSString *newNonce = [challenge objectForKey:@"nonce"];
                if (stale && newNonce && ![nonce isEqual:newNonce] &&
                    [stale caseInsensitiveCompare:@"true"] == NSOrderedSame) {
                    OWAuthSchemeHTTPDigest *newCred;
                    
                    newCred = [[OWAuthSchemeHTTPDigest alloc] initAsCopyOf:self];
                    [newCred setNonce:newNonce opaque:[challenge objectForKey:@"opaque"]];
                    [OWAuthorizationRequest cacheCredentialIfAbsent:newCred];
                    [newCred release];
                }
            }
        }
    }
    
    [super authorizationSucceeded:success response:response];
}

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary = [super debugDictionary];

    [debugDictionary setObject:nonce forKey:@"nonce"];
    if (opaque)
        [debugDictionary setObject:opaque forKey:@"opaque"];

    return debugDictionary;
}


@end

