// Copyright 2000-2002 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWSimpleTarget.h"

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import "OWContentInfo.h"
#import "OWContentContainer.h"
#import "OWContentType.h"
#import "OWDataStream.h"
#import "OWDataStreamCursor.h"
#import "OWHeaderDictionary.h"
#import "OWPipeline.h"
#import "OWProcessor.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OWF/OWSimpleTarget.m,v 1.11 2002/03/09 01:53:48 kc Exp $")

@interface OWSimpleTarget (Private)
- (void)_setResultingContent:(id <OWContent>)someContent;
@end

@implementation OWSimpleTarget

enum { OWSimpleTargetNoResult, OWSimpleTargetHaveResult };

// Init and dealloc

- initWithParentContentInfo:(OWContentInfo *)contentInfo targetContentType:(OWContentType *)contentType
{
    if (![super init])
        return nil;

    OFWeakRetainConcreteImplementation_INIT;

    parentContentInfo = [contentInfo retain];
    targetContentType = [contentType retain];
    resultLock = [[NSConditionLock alloc] initWithCondition:OWSimpleTargetNoResult];
    acceptsAlternateContent = NO;
    targetTypeFormatString = nil;

    return self;
}

- (void)dealloc;
{
    OFWeakRetainConcreteImplementation_DEALLOC;

    [parentContentInfo release];
    [targetContentType release];
    [resultLock release];
    [resultingContent release];
    [targetTypeFormatString release];
    
    [super dealloc];
}

// API

- (void)setAcceptsAlternateContent:(BOOL)shouldAcceptAlternateContent;
{
    acceptsAlternateContent = shouldAcceptAlternateContent;
}

- (id <OWContent>)resultingContent;
{
    id <OWContent> someContent;

    [resultLock lockWhenCondition:OWSimpleTargetHaveResult];
    someContent = [resultingContent retain];
    [resultLock unlock];
    return [someContent autorelease];
}

// OFWeakRetain protocol

OFWeakRetainConcreteImplementation_IMPLEMENTATION

- (void)invalidateWeakRetains;
{
    [OWPipeline invalidatePipelinesForTarget:self];
}

// OWTarget protocol

- (OWContentType *)targetContentType;
{
    return targetContentType;
}

- (OWTargetContentDisposition)pipeline:(OWPipeline *)aPipeline hasContent:(id <OWContent>)someContent;
{
    [self _setResultingContent:someContent];
    return OWTargetContentDisposition_ContentAccepted;
}

- (OWTargetContentDisposition)pipeline:(OWPipeline *)aPipeline hasAlternateContent:(id <OWContent>)someContent;
{
    if (acceptsAlternateContent) {
        [self _setResultingContent:someContent];
        return OWTargetContentDisposition_ContentAccepted;
    } else {
        [self _setResultingContent:nil];
        return OWTargetContentDisposition_ContentRejectedCancelPipeline;
    }
}

- (OWTargetContentDisposition)pipeline:(OWPipeline *)aPipeline hasErrorContent:(id <OWContent>)someContent;
{
    [self _setResultingContent:nil];
    return OWTargetContentDisposition_ContentRejectedCancelPipeline;
}

- (void)pipelineDidEnd:(OWPipeline *)aPipeline;
{
    [self _setResultingContent:resultingContent];
}

- (void)setTargetTypeFormatString:(NSString *)newFormatString
{
    if (targetTypeFormatString != nil)
        [targetTypeFormatString autorelease];
    targetTypeFormatString = [newFormatString retain];
}

- (NSString *)targetTypeFormatString;
{
    if (targetTypeFormatString)
        return targetTypeFormatString;
    else	
        return NSLocalizedStringFromTableInBundle(@"%@ File", @"OWF", [OWSimpleTarget bundle], simpleTarget targetTypeFormatString - generic description when no type string is supplied);
}

- (OWContentInfo *)parentContentInfo;
{
    return parentContentInfo;
}

@end

@implementation OWSimpleTarget (Private)

- (void)_setResultingContent:(id <OWContent>)someContent;
{
    [resultLock lock];
    if (resultingContent != someContent) {
        [resultingContent release];
        resultingContent = [someContent retain];
    }
    [resultLock unlockWithCondition:OWSimpleTargetHaveResult];
}

@end
