// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAFontCache.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/OAFontCache.m,v 1.10 2000/01/19 23:35:42 kc Exp $")

typedef struct _OAFontCacheKey {
    NSString *familyName;
    OAFontAttributes attributes;
} OAFontCacheKey;

static unsigned OAFontCacheKeyHash(NSMapTable *table, const void *key)
{
    const OAFontCacheKey *fontCacheKey = (const OAFontCacheKey *)key;

    return [fontCacheKey->familyName hash] +
           ((unsigned int)fontCacheKey->attributes.size << 2) +
           (fontCacheKey->attributes.bold << 1) +
           fontCacheKey->attributes.italic;
}

static BOOL OAFontCacheKeyIsEqual(NSMapTable *table, const void *key1, const void *key2)
{
    const OAFontCacheKey *fontCacheKey1 = (const OAFontCacheKey *)key1;
    const OAFontCacheKey *fontCacheKey2 = (const OAFontCacheKey *)key2;

    if (![fontCacheKey1->familyName isEqualToString:fontCacheKey2->familyName])
        return NO;
    return fontCacheKey1->attributes.size == fontCacheKey2->attributes.size &&
           fontCacheKey1->attributes.bold == fontCacheKey2->attributes.bold &&
           fontCacheKey1->attributes.italic == fontCacheKey2->attributes.italic;
}

static NSMapTable *fontMapTable;
static NSLock *fontMapTableLock;
static NSFontManager *fontManager;

@implementation OAFontCache

+ (void)initialize;
{
    static BOOL initialized = NO;
    NSMapTableKeyCallBacks keyCallBacks;
    
    [super initialize];
    if (initialized)
        return;
    initialized = YES;

    keyCallBacks.hash = OAFontCacheKeyHash;
    keyCallBacks.isEqual = OAFontCacheKeyIsEqual;
    keyCallBacks.retain = NULL;
    keyCallBacks.release = NULL;
    keyCallBacks.describe = NULL;
    fontMapTable = NSCreateMapTable(keyCallBacks, NSObjectMapValueCallBacks, 32);
    fontMapTableLock = [[NSLock alloc] init];
    
    fontManager = [[NSFontManager sharedFontManager] retain];
}

+ (NSFont *)fontWithFamily:(NSString *)aFamily attributes:(OAFontAttributes)someAttributes;
{
    NSFont *font;
    OAFontCacheKey pseudo, *key;

    pseudo.familyName = aFamily;
    pseudo.attributes = someAttributes;

    [fontMapTableLock lock];
    font = NSMapGet(fontMapTable, &pseudo);
    [fontMapTableLock unlock];
    if (!font) {
        //
        // Perform the thread-unsafe NSFontManager operation
        //

        [NSThread lockMainThread]; // Remove this once NSFont is thread-safe

        NS_DURING {
            if (!(font = [NSFont fontWithName:aFamily size:someAttributes.size]))
                font = [NSFont userFontOfSize:someAttributes.size];

            // Only attempt to convert fonts if we actually have a font
            if (font && someAttributes.bold)
                font = [fontManager convertFont:font toHaveTrait:NSBoldFontMask];
            if (font && someAttributes.italic)
                font = [fontManager convertFont:font toHaveTrait:NSItalicFontMask];
        } NS_HANDLER {
            NSLog(@"Warning: Exception calculating font with family %@: %@", aFamily, [localException reason]);
        } NS_ENDHANDLER;
        
        [NSThread unlockMainThread];

        if (!font)
            font = [OFNull nullObject];

        [font retain];

        //
        // Done being thread-unsafe.  Store the results to avoid future lookups
        //

        // Create a new entry
        key = NSZoneMalloc(NULL, sizeof(OAFontCacheKey));
        key->familyName = [aFamily copy];
        key->attributes = someAttributes;

        [fontMapTableLock lock];
        NSMapInsert(fontMapTable, key, font);
        [fontMapTableLock unlock];
    }
                
    return [font isNull] ? nil : font;
}

+ (NSFont *)fontWithFamily:(NSString *)aFamily size:(float)size bold:(BOOL)bold italic:(BOOL)italic;
{
    OAFontAttributes attributes;

    attributes.size = size;
    attributes.bold = bold;
    attributes.italic = italic;
    return [self fontWithFamily:aFamily attributes:attributes];
}

+ (NSFont *)fontWithFamily:(NSString *)aFamily size:(float)size;
{
    return [self fontWithFamily:aFamily size:size bold:NO italic:NO];
}

@end
