// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/NSView-OAExtensions.h>

#import <CoreGraphics/CGContext.h>
#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OmniAppKit/NSFont-OAExtensions.h>
#import <OmniAppKit/NSApplication-OAExtensions.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/OpenStepExtensions.subproj/NSView-OAExtensions.m,v 1.20 2000/07/08 03:25:32 krevis Exp $")


#define TRANSITION_TIME (0.25)


@implementation NSView (OAExtensions)

// Drawing

- (void)drawRoundedRect:(NSRect)rect cornerRadius:(float)radius
   color:(NSColor *)color;
{
    CGContextRef context;

    context = [[NSGraphicsContext currentContext] graphicsPort];
    [color set];

    CGContextBeginPath(context);
    CGContextMoveToPoint(context, NSMinX(rect), NSMinY(rect) + radius);
    CGContextAddLineToPoint(context, NSMinX(rect), NSMaxY(rect) - radius);
    CGContextAddArcToPoint(context, NSMinX(rect), NSMaxY(rect), NSMinX(rect) + radius, NSMaxY(rect), radius);
    CGContextAddLineToPoint(context, NSMaxX(rect) - radius, NSMaxY(rect));
    CGContextAddArcToPoint(context, NSMaxX(rect), NSMaxY(rect), NSMaxX(rect), NSMaxY(rect) - radius, radius);
    CGContextAddLineToPoint(context, NSMaxX(rect), NSMinY(rect) + radius);
    CGContextAddArcToPoint(context, NSMaxX(rect), NSMinY(rect), NSMaxX(rect) - radius, NSMinY(rect), radius);
    CGContextAddLineToPoint(context, NSMinX(rect) + radius, NSMinY(rect));
    CGContextAddArcToPoint(context, NSMinX(rect), NSMinY(rect), NSMinX(rect), NSMinY(rect) + radius, radius);
    CGContextClosePath(context);
    CGContextFillPath(context);
}

- (void)drawHorizontalSelectionInRect:(NSRect)rect;
{
    double height;
    
    [[NSColor selectedControlColor] set];
    NSRectFill(rect);

    [[NSColor controlShadowColor] set];
    height = NSHeight(rect);
    rect.size.height = 1.0;
    NSRectFill(rect);
    rect.origin.y += height;
    NSRectFill(rect);
}

- (void) drawSelfAndSubviewsInRect: (NSRect) rect;
{
    unsigned int subviewIndex, subviewCount;
    
    [self drawRect: rect];
    subviewCount = [_subviews count];
    for (subviewIndex = 0; subviewIndex < subviewCount; subviewIndex++) {
        NSRect subviewRect;
        NSView *subview;
        
        subview = [_subviews objectAtIndex: subviewIndex];
        subviewRect = [self convertRect: rect toView: subview];
        subviewRect = NSIntersectionRect(subviewRect, [subview bounds]);
        if (NSWidth(subviewRect) > 0.0) {
            [subview lockFocus];
            [subview drawSelfAndSubviewsInRect: subviewRect];
            [subview unlockFocus];
        }
    }
}


// Scrolling

- (void)scrollDownByPercentage:(float)percentage;
{
    NSRect visibleRect;
    NSClipView *clipView;
    NSRect oldClipBounds, newClipBounds;

    if (![[self superview] respondsToSelector:@selector(documentVisibleRect)])
        return;

    clipView = (NSClipView *)[self superview];
    visibleRect = [self visibleRect];
    visibleRect.origin.y += percentage * NSHeight(visibleRect);
    oldClipBounds = [clipView bounds];
    [self scrollPoint:visibleRect.origin];
    newClipBounds = [clipView bounds];
    if (NSMinY(oldClipBounds) == NSMinY(newClipBounds) &&
        NSMinX(oldClipBounds) == NSMinX(newClipBounds))
        return;
    if (![self canDraw])
        [[clipView superview] setNeedsDisplay:YES];
}

- (float)fraction;
{
    NSRect visibleRect;
    float fraction;

    visibleRect = [self visibleRect];
    
    if (NSHeight(visibleRect) >= NSHeight([self bounds]))
        return 0.0;

    fraction = NSMinY(visibleRect) /
      (NSHeight([self bounds]) - NSHeight(visibleRect));

    if (fraction <= 0.0)
        return 0.0;
    else if (fraction >= 1.0)
        return 1.0;
    else
        return fraction;
}

- (void)setFraction:(float)newFract;
{
    NSRect desiredRect;
    NSClipView *clipView;
    NSRect oldClipBounds, newClipBounds;

    if (![[self superview] respondsToSelector:@selector(documentVisibleRect)])
	return;

    desiredRect = [self visibleRect];

    if (NSHeight(desiredRect) >= NSHeight([self bounds]))
	return;			/* we're entirely visible */

    (&desiredRect)->origin.y = newFract *
      (NSHeight([self bounds]) - NSHeight(desiredRect));

    if (NSMinY(desiredRect) < 0.0)
	(&desiredRect)->origin.y = 0.0;
    else if (NSMaxY(desiredRect) > NSHeight([self bounds]))
	(&desiredRect)->origin.y = NSHeight([self bounds]) - NSHeight(desiredRect);

    clipView = (NSClipView *)[self superview];
    oldClipBounds = [clipView bounds];
    [self scrollPoint:desiredRect.origin];
    newClipBounds = [clipView bounds];
    if (NSMinY(oldClipBounds) == NSMinY(newClipBounds) &&
	NSMinX(oldClipBounds) == NSMinX(newClipBounds))
	return;
    if (![self canDraw])
	[[clipView superview] setNeedsDisplay: YES];
}


// Dragging

- (BOOL)shouldStartDragFromMouseDownEvent:(NSEvent *)event dragSlop:(float)dragSlop finalEvent:(NSEvent **)finalEventPointer;
{
    NSPoint eventLocation;
    NSRect slopRect;
    NSDate *distantFuture;

    OBPRECONDITION([event type] == NSLeftMouseDown);

    eventLocation = [event locationInWindow];
    slopRect = NSInsetRect(NSMakeRect(eventLocation.x, eventLocation.y, 0.0, 0.0), -dragSlop, -dragSlop);

    distantFuture = [NSDate distantFuture];
    while (1) {
        NSEvent *nextEvent;

        nextEvent = [NSApp nextEventMatchingMask:NSLeftMouseDraggedMask|NSLeftMouseUpMask untilDate:distantFuture inMode:NSEventTrackingRunLoopMode dequeue:YES];
        if (finalEventPointer)
            *finalEventPointer = nextEvent;

        if ([nextEvent type] == NSLeftMouseUp) {
	    return NO;
	    break;
	} else if (!NSMouseInRect([nextEvent locationInWindow], slopRect, NO)) {
            return YES;
	    break;
	}
    }
}

//
// Resizing
//

// TODO: If the difference between frames is small, we should ensure that we change by at least one pixel per loop in at least one direction.  In this case, we could end up looping for LESS than TRANSITION_TIME.
- (void) morphToFrame: (NSRect) newFrame;
{
    NSRect          currentFrame;
    NSTimeInterval  start, current, elapsed;
    
    currentFrame = [self frame];

    start = [NSDate timeIntervalSinceReferenceDate];    
    while (YES) {
        float  ratio;
        NSRect stepFrame;
        
        current = [NSDate timeIntervalSinceReferenceDate];
        elapsed = current - start;
        if (elapsed >  TRANSITION_TIME || [NSApp peekEvent])
            break;

        ratio = elapsed / TRANSITION_TIME;
        stepFrame.origin.x = ratio * newFrame.origin.x + (1.0 - ratio) * currentFrame.origin.x;
        stepFrame.origin.y = ratio * newFrame.origin.y + (1.0 - ratio) * currentFrame.origin.y;
        stepFrame.size.width = ratio * newFrame.size.width + (1.0 - ratio) * currentFrame.size.width;
        stepFrame.size.height = ratio * newFrame.size.height + (1.0 - ratio) * currentFrame.size.height;
        
        [self setFrame: stepFrame];
        [_window display];
        [_window flushWindow];
    }
    
    // Make sure we don't end up with round off errors
    [self setFrame: newFrame];
    [_window display];
    [_window flushWindow];
}


//
// View fade in/out
//

/*
The approach taken in both -fadeInSubview: and -fadeOutAndRemoveFromSuperview is to build two images and fade between them.  We could build only one image and ask the opaque superview to draw, but this could be arbitrarily expensive.  Instead, by only asking the view to draw once, we can build a more consistently performant method.
*/

- (void) fadeInSubview: (NSView *) subview;
{
    NSBitmapImageRep *oldImageRep, *newImageRep;
    NSImage          *newImage;
    NSTimeInterval    start, current, elapsed;
    NSRect            subviewFrame, subviewBounds, opaqueRect, localRect;
    NSWindow         *window;
    NSView           *opaqueView;
    
    if (!subview)
        return;

    subviewFrame = [subview frame];
    window = [self window];
    if (!window || ![window isVisible]) {
        // Don't lock focus since that'll raise an exception.  Also, we're not
        // visible.  Showing off by yourself is just silly.
        [self addSubview: subview];
        return;
    }
    
    opaqueView = [self opaqueAncestor];

    // Capture the old contents of the window
    [self lockFocus];
    oldImageRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: subviewFrame] autorelease];
    [self unlockFocus];
    //[[oldImageRep TIFFRepresentation] writeToFile: @"/tmp/old-fadein.tiff" atomically: YES];
    

    // Some views are really persistent about getting drawn when we don't
    // want them to (like NSOutlineView).  Turning off needs display on 
    // the window and all of its views doesn't work. 
    // We'll put the smack down on that...
    [_window disableFlushWindow];

    NS_DURING {
        [self addSubview: subview];
        subviewBounds = [subview bounds];
        opaqueRect = [subview convertRect: subviewBounds toView: opaqueView];
        
        [opaqueView lockFocus];
        [opaqueView drawSelfAndSubviewsInRect: opaqueRect];
        newImageRep = [[NSBitmapImageRep alloc] initWithFocusedViewRect: opaqueRect];
        [opaqueView unlockFocus];
        //[[newImageRep TIFFRepresentation] writeToFile: @"/tmp/new-fadein.tiff" atomically: YES];
    } NS_HANDLER {
        [_window enableFlushWindow];
        [localException raise];
    } NS_ENDHANDLER;
    
    // Make sure all the drawing happens while window flushing is off
    [NSApp peekEvent];
    [_window enableFlushWindow];
    
    newImage = [[[NSImage alloc] initWithSize: opaqueRect.size] autorelease];
    [newImage addRepresentation: newImageRep];
    [newImageRep release];
    
    localRect = [subview convertRect: [subview bounds] toView: self];

    // Now, fade from the starting image to the ending image
    start = [NSDate timeIntervalSinceReferenceDate];    
    while (YES) {
        current = [NSDate timeIntervalSinceReferenceDate];
        elapsed = current - start;
        if (elapsed >  TRANSITION_TIME || [NSApp peekEvent])
            break;

        [self lockFocus];
        [oldImageRep drawAtPoint: localRect.origin];
        [newImage dissolveToPoint: localRect.origin fraction: elapsed / TRANSITION_TIME];
        [self unlockFocus];
        
        [window flushWindow];
    }
    
    // Make sure the final version gets drawn
    [subview displayRect: subviewBounds];
    [_window flushWindow];
}

/* See notes above -fadeInSubview: for design of this method */
- (void) fadeOutAndRemoveFromSuperview;
{
    NSBitmapImageRep *oldImageRep, *newImageRep;
    NSImage          *newImage;
    NSTimeInterval    start, current, elapsed;
    NSView           *superview, *opaqueView;
    NSRect            opaqueRect, superRect;
    NSWindow         *window;
    
    // Hold onto this (since we are going to get removed)
    superview = [self superview];
    if (!superview)
        return;

    window = [superview window];
    if (!window || ![window isVisible]) {
        // Don't lock focus since that'll raise an exception.  Also, we're not
        // visible.  Showing off by yourself is just silly.
        [self removeFromSuperview];
        return;
    }

    opaqueView = [self opaqueAncestor];
    opaqueRect = [self convertRect: _bounds toView: opaqueView];
    superRect = [self convertRect: _bounds toView: superview];
    
    [self lockFocus];
    oldImageRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: _bounds] autorelease];
    [self unlockFocus];
    
    // When we get removed, we might get deallocated.  Make sure that this doesn't
    // happen util the animation is done.  Also, tell our window that it doesn't
    // need to be drawn since otherwise the NSEvent stuff below will cause it to get drawn
    [[self retain] autorelease];
    [self removeFromSuperview];
    [window setViewsNeedDisplay: NO];
    
    // Build the new image now that we are out of the way
    [opaqueView lockFocus];
    [opaqueView drawSelfAndSubviewsInRect: opaqueRect];
    newImageRep = [[NSBitmapImageRep alloc] initWithFocusedViewRect: opaqueRect];
    [opaqueView unlockFocus];
    
    newImage = [[[NSImage alloc] initWithSize: opaqueRect.size] autorelease];
    [newImage addRepresentation: newImageRep];
    [newImageRep release];
    
    
    // Now, fade from the starting image to the ending image
    start = [NSDate timeIntervalSinceReferenceDate];    
    while (YES) {
        current = [NSDate timeIntervalSinceReferenceDate];
        elapsed = current - start;
        if (elapsed >  TRANSITION_TIME || [NSApp peekEvent])
            break;

        [superview lockFocus];
        [oldImageRep drawAtPoint: superRect.origin];
        [newImage dissolveToPoint: superRect.origin fraction: elapsed / TRANSITION_TIME];
        [superview unlockFocus];
        
        [window flushWindow];
    }

    // Make sure the final version gets drawn
    [opaqueView displayRect: opaqueRect];
    [window flushWindow];
}

- (NSImage *) imageForRect: (NSRect) rect;
{
    NSBitmapImageRep *imageRep;
    NSImage *image;
    NSRect visibleRect, intersection;
    
    visibleRect = [self visibleRect];
    //NSLog(@"visible rect = %@", NSStringFromRect(visibleRect));
    //NSLog(@"requested rect = %@", NSStringFromRect(rect));
    
    intersection = NSIntersectionRect([self visibleRect], rect);
    if (!NSEqualRects(rect, intersection)) {
        [NSException raise: NSInvalidArgumentException
                    format: @"-[NSView imageForRect:] -- Requested rect %@ is not totally contained in the visible rect %@", NSStringFromRect(rect), NSStringFromRect(visibleRect)];
    }
    
    [self lockFocus];
    imageRep = [[NSBitmapImageRep alloc] initWithFocusedViewRect: rect];
    [self unlockFocus];

    NSLog(@"imageRep = %@", imageRep);
        
    image = [[[NSImage alloc] initWithSize: rect.size] autorelease];
    [image addRepresentation: imageRep];
    [imageRep release];
    

    
    return image;
}

/*" oldImage should have the same size as the visible rect of the receiver.  Starts oldImage in the visible rect and slides it so that it moves in the specified direction.  The exposed edge of the view is filled with progressively greater slices of newImage.  newImage need not be the same size as the visible rect of the receiver, but it must be the same length along the axis perpendicular to the movement direction. "*/

#warning This should really take a target subrect in the view in which all the activity will happen.  This would allow us to do some of the OAExtenededOutlineView stuff in terms of this.
- (void) slideOutOldImage: (NSImage *) oldImage
                 newImage: (NSImage *) newImage
           slideDirection: (OAImageSlideDirection) direction;
{
    NSSize oldSize, newSize;
    NSRect visibleRect;
#ifdef TIME_LIMIT        
    NSTimeInterval start, current, elapsed;
#endif
    unsigned int pixels, slideDistance;
    
    //NSLog(@"direction = %d", direction);
    
    // Only doing up for now.
    if (direction != OAUpSlideDirection)
        return;
    
    oldSize = [oldImage size];
    newSize = [newImage size];
    visibleRect = [self visibleRect];
    //NSLog(@"oldSize = %@", NSStringFromSize(oldSize));
    //NSLog(@"newSize = %@", NSStringFromSize(newSize));
    //NSLog(@"visibleRect = %@", NSStringFromRect(visibleRect));
    
#warning  Add checks for the sizes of the images vs the current visible rect and the direction of movement.

#warning Make this handle non-flipped views
    OBASSERT([self isFlipped]);
    
    slideDistance = newSize.height;
#ifdef TIME_LIMIT
    start = [NSDate timeIntervalSinceReferenceDate];    
#else
    pixels = 0;
#endif
    do {
        NSRect oldSubRect, newSubRect;
        NSPoint oldPoint, newPoint;

#ifdef TIME_LIMIT        
        current = [NSDate timeIntervalSinceReferenceDate];
        elapsed = current - start;
        
        pixels = (elapsed / TRANSITION_TIME) * slideDistance;
        if (pixels > slideDistance /*|| [NSApp peekEvent]*/)
            pixels = slideDistance;
#else
        pixels++;
#endif

            
//        NSLog(@"pixels = %d", pixels);
        
        oldSubRect.origin.x = 0;
        oldSubRect.origin.y = 0;
        oldSubRect.size.width = oldSize.width;
        oldSubRect.size.height = oldSize.height - pixels;
        oldPoint.x = visibleRect.origin.x;
        oldPoint.y = NSMaxY(visibleRect) - pixels;
        
        newSubRect.origin.x = 0;
        newSubRect.origin.y = newSize.height - pixels;
        newSubRect.size.width = newSize.width;
        newSubRect.size.height = pixels;
        newPoint.x = visibleRect.origin.x;
        newPoint.y = NSMaxY(visibleRect);
        
//        NSLog(@"oldSubRect = %@", NSStringFromRect(oldSubRect));
//        NSLog(@"oldPoint = %@", NSStringFromPoint(oldPoint));
//        NSLog(@"newSubRect = %@", NSStringFromRect(newSubRect));
//        NSLog(@"newPoint = %@", NSStringFromPoint(newPoint));
        
        [self lockFocus];
        [oldImage compositeToPoint: oldPoint fromRect: oldSubRect operation: NSCompositeCopy];
        [newImage compositeToPoint: newPoint fromRect: newSubRect operation: NSCompositeCopy];
        [self unlockFocus];
        
        [_window flushWindow];
    } while (pixels < slideDistance);
}

// Debugging

unsigned int NSViewMaxDebugDepth = 10;

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [NSMutableDictionary dictionary];
    [debugDictionary setObject:OBShortObjectDescription(self) forKey:@"__self__"];
    [debugDictionary setObject:NSStringFromRect([self frame]) forKey:@"01_frame"];
    if (!NSEqualSizes([self bounds].size, [self frame].size) || !NSEqualPoints([self bounds].origin, NSZeroPoint))
        [debugDictionary setObject:NSStringFromRect([self bounds]) forKey:@"02_bounds"];
    if ([[self subviews] count] > 0)
        [debugDictionary setObject:[self subviews] forKey:@"subviews"];
    return debugDictionary;
}

- (NSString *)descriptionWithLocale:(NSDictionary *)locale indent:(unsigned int)level;
{
    if (level < NSViewMaxDebugDepth)
        return [[self debugDictionary] descriptionWithLocale:locale indent:level];
    else
        return [self shortDescription];
}

- (NSString *)description;
{
    return [self descriptionWithLocale:nil indent:0];
}

- (NSString *)shortDescription;
{
    return [super description];
}

@end
