// Copyright 2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OAExtendedOutlineView.h"

#import <OmniBase/OmniBase.h>
#import <AppKit/AppKit.h>

#import "NSImage-OAExtensions.h"
#import "NSOutlineView-OAExtensions.h"
#import "NSString-OAExtensions.h"
#import "NSView-OAExtensions.h"
#import "OAInspector.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OAExtendedOutlineView.m,v 1.42 2000/07/09 21:07:01 kc Exp $")

@interface OAExtendedOutlineView (Private)
- (void)startDrag:(NSEvent *)event;
- (void)registerForDragging;
- (void)expandItemAndChildren:(id)item;
- (void)collapseItemAndChildren:(id)item;
@end

#define DISCLOSURE_TRIANGLE_WIDTH 17.0

@implementation OAExtendedOutlineView

- (id)initWithFrame:(NSRect)rect;
{
    if (![super initWithFrame:rect])
        return nil;

    shouldEditNextItemWhenEditingEnds = YES;

    return self;
}

- initWithCoder:(NSCoder *)coder;
{
    if (!(self = [super initWithCoder:coder]))
        return nil;

    shouldEditNextItemWhenEditingEnds = YES;
    
    return self;
}

- (void)dealloc;
{
    [self unregisterDraggedTypes];
    [super dealloc];
}

// API

- (id)parentItemForRow:(int)row child:(unsigned int *)childIndexPointer;
{
    int originalLevel;
    
    originalLevel = [self levelForRow:row];
    return [self parentItemForRow:row indentLevel:originalLevel child:childIndexPointer];
}

- (id)parentItemForRow:(int)row indentLevel:(int)childLevel child:(unsigned int *)childIndexPointer;
{
    unsigned int childIndex;

    childIndex = 0;

    while (row-- >= 0) {
        int currentLevel;
        
        currentLevel = [self levelForRow:row];
        if (currentLevel < childLevel) {
            if (childIndexPointer)
                *childIndexPointer = childIndex;
            return [self itemAtRow:row];
        } else if (currentLevel == childLevel)
            childIndex++;
    }
    if (childIndexPointer)
        *childIndexPointer = childIndex;
    return nil;
}

- (BOOL)shouldEditNextItemWhenEditingEnds;
{
    return shouldEditNextItemWhenEditingEnds;
}

- (void)setShouldEditNextItemWhenEditingEnds:(BOOL)value;
{
    shouldEditNextItemWhenEditingEnds = value;
}

// Actions

- (IBAction)expandAll:(id)sender;
{
    NSArray *selectedItems;

    selectedItems = [self selectedItems];
    [self expandItemAndChildren:nil];
    [self setSelectedItems:selectedItems];
}

- (IBAction)contractAll:(id)sender;
{
    NSArray *selectedItems;

    selectedItems = [self selectedItems];
    [self collapseItemAndChildren:nil];
    [self setSelectedItems:selectedItems];
}

- (IBAction)expandSelection:(id)sender;
{
    NSArray *selectedItems;
    NSEnumerator *itemEnumerator;
    id item;
        
    selectedItems = [self selectedItems];
    itemEnumerator = [selectedItems objectEnumerator];
    while ((item = [itemEnumerator nextObject]))
        [self expandItemAndChildren:item];
    
    [self setSelectedItems:selectedItems];
}

- (IBAction)contractSelection:(id)sender;
{
    NSArray *selectedItems;
    NSEnumerator *itemEnumerator;
    id item;
    
    selectedItems = [self selectedItems];
    itemEnumerator = [selectedItems reverseObjectEnumerator];
    while ((item = [itemEnumerator nextObject]))
        [self collapseItemAndChildren:item];
    
    [self setSelectedItems:selectedItems];
}

// Just like NSText

- (IBAction)copy:(id)sender;
{
    if ([[self dataSource] respondsToSelector:@selector(outlineView:copyItem:toPasteboard:)])
        [[self dataSource] outlineView:self copyItem:[self selectedItem] toPasteboard:[NSPasteboard generalPasteboard]];
}

- (IBAction)cut:(id)sender;
{
    [self copy:sender];
    [self delete:sender];
}

- (IBAction)delete:(id)sender;
{
    if ([[self dataSource] respondsToSelector:@selector(outlineView:deleteItem:parentItem:)]) {
        id selectedItem;
        int selectedRow;    
        int numberOfRows;
            
        selectedItem = [self selectedItem];
        selectedRow = [self selectedRow];
        
        [[self dataSource] outlineView:self deleteItem:selectedItem parentItem:[self parentItemForRow:[self selectedRow] child:NULL]];
        [self reloadData];
        
        // Maintain the selection after deletions
        numberOfRows = [self numberOfRows];
        if (selectedRow > (numberOfRows - 1))
            [self selectRow:(numberOfRows - 1) byExtendingSelection:NO];
        else
            [self selectRow:selectedRow byExtendingSelection:NO];
    }
}

- (IBAction)paste:(id)sender;
{
    if ([[self dataSource] respondsToSelector:@selector(outlineView:pasteItemFromPasteboard:parentItem:child:)]) {
        int selectedRow, numberOfRows;
        id parentItem;
        int childIndex;
    
        numberOfRows = [self numberOfRows];
        selectedRow = [self selectedRow];
        if (numberOfRows == 0 || selectedRow == -1) {
            parentItem = nil;
            childIndex = 0;
        } else {
            parentItem = [self parentItemForRow:selectedRow child:&childIndex];
            childIndex++; // Paste after current line
        }
            
        [[self dataSource] outlineView:self pasteItemFromPasteboard:[NSPasteboard generalPasteboard] parentItem:parentItem child:childIndex];
        [self reloadData];
    }
}


// NSResponder

- (BOOL)becomeFirstResponder;
{
    BOOL willBecome;

    willBecome = [super becomeFirstResponder];
    if (willBecome)
        [[NSNotificationCenter defaultCenter] postNotificationName:OAInspectorSelectionDidChangeNotification object:self];
    return willBecome;
}

- (BOOL)resignFirstResponder;
{
    BOOL willResign;

    willResign = [super resignFirstResponder];
    if (willResign)
        [[NSNotificationCenter defaultCenter] postNotificationName:OAInspectorSelectionDidChangeNotification object:self];
    return willResign;
}

- (void)keyDown:(NSEvent *)theEvent;
{
    NSString *characters;
    unsigned int modifierFlags;

    characters = [theEvent characters];
    modifierFlags = [theEvent modifierFlags];

    if ([characters hasPrefix:@"e"]) {
        // Mmmm, arbitrary constants
        [self editColumn:0 row:[self selectedRow] withEvent:nil select:YES];
    } else if ([characters hasPrefix:@"<"]) {
        [self contractAll:nil];
        [self selectRow:0 byExtendingSelection:NO];
    } else if ([characters hasPrefix:@">"]) {
        id selectedItem;
        int row;
        
        selectedItem = [self itemAtRow:[self selectedRow]];
        [self expandAll:nil];

        if ((row = [self rowForItem:selectedItem]) >= 0)
            [self selectRow:row byExtendingSelection:NO];
    } else if ([characters characterAtIndex:0] == NSLeftArrowFunctionKey && modifierFlags & NSShiftKeyMask)
        [self contractSelection:nil];
    else if ([characters characterAtIndex:0] == NSRightArrowFunctionKey && modifierFlags & NSShiftKeyMask)
        [self expandSelection:nil];
    else
        [self interpretKeyEvents:[NSArray arrayWithObject:theEvent]];
}

- (void)deleteForward:(id)sender;
{
    if ([self selectedItem])
        [self delete:nil];
}

- (void)deleteBackward:(id)sender;
{
    if ([self selectedItem])
        [self delete:nil];
}

- (void)insertTab:(id)sender;
{
    id selectedItem;
    int selectedRow;
    int levelOfSelectedItem;
    int currentRow;
    
    // We can't do this if they don't implement outlineView:parentItem:moveChildren:toNewParentItem
    if ([[self dataSource] respondsToSelector:@selector(outlineView:parentItem:moveChildren:toNewParentItem:)] == NO)
        return;
    
    selectedRow = [self selectedRow];
    if (selectedRow == 0 || selectedRow == -1)
        return;
        
    selectedItem = [self itemAtRow:selectedRow];
    levelOfSelectedItem = [self levelForItem:selectedItem];
    
    currentRow = selectedRow - 1;
    while (currentRow >= 0) {
        id potentialParent;
        int levelOfPotentialParent;
        
        potentialParent = [self itemAtRow:currentRow];
        levelOfPotentialParent = [self levelForItem:potentialParent];
        
        if (levelOfPotentialParent == levelOfSelectedItem) {
            NSArray *movingChildren;
            id previousParent;
            unsigned int childIndex;
            
            movingChildren = [NSArray arrayWithObject:selectedItem];
            previousParent = [self parentItemForRow:selectedRow child:&childIndex];
    
            // If you're the zeroth child of your parent, you cannot be indented any more.
            if (childIndex == 0)
                return;

            [[self dataSource] outlineView:self parentItem:previousParent moveChildren:movingChildren toNewParentItem:potentialParent];
            [self expandItem:potentialParent];

            // Reload those items which were affected
            if (previousParent == nil) {
                [self reloadData];
            } else {
                [self reloadItem:previousParent reloadChildren:YES];
                [self reloadItem:potentialParent reloadChildren:YES];
            }
            
            [self selectRow:[self rowForItem:selectedItem] byExtendingSelection:NO];

            break;
        }
        
        // Move upwards through the list
        currentRow--;
    }
}

- (void)insertBacktab:(id)sender;
{
    NSMutableArray *peersAfterSelectedItem;
    id selectedItem;
    id parent;
    id parentsParent;
    int selectedRow;
    int levelOfSelectedItem;
    int peerIndex;
    int parentChildrenCount;
    int parentIndex;
    
    // Determine if the dataSource supports the required extension methods for this feature, and if so, determine if this operation is valid for the current selection
    if ([[self dataSource] respondsToSelector:@selector(outlineView:parentItem:moveChildren:toNewParentItem:beforeIndex:)] == NO)
        return;
    
    selectedRow = [self selectedRow];
    selectedItem = [self itemAtRow:selectedRow];
    levelOfSelectedItem = [self levelForItem:selectedItem];

    if (selectedRow == -1 || selectedRow == 0 || levelOfSelectedItem == 0)
        return;

    parent = [self parentItemForRow:selectedRow child:&peerIndex];
    parentChildrenCount = [[self dataSource] outlineView:self numberOfChildrenOfItem:parent];
    peersAfterSelectedItem = [NSMutableArray array];
    
    // The peers of the selection which come after it will become children of the selected item
    for (peerIndex += 1; peerIndex < parentChildrenCount; peerIndex++) {
        id peer;
        
        peer = [[self dataSource] outlineView:self child:peerIndex ofItem:parent];
        [peersAfterSelectedItem addObject:peer];
    }

    // If there were any peers after the selection, move them to the end of the selected item's list of children
    if ([peersAfterSelectedItem count] > 0) {
        [[self dataSource] outlineView:self parentItem:parent moveChildren:peersAfterSelectedItem toNewParentItem:selectedItem beforeIndex:-1];
    
        // Make sure the selected item is expanded
        if ([[self dataSource] outlineView:self isItemExpandable:selectedItem])
            [self expandItem:selectedItem];
    }

    parentsParent = [self parentItemForRow:[self rowForItem:parent] child:&parentIndex];

    // Make the selection become a peer of its parent
    [[self dataSource] outlineView:self parentItem:parent moveChildren:[NSArray arrayWithObject:selectedItem] toNewParentItem:parentsParent beforeIndex:(parentIndex + 1)];
    
    [self reloadData];

    // Reselect row since, in some situations, NSOutlineView gets confused with regards to what should be selected
    [self selectRow:[self rowForItem:selectedItem] byExtendingSelection:NO];

}

- (void)insertNewline:(id)sender;
{
    if ([[self dataSource] respondsToSelector:@selector(outlineView:createNewItemAsChild:ofItem:)]) {
        int selectedRow, numberOfRows;
        id parentItem;
        int childIndex;
    
        numberOfRows = [self numberOfRows];
        selectedRow = [self selectedRow];
        if (numberOfRows == 0 || selectedRow == -1) {
            parentItem = nil;
            childIndex = 0;
        } else {
            parentItem = [self parentItemForRow:selectedRow child:&childIndex];
            childIndex++; // Insert after current line
        }

        if ([[self dataSource] outlineView:self createNewItemAsChild:childIndex ofItem:parentItem]) {
            id item;
            int newItemRow;
            NSArray *tableColumns;
            int columnIndex, columnCount;
        
            [self reloadData];
            item = [[self dataSource] outlineView:self child:childIndex ofItem:parentItem];
            newItemRow = [self rowForItem:item];
            [self selectRow:newItemRow byExtendingSelection:NO];
            
            // Edit the first editable column            
            tableColumns = [self tableColumns];
            columnCount = [tableColumns count];
            for (columnIndex = 0; columnIndex < columnCount; columnIndex++) {
                NSTableColumn *column;
                
                column = [tableColumns objectAtIndex:columnIndex];
                if ([column isEditable] && [[self delegate] outlineView:self shouldEditTableColumn:column item:item]) {
                    [self editColumn:columnIndex row:newItemRow withEvent:nil select:YES];
                    break;
                }
            }
        }
    }
}

- (void)moveUp:(id)sender;
{
    int selectedRow;
    
    selectedRow = [self selectedRow];
    if (selectedRow == 0 || selectedRow == -1)
        return;
        
    [self selectRow:--selectedRow byExtendingSelection:NO];
    [self scrollRowToVisible:selectedRow];
}

- (void)moveDown:(id)sender;
{
    int selectedRow;
    
    selectedRow = [self selectedRow];
    if (selectedRow == -1 || (selectedRow >= [self numberOfRows] - 1))
        return;
        
    [self selectRow:++selectedRow byExtendingSelection:NO];
    [self scrollRowToVisible:selectedRow];
}

- (void)moveLeft:(id)sender
{
    id selectedItem;
    int selectedRow;

    selectedRow = [self selectedRow];
    selectedItem = [self itemAtRow:selectedRow];

    if ([[self dataSource] outlineView:self isItemExpandable:selectedItem])
        [self collapseItem:selectedItem];
}

- (void)moveRight:(id)sender;
{
    id selectedItem;
    int selectedRow;

    selectedRow = [self selectedRow];
    selectedItem = [self itemAtRow:selectedRow];

    if ([[self dataSource] outlineView:self isItemExpandable:selectedItem])
        [self expandItem:selectedItem];
}

// NSView

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent;
{
    return YES;
}

- (BOOL)shouldDelayWindowOrderingForEvent:(NSEvent *)anEvent 
{
    return YES;
}

- (void)setFrameSize:(NSSize)newSize;
{
    NSRect contentViewFrame;

    // Normally, our frame size is exactly tall enough to fit all of our rows. However, this causes a problem when the outline view is shorter than the scroll view's content view (when there are not enough rows to fill up the whole area). When we drag to the bottom of the outline, the drag image gets clipped to the last row. To fix this problem, we force ourself to always be at least as tall as the scroll view's content view.
    contentViewFrame = [[[self enclosingScrollView] contentView] frame];
    newSize.height = MAX(newSize.height, contentViewFrame.size.height);

    [super setFrameSize:newSize];
}

- (void)drawRect:(NSRect)rect;
{
    NSRect rowFrame;

    [super drawRect:rect];

/* Commenting this out since it's ugly, blinky, and makes it difficult to see the drag destination indicator.
    if (draggingSourceItem && (!flags.isDraggingDestination || flags.isDraggingDestinationAcceptable)) {
        rowFrame = [self rectOfRow:dragSourceRow];
        [[NSColor colorWithDeviceRed:0.2 green:0.2 blue:0.4 alpha:0.8] set];
        NSRectFillUsingOperation(rowFrame, NSCompositeSourceOver);
    } */

    if (flags.isDraggingDestination && flags.isDraggingDestinationAcceptable) {
        NSRect outlineTableColumnRect;
        NSPoint insertionPoint;
        NSBezierPath *path;
        
        if (dragDestinationRow >= [self numberOfRows]) {
            rowFrame = [self rectOfRow:dragDestinationRow-1];
            rowFrame.origin.y = NSMaxY(rowFrame);
        } else
            rowFrame = [self rectOfRow:dragDestinationRow];
        
        outlineTableColumnRect = [self rectOfColumn:[[self tableColumns] indexOfObject:[self outlineTableColumn]]];
        
        insertionPoint.x = NSMinX(outlineTableColumnRect) + [self indentationPerLevel] * dragDestinationLevel + DISCLOSURE_TRIANGLE_WIDTH;
        insertionPoint.y = rowFrame.origin.y;

        path = [NSBezierPath bezierPath];
        [path appendBezierPathWithArcWithCenter:insertionPoint radius:4.5 startAngle:0 endAngle:360];
        [[NSColor whiteColor] set];
        [path fill];

        path = [NSBezierPath bezierPath];
        [path appendBezierPathWithArcWithCenter:insertionPoint radius:2.5 startAngle:0 endAngle:360];
        [path setLineWidth:1.5];
        [[NSColor blackColor] set];
        [path stroke];
        
        path = [NSBezierPath bezierPath];
        [path moveToPoint:NSMakePoint(insertionPoint.x + 2, insertionPoint.y)];
        [path relativeLineToPoint:NSMakePoint(NSMaxX(rowFrame), 0)];
        [path setLineWidth:2.0];
        [path stroke];
    }
}

// NSControl

- (void)mouseDown:(NSEvent *)event;
{
    NSPoint eventLocationInWindow, eventLocation;
    int columnIndex, rowIndex;
    NSRect slopRect;
    const int dragSlop = 4;
    NSEvent *mouseDragCurrentEvent;

    if (![[self dataSource] respondsToSelector:@selector(outlineView:copyItem:toPasteboard:)]) {
        [super mouseDown:event];
        return;
    }

    eventLocationInWindow = [event locationInWindow];
    eventLocation = [self convertPoint:eventLocationInWindow fromView:nil];
    columnIndex = [self columnAtPoint:eventLocation];
    rowIndex = [self rowAtPoint:eventLocation];
    if (rowIndex == -1 || columnIndex == -1)
        return;

    if ([[self tableColumns] objectAtIndex:columnIndex] == [self outlineTableColumn]) {
        NSRect cellRect;
        
        cellRect = [self frameOfCellAtColumn:columnIndex row:rowIndex];
        if (eventLocation.x < NSMinX(cellRect)) {
            [super mouseDown:event];
            return;
        }
    }

    slopRect = NSInsetRect(NSMakeRect(eventLocationInWindow.x, eventLocationInWindow.y, 0.0, 0.0), -dragSlop, -dragSlop);
    while (1) {
        NSEvent *nextEvent;

        nextEvent = [NSApp nextEventMatchingMask:NSLeftMouseDraggedMask|NSLeftMouseUpMask untilDate:[NSDate distantFuture] inMode:NSEventTrackingRunLoopMode dequeue:NO];
        mouseDragCurrentEvent = nextEvent;

        if ([nextEvent type] == NSLeftMouseUp) {
            break;
        } else {
            [NSApp nextEventMatchingMask:NSLeftMouseDraggedMask untilDate:[NSDate distantFuture] inMode:NSEventTrackingRunLoopMode dequeue:YES];
            if (!NSMouseInRect([nextEvent locationInWindow], slopRect, NO)) {
                [self startDrag:event];
                return;
            }
        }
    }

    [super mouseDown:event];
}

// NSTableView

- (void)setDataSource:(id)aSource;
{
    [super setDataSource:aSource];
    
    [self registerForDragging];
}

- (void)textDidEndEditing:(NSNotification *)notification;
{
    if (shouldEditNextItemWhenEditingEnds == NO) {
        // This is ugly, but just about the only way to do it. NSTableView is determined to select and edit something else, even the text field that it just finished editing, unless we mislead it about what key was pressed to end editing.
        NSMutableDictionary *newUserInfo;
        NSNotification *newNotification;
        
        newUserInfo = [NSMutableDictionary dictionaryWithDictionary:[notification userInfo]];
        [newUserInfo setObject:[NSNumber numberWithInt:0] forKey:@"NSTextMovement"];
        newNotification = [NSNotification notificationWithName:[notification name] object:[notification object] userInfo:newUserInfo];
        [super textDidEndEditing:newNotification];

        // For some reason we lose firstResponder status when when we do the above.
        [[self window] makeFirstResponder:self];
    } else {
        [super textDidEndEditing:notification];
    }
}


// NSOutlineView


// NSDraggingDestination

- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender;
{
    flags.isDraggingDestination = YES;
    dragDestinationRow = NSNotFound;
    return [self draggingUpdated:sender];
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender;
{
    NSPoint draggedImageLocation;
    int requestedRow;
    int maximumLevel, minimumLevel, requestedLevel;
    BOOL allowDrag;
    
    draggedImageLocation = [self convertPoint:[sender draggedImageLocation] fromView:nil];
    draggedImageLocation.x -= DISCLOSURE_TRIANGLE_WIDTH + NSMinX([self rectOfColumn:[[self tableColumns] indexOfObject:[self outlineTableColumn]]]);

    // Figure out the row at which this drag would drop
    requestedRow = [self rowAtPoint:draggedImageLocation];
    if (requestedRow == -1)
        requestedRow = [self numberOfRows];

    // Kind of a special case: if we are trying to drop right AFTER the row we are dragging, then that's really the same as dropping ON the current row, so we change the requestedRow so when we special-case the drawing we only have to handle one special case.
    if (draggingSourceItem && requestedRow == dragSourceRow+1)
        requestedRow = dragSourceRow;
    
    // Figure out the indention level at which this drag would drop
    if (requestedRow == 0) {
        requestedLevel = 0;
    } else {
        maximumLevel = [self levelForRow:requestedRow-1]+1;
        
        if (draggingSourceItem && requestedRow == dragSourceRow) {
            if (requestedRow >= [self numberOfRows]-1)
                minimumLevel = 0;
            else
                minimumLevel = MAX([self levelForRow:requestedRow+1]-1, 0);
        } else {
            if (requestedRow == [self numberOfRows])
                minimumLevel = 0;
            else
                minimumLevel = MAX([self levelForRow:requestedRow]-1, 0);
        }
        requestedLevel = MAX(MIN(rint(draggedImageLocation.x / [self indentationPerLevel]), maximumLevel), minimumLevel);
    }

    // Give the dataSource a chance to change or deny which indention level we'll drop to.
    do {
        id parentItem;
        unsigned int childIndex;
        
        // Is it OK to drop where we are right now?
        parentItem = [self parentItemForRow:requestedRow indentLevel:requestedLevel child:&childIndex];
        allowDrag = [[self dataSource] outlineView:self allowPasteItemFromPasteboard:[sender draggingPasteboard] parentItem:parentItem child:childIndex];
        if (!allowDrag) {
            int normalDestinationLevel;
            
            normalDestinationLevel = [self levelForRow:requestedRow];
            if (requestedLevel == normalDestinationLevel) {
                flags.isDraggingDestinationAcceptable = NO;
                [self setNeedsDisplay:YES];
                return NSDragOperationNone;
            }
            requestedLevel = normalDestinationLevel;
        }
    } while (!allowDrag);
    
    // The drag is allowable (collapse drawing so we don't flicker if we've already this state)!
    if (requestedRow != dragDestinationRow || requestedLevel != dragDestinationLevel) {
        dragDestinationLevel = requestedLevel;
        dragDestinationRow = requestedRow;

        // Don't accept the drag if we're the source and we're dropping right where we started dragging.
        flags.isDraggingDestinationAcceptable = draggingSourceItem == nil || dragDestinationRow != dragSourceRow || dragDestinationLevel != dragSourceLevel;
    
        [self setNeedsDisplay:YES];
    }
    
    if (flags.isDraggingDestinationAcceptable)
        return [sender draggingSourceOperationMask];
    else
        return NSDragOperationNone;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender;
{
    flags.isDraggingDestination = NO;
    [self setNeedsDisplay:YES];
}

//- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender;
- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender;
{
    id parentItem;
    unsigned int childIndex;
    
    if (!flags.isDraggingDestinationAcceptable)
        return NO;

    if ([[self dataSource] respondsToSelector:@selector(undoManagerForOutlineView:)])
        [[[self dataSource] undoManagerForOutlineView:self] beginUndoGrouping];

    // If we are the drag source, we paste the item in the new position and later delete it in the old position. We don't want the intermediate step to be displayed, so we turn off the window updating until we're finished.
    if (draggingSourceItem != nil)
        [[self window] setAutodisplay:NO];
        
    parentItem = [self parentItemForRow:dragDestinationRow indentLevel:dragDestinationLevel child:&childIndex];
    [[self dataSource] outlineView:self pasteItemFromPasteboard:[sender draggingPasteboard] parentItem:parentItem child:childIndex];
    [self reloadData];

    // Must we move some children from our new sibling (who is above us now)?
    if (dragDestinationLevel < [self levelForRow:dragDestinationRow]) {
        id previousParent, newParent;
        NSMutableArray *movingChildren;
        unsigned int movingChildIndex, previousParentChildCount;

        previousParent = [self parentItemForRow:dragDestinationRow child:&movingChildIndex];
        previousParentChildCount = [[self dataSource] outlineView:self numberOfChildrenOfItem:previousParent];
        newParent = [[self dataSource] outlineView:self child:childIndex ofItem:parentItem];
        
        movingChildren = [NSMutableArray arrayWithCapacity:previousParentChildCount];
        for (; movingChildIndex < previousParentChildCount; movingChildIndex++) {
            id movingChild;
            
            movingChild = [[self dataSource] outlineView:self child:movingChildIndex ofItem:previousParent];
            if (movingChild == draggingSourceItem || movingChild == newParent)
                continue;
                
            [movingChildren addObject:movingChild];
        }

        [[self dataSource] outlineView:self parentItem:previousParent moveChildren:movingChildren toNewParentItem:newParent];
        [self reloadData];
    }
    
    if ([[self dataSource] respondsToSelector:@selector(undoManagerForOutlineView:)]) {
        [[[self dataSource] undoManagerForOutlineView:self] setActionName:@"Drag Operation"];
        [[[self dataSource] undoManagerForOutlineView:self] endUndoGrouping];
    }

    flags.justAcceptedDrag = YES;
    return YES;
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender;
{
    flags.isDraggingDestination = NO;
    if (draggingSourceItem != nil) {
        // We're the source of the drag:  we'll draw ourselves after we delete the source item.
    } else {
        [self setNeedsDisplay:YES];
    }
}


// NSDraggingSource

- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)isLocal;
{
    if (isLocal)
        return NSDragOperationGeneric;
    else
        return NSDragOperationCopy;
}

//- (void)draggedImage:(NSImage *)image beganAt:(NSPoint)screenPoint;
//- (void)draggedImage:(NSImage *)image endedAt:(NSPoint)screenPoint deposited:(BOOL)flag;
//- (BOOL)ignoreModifierKeysWhileDragging;

// OAFindControllerTarget protocol

- (BOOL)findString:(NSString *)textPattern ignoreCase:(BOOL)ignoreCase backwards:(BOOL)backwards wrap:(BOOL)wrap;
{
#warning Need to implement Find
    return NO;
}

@end

@implementation OAExtendedOutlineView (Private)

- (void)startDrag:(NSEvent *)event;
{
    int outlineTableColumnIndex;
    NSRect rowFrame;
    double xOffsetOfFirstColumn;
    NSPoint eventLocation;
    NSCachedImageRep *cachedImageRep;
    NSRect imageFrame;
    NSView *contentView;
    NSImage *dragImage;
    NSPasteboard *pasteboard;
    id parentItem;
    NSArray *selectedItems;
    
    eventLocation = [self convertPoint:[event locationInWindow] fromView:nil];
    dragSourceRow = [self rowAtPoint:eventLocation];
    dragSourceLevel = [self levelForRow:dragSourceRow];

    [self selectRow:dragSourceRow byExtendingSelection:NO];
    [self collapseItem:[self itemAtRow:dragSourceRow]];

    // Cache an image for the current row
    outlineTableColumnIndex = [[self tableColumns] indexOfObject:[self outlineTableColumn]];
    rowFrame = [self rectOfRow:dragSourceRow];
    xOffsetOfFirstColumn = [self frameOfCellAtColumn:outlineTableColumnIndex row:dragSourceRow].origin.x;
    imageFrame = NSMakeRect(0, 0, NSWidth(rowFrame)-xOffsetOfFirstColumn, NSHeight(rowFrame));

    cachedImageRep = [[NSCachedImageRep alloc] initWithSize:imageFrame.size depth:[[NSScreen mainScreen] depth] separate:YES alpha:YES];
    contentView = [[cachedImageRep window] contentView];

    [contentView lockFocus];
    {
        id item;
        int columnIndex;
    
        [[NSColor colorWithDeviceWhite:0.0 alpha:0.0] set];
        NSRectFillUsingOperation(imageFrame, NSCompositeCopy);
    
        item = [self itemAtRow:dragSourceRow];
        for (columnIndex = outlineTableColumnIndex; columnIndex < [self numberOfColumns]; columnIndex++) {
            NSTableColumn *tableColumn;
            NSCell *cell;
            NSRect cellRect;
            id objectValue;
            
            tableColumn = [[self tableColumns] objectAtIndex:columnIndex];
            objectValue = [[self dataSource] outlineView:self objectValueForTableColumn:tableColumn byItem:item];

            cellRect = [self frameOfCellAtColumn:columnIndex row:dragSourceRow];
            cellRect.origin = NSMakePoint(NSMinX(cellRect) - xOffsetOfFirstColumn, 0);
            cell = [tableColumn dataCellForRow:dragSourceRow];
            
            if ([objectValue isKindOfClass:[NSString class]])
                [objectValue drawOutlinedWithFont:nil color:nil backgroundColor:nil rectangle:[cell titleRectForBounds:cellRect]];
            else {
                [cell setCellAttribute:NSCellHighlighted to:0];
                [cell setObjectValue:objectValue];
        
                [cell drawWithFrame:cellRect inView:contentView];
            }
        }
    }
    [contentView unlockFocus];

    dragImage = [[NSImage alloc] init];
    [dragImage addRepresentation:cachedImageRep];
    [cachedImageRep release];

    // Let's start the drag.
    if ([[self dataSource] respondsToSelector:@selector(undoManagerForOutlineView:)])
        [[[self dataSource] undoManagerForOutlineView:self] beginUndoGrouping];
        
    selectedItems = [self selectedItems];

    flags.justAcceptedDrag = NO;
    pasteboard = [NSPasteboard pasteboardWithUniqueName];

    draggingSourceItem = [[self itemAtRow:dragSourceRow] retain];
    parentItem = [self parentItemForRow:[self rowForItem:draggingSourceItem] child:NULL];
    [[self dataSource] outlineView:self copyItem:draggingSourceItem toPasteboard:pasteboard];
    
    [self dragImage:dragImage at:NSMakePoint(NSMinX(rowFrame)+xOffsetOfFirstColumn, NSMaxY(rowFrame) - 1) offset:NSMakeSize(0, 0) event:event pasteboard:pasteboard source:self slideBack:YES];
    [self setNeedsDisplay:YES];

    [dragImage release];
    
    // Only delete if the drag was accepted in THIS outlineView
    if (flags.justAcceptedDrag) {
        [[self dataSource] outlineView:self deleteItem:draggingSourceItem parentItem:parentItem];

        // NOW we can display again.
        [[self window] setAutodisplay:YES];
        [self setNeedsDisplay:YES];
    }
            
    [draggingSourceItem release];
    draggingSourceItem = nil; // Do this here so reloadData, below, draws correctly
    
    if (flags.justAcceptedDrag)
        [self reloadData];
    
    [self setSelectedItems:selectedItems];

    if ([[self dataSource] respondsToSelector:@selector(undoManagerForOutlineView:)]) {
        [[[self dataSource] undoManagerForOutlineView:self] setActionName:@"Drag Operation"];
        [[[self dataSource] undoManagerForOutlineView:self] endUndoGrouping];
    }

    flags.justAcceptedDrag = NO;
}

- (void)registerForDragging;
{
    if ([[self dataSource] respondsToSelector:@selector(outlineViewAcceptedPasteboardTypes:)])
        [self registerForDraggedTypes:[[self dataSource] outlineViewAcceptedPasteboardTypes:self]];
}

- (void)expandItemAndChildren:(id)item;
{
    if (item == nil || [[self dataSource] outlineView:self isItemExpandable:item]) {
        unsigned int childIndex, childCount;

        if (item != nil)
            [self expandItem:item];
    
        childCount = [[self dataSource] outlineView:self numberOfChildrenOfItem:item];
        for (childIndex = 0; childIndex < childCount; childIndex++)
            [self expandItemAndChildren:[[self dataSource] outlineView:self child:childIndex ofItem:item]];
    }
}

- (void)collapseItemAndChildren:(id)item;
{
    if (item == nil || [[self dataSource] outlineView:self isItemExpandable:item]) {
        unsigned int childIndex, childCount;

        childCount = [[self dataSource] outlineView:self numberOfChildrenOfItem:item];
        for (childIndex = 0; childIndex < childCount; childIndex++)
            [self collapseItemAndChildren:[[self dataSource] outlineView:self child:childIndex ofItem:item]];
            
        if (item != nil)
            [self collapseItem:item];
    }
}

@end


