// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAStackView.h>

#import <Foundation/NSArray.h>
#import <Foundation/NSException.h>
#import <OmniBase/OmniBase.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OAStackView.m,v 1.7 2000/07/10 20:48:13 bungi Exp $")


@interface OAStackView (PrivateAPI)
- (void) _loadSubviews;
- (void) _layoutSubviews;
@end

/*"
OAStackView assumes that all of its subviews line up in one direction (only vertical stacks are supported currently).  When a view is removed, the space is taken up by other views (currently the last view takes all the extra space) and the gap is removed by sliding adjacent views into that space.
"*/
@implementation OAStackView

//
// API
//

- (id) dataSource;
{
    return dataSource;
}

- (void) setDataSource: (id) aDataSource;
{
    dataSource = aDataSource;
    flags.needsReload = 1;

    // This is really a bug.  If we don't do this (not sure if the layout is necessary,
    // but the reload is), then the first window in OmniWeb will not show up (it gets an
    // exception down in the drawing code).  While it seems permissible to ask the data
    // source as soon as we have one, the data source might have some setup of its own
    // left to do.  This way, we force it to be valid immediately which could be bad, but
    // not much we can do with NSView putting the smack down on us.
    [self _loadSubviews];
    [self _layoutSubviews];
}

- (void) reloadSubviews;
{
    unsigned int index;
    
    index = [_subviews count];
    while (index--)
        [[_subviews objectAtIndex: index] removeFromSuperview];

    flags.needsReload = 1;
    flags.needsLayout = 1;
    
    // This won't happen if we didn't originally contain any views
    [self setNeedsDisplay: YES];
}

- (void) subviewSizeChanged;
{
    flags.needsLayout = 1;
    [self setNeedsDisplay: YES];
}

//
// NSView subclass
//

- (void) drawRect: (NSRect) rect;
{
    if (flags.needsReload)
        [self _loadSubviews];
    if (flags.needsLayout)
        [self _layoutSubviews];

    // This doesn't draw the subviews, we're just hooking the reset of
    // the subviews here since this should get done before they are drawn.
    [super drawRect: rect];
}

// This doesn't protect against having a subview removed, but some checking is better than none.
- (void) addSubview: (NSView *) view;
{
    [NSException raise: NSInternalInconsistencyException
                format: @"Do not add views directly to a OAStackView -- use the dataSource"];
}

@end

@implementation OAStackView (PrivateAPI)

- (void) _loadSubviews;
{
    NSArray *subviews;
    unsigned int subviewIndex, subviewCount;
    
    flags.needsReload = 0;
    flags.needsLayout = 1;
    
    subviews = [dataSource subviewsForStackView: self];
    subviewCount = [subviews count];
    for (subviewIndex = 0; subviewIndex < subviewCount; subviewIndex++) {
        NSView *view;
        
        // Get the view and set the autosizing flags correctly.  This will mean
        // that the layout will be correct when we get resized due to the normal
        // NSView resizing logic
        view = [subviews objectAtIndex: subviewIndex];
        [view setAutoresizingMask: NSViewWidthSizable | NSViewMinYMargin];
        
        // Call super to work around the -addSubview: check above
        [super addSubview: view];
    }
    
    // The last view has different resizing flags
    if (subviewCount)
        [[subviews objectAtIndex: subviewCount - 1] setAutoresizingMask: NSViewWidthSizable | NSViewHeightSizable];
}

- (void) _layoutSubviews;
{
    unsigned int viewIndex, viewCount;
    NSView *view;
    NSRect spaceLeft;
    NSRect subviewFrame;

    flags.needsLayout = 0;

    spaceLeft = [self bounds];
    //NSLog(@"total bounds = %@", NSStringFromRect(spaceLeft));
    
    viewCount = [_subviews count];
    for (viewIndex = 0; viewIndex < viewCount - 1; viewIndex++) {
        view = [_subviews objectAtIndex: viewIndex];
        subviewFrame = [view frame];
        //NSLog(@"  frame = %@", NSStringFromRect(subviewFrame));
        subviewFrame = NSMakeRect(NSMinX(spaceLeft),
                                  NSMaxY(spaceLeft) - NSHeight(subviewFrame),
                                  NSWidth(spaceLeft),
                                  NSHeight(subviewFrame));
        //NSLog(@"  new subviewFrame = %@", NSStringFromRect(subviewFrame));
        [view setFrame: subviewFrame];
        //NSLog(@"  view = %@", view);

        spaceLeft.size.height -= subviewFrame.size.height;
    }
    
    // The last view takes up the remaining space
    //NSLog(@"  spaceLeft = %@", NSStringFromRect(spaceLeft));
    [[_subviews objectAtIndex: viewCount - 1] setFrame: spaceLeft];
    //NSLog(@"  last view = %@", [_subviews objectAtIndex: viewCount - 1]);
    
    //NSLog(@"-[OAStackView layoutSubview]: %@", self);
}

@end

@implementation NSView (OAStackViewHelper)

- (OAStackView *) enclosingStackView;
{
    NSView *view;
    Class stackViewClass;
    
    view = [self superview];
    stackViewClass = [OAStackView class];
    
    while (view && ![view isKindOfClass: stackViewClass])
        view = [view superview];
        
    return (OAStackView *)view;
}

@end
