// Copyright 2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OATableView.h"

#import <OmniBase/OmniBase.h>
#import <AppKit/AppKit.h>

#import "NSString-OAExtensions.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OATableView.m,v 1.2 2000/07/14 13:02:57 kc Exp $")

@interface OATableView (Private)

- (void)startDrag:(NSEvent *)event;

@end

@implementation OATableView

// API

- (IBAction)copy:(id)sender;
{
    int selectedRow, selectedColumn;
    selectedRow = [self selectedRow];
    selectedColumn = [self selectedColumn];

    if (selectedRow >= 0 || selectedColumn >= 0) {
        if ([[self dataSource] respondsToSelector:@selector(tableView:copyObjectValueForTableColumn:row:toPasteboard:)]) {
            NSTableColumn *column;

            if (selectedColumn >= 0)
                column = [[self tableColumns] objectAtIndex:selectedColumn];
            else
                column = nil;
        
            [[self dataSource] tableView:self copyObjectValueForTableColumn:column row:selectedRow toPasteboard:[NSPasteboard generalPasteboard]];
        }
    }
}

// NSView

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent;
{
    return YES;
}

- (BOOL)shouldDelayWindowOrderingForEvent:(NSEvent *)anEvent 
{
    return YES;
}

// NSControl

- (void)mouseDown:(NSEvent *)event;
{
    NSPoint eventLocationInWindow, eventLocation;
    int columnIndex, rowIndex;
    NSRect slopRect;
    const int dragSlop = 4;
    NSEvent *mouseDragCurrentEvent;

    if (![[self dataSource] respondsToSelector:@selector(tableView:copyObjectValueForTableColumn:row:toPasteboard:)]) {
        [super mouseDown:event];
        return;
    }

    eventLocationInWindow = [event locationInWindow];
    eventLocation = [self convertPoint:eventLocationInWindow fromView:nil];
    columnIndex = [self columnAtPoint:eventLocation];
    rowIndex = [self rowAtPoint:eventLocation];
    if (rowIndex == -1 || columnIndex == -1)
        return;

    slopRect = NSInsetRect(NSMakeRect(eventLocationInWindow.x, eventLocationInWindow.y, 0.0, 0.0), -dragSlop, -dragSlop);
    while (1) {
        NSEvent *nextEvent;

        nextEvent = [NSApp nextEventMatchingMask:NSLeftMouseDraggedMask|NSLeftMouseUpMask untilDate:[NSDate distantFuture] inMode:NSEventTrackingRunLoopMode dequeue:NO];
        mouseDragCurrentEvent = nextEvent;

        if ([nextEvent type] == NSLeftMouseUp) {
            break;
        } else {
            [NSApp nextEventMatchingMask:NSLeftMouseDraggedMask untilDate:[NSDate distantFuture] inMode:NSEventTrackingRunLoopMode dequeue:YES];
            if (!NSMouseInRect([nextEvent locationInWindow], slopRect, NO)) {
                [self startDrag:event];
                return;
            }
        }
    }

    [super mouseDown:event];
}

// NSDraggingSource

- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)isLocal;
{
    if (isLocal)
        return NSDragOperationGeneric;
    else
        return NSDragOperationCopy;
}

@end


@implementation OATableView (Private)

- (void)startDrag:(NSEvent *)event;
{
    NSPoint eventLocation;
    int dragSourceRow;
    int selectedRow;
    NSRect rowFrame;
    double xOffsetOfFirstColumn;
    NSRect imageFrame;
    NSCachedImageRep *cachedImageRep;
    NSView *contentView;
    NSImage *dragImage;
    NSPasteboard *pasteboard;
    
    eventLocation = [self convertPoint:[event locationInWindow] fromView:nil];
    dragSourceRow = [self rowAtPoint:eventLocation];

    selectedRow = [self selectedRow];
    [self selectRow:dragSourceRow byExtendingSelection:NO];

    // Cache an image for the current row
    rowFrame = [self rectOfRow:dragSourceRow];
    xOffsetOfFirstColumn = [self frameOfCellAtColumn:0 row:dragSourceRow].origin.x;
    imageFrame = NSMakeRect(0, 0, NSWidth(rowFrame)-xOffsetOfFirstColumn, NSHeight(rowFrame));

    cachedImageRep = [[NSCachedImageRep alloc] initWithSize:imageFrame.size depth:[[NSScreen mainScreen] depth] separate:YES alpha:YES];
    contentView = [[cachedImageRep window] contentView];

    [contentView lockFocus];
    {
        int columnIndex, columnCount;
    
        [[NSColor colorWithDeviceWhite:0.0 alpha:0.0] set];
        NSRectFillUsingOperation(imageFrame, NSCompositeCopy);

        columnCount = [self numberOfColumns];
        for (columnIndex = 0; columnIndex < columnCount; columnIndex++) {
            NSTableColumn *tableColumn;
            NSCell *cell;
            NSRect cellRect;
            id objectValue;
            
            tableColumn = [[self tableColumns] objectAtIndex:columnIndex];
            objectValue = [[self dataSource] tableView:self objectValueForTableColumn:tableColumn row:dragSourceRow];

            cellRect = [self frameOfCellAtColumn:columnIndex row:dragSourceRow];
            cellRect.origin = NSMakePoint(NSMinX(cellRect) - xOffsetOfFirstColumn, 0);
            cell = [tableColumn dataCellForRow:dragSourceRow];
            
            if ([objectValue isKindOfClass:[NSString class]])
                [objectValue drawOutlinedWithFont:nil color:nil backgroundColor:nil rectangle:[cell titleRectForBounds:cellRect]];
            else {
                [cell setCellAttribute:NSCellHighlighted to:0];
                [cell setObjectValue:objectValue];
        
                [cell drawWithFrame:cellRect inView:contentView];
            }
        }
    }
    [contentView unlockFocus];

    dragImage = [[NSImage alloc] init];
    [dragImage addRepresentation:cachedImageRep];
    [cachedImageRep release];

    // Let's start the drag.
    pasteboard = [NSPasteboard pasteboardWithUniqueName];
    [[self dataSource] tableView:self copyObjectValueForTableColumn:[[self tableColumns] objectAtIndex:0] row:dragSourceRow toPasteboard:pasteboard];
    
    [self dragImage:dragImage at:NSMakePoint(NSMinX(rowFrame)+xOffsetOfFirstColumn, NSMaxY(rowFrame) - 1) offset:NSMakeSize(0, 0) event:event pasteboard:pasteboard source:self slideBack:YES];

    [dragImage release];
    
    [self selectRow:selectedRow byExtendingSelection:NO];
}


@end
