// Copyright 2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAWindowCascade.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OAWindowCascade.m,v 1.2 2000/07/06 06:06:47 kc Exp $")

// TJW: We should subscribe to screen parameter changes and either reset our cascadeRect, or invalidate ourselves if our screen is disabled.

#define WINDOW_TILE_STEP (15.0)

static CFStringRef DockApplicationID = (CFStringRef)@"com.apple.dock";

#define MENU_BAR_HEIGHT (22)
#define AVOID_INSET (8)

@implementation OAWindowCascade

- initWithScreen: (NSScreen *) aScreen;
{
    NSRect screenFrame, avoidRect;
    
    screen = [aScreen retain];
    cascadeRect = screenFrame = [screen frame];
    
    // If we are on the main screen, avoid the menu and dock
    if (screen == [NSScreen mainScreen]) {
        // We assume that the menu bar and dock appear on the main screen
        Boolean autohide, keyValid;
        
        // Exclude the menu bar
        avoidRect.origin.x = screenFrame.origin.x;
        avoidRect.origin.y = NSMaxY(screenFrame) - MENU_BAR_HEIGHT;
        avoidRect.size.width = screenFrame.size.width;
        avoidRect.size.height = MENU_BAR_HEIGHT;
        cascadeRect = OFLargestRectAvoidingRect(cascadeRect, avoidRect);
        
        // Exclude the dock if it is always exposed.
        autohide = CFPreferencesGetAppBooleanValue(CFSTR("autohide"), DockApplicationID, &keyValid);
        if (!autohide) {
            CFPropertyListRef heightPlist;
            float height = 0.0;
            
            heightPlist = CFPreferencesCopyAppValue(CFSTR("tilesize"), DockApplicationID);
            if (heightPlist) {
                if (CFGetTypeID(heightPlist) == CFNumberGetTypeID())
                    CFNumberGetValue(heightPlist, kCFNumberFloat32Type, &height);
                CFRelease(heightPlist);
            }

            if (height > 0.0) {
                avoidRect.origin = screenFrame.origin;
                avoidRect.size.width = screenFrame.size.width;
                avoidRect.size.height = height;
                cascadeRect = OFLargestRectAvoidingRect(cascadeRect, avoidRect);
            }
        }
    }
    
    return self;
}

- (void) dealloc;
{
    [screen release];
    [super dealloc];
}

- (NSScreen *) screen;
{
    return screen;
}

- (NSRect) nextWindowFrameForSize: (NSSize) windowSize
                  avoidingWindows: (NSArray *) windowsToAvoid;
{
    NSRect candidate;
    NSRect avoidRect, availableRect;
    unsigned int windowIndex;
    NSWindow *window;
    NSPoint topLeft;
    
    // Trim the cascade rect down based on the windows to avoid
    windowIndex = [windowsToAvoid count];
    availableRect = cascadeRect;
    while (windowIndex--) {
        window = [windowsToAvoid objectAtIndex: windowIndex];
        if ([window screen] != screen)
            continue;
        avoidRect = [window frame];
        availableRect = OFLargestRectAvoidingRect(availableRect, avoidRect);
    }
    
    // If we can't avoid them all, to hell with even trying to avoid any
    if (NSIsEmptyRect(availableRect))
        availableRect = cascadeRect;
    
    // Don't butt exactly adjacent to stuff we are avoiding
    availableRect = NSInsetRect(availableRect, AVOID_INSET, AVOID_INSET); 
    
    // Tile inside the available rect.  Two calls to this function across
    // movement of the windows to avoid might produce discontinuous tilings.
    // That should be pretty rare, though.
retry:
    topLeft.x = NSMinX(availableRect) + step * WINDOW_TILE_STEP;
    topLeft.y = NSMaxY(availableRect) - step * WINDOW_TILE_STEP;
    candidate.origin.x = topLeft.x;
    candidate.origin.y = topLeft.y - windowSize.height;
    candidate.size = windowSize;
    
    // If it doesn't fit, reset the cascade step and retry
    if (step && !NSContainsRect(cascadeRect, candidate)) {
        step = 0;
        goto retry;
    }
    
    step++;
    return candidate;
}

@end
