// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAApplication.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import "NSView-OAExtensions.h"
#import "OAAppkitQueueProcessor.h"
#import "OAScrollView.h" // For OAScrollViewPageDownPercentage

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/OAApplication.m,v 1.23 2000/10/24 16:39:06 kc Exp $")

@interface OAApplication (Private)
- (void)processMouseButtonsChangedEvent:(NSEvent *)event;
@end

@implementation OAApplication

+ (void)setupOmniApplication;
{
    [OBPostLoader processClasses];
}

+ (NSApplication *)sharedApplication;
{
    static OAApplication *omniApplication = nil;

    if (omniApplication)
        return omniApplication;

    omniApplication = (id)[super sharedApplication];
    [OAApplication setupOmniApplication];
    return omniApplication;
}

- (void)run;
{
    BOOL stillRunning = YES;

    exceptionCount = 0;
    exceptionCheckpointDate = [[NSDate alloc] init];
    do {
        NS_DURING {
            [super run];
            stillRunning = NO;
        } NS_HANDLER {
            if (++exceptionCount % 300 == 0) {
                if ([exceptionCheckpointDate timeIntervalSinceNow] > -3.0)
                    stillRunning = NO; // 300 exceptions in 3 seconds: abort
                [exceptionCheckpointDate release];
                exceptionCheckpointDate = [[NSDate alloc] init];
                exceptionCount = 0;
            }
            if (localException) {
                if (_appFlags._hasBeenRun)
                    [self handleRunException:localException];
                else
                    [self handleInitException:localException];
            }
        } NS_ENDHANDLER;
    } while (stillRunning && _appFlags._hasBeenRun);
}

#define OASystemDefinedEvent_MouseButtonsChangedSubType 7

- (void)sendEvent:(NSEvent *)event;
{
    // The -timestamp method on NSEvent doesn't seem to return an NSTimeInterval based off the same reference date as NSDate (which is what we want).
    lastEventTimeInterval = [NSDate timeIntervalSinceReferenceDate];

    NS_DURING {
        if ([event type] == NSSystemDefined && [event subtype] == OASystemDefinedEvent_MouseButtonsChangedSubType) {
            [self processMouseButtonsChangedEvent:event];
            [super sendEvent:event];
        } else if ([event type] == NSScrollWheel) {
            if ([self scrollWheelButtonIsDown]) {
                NSView *contentView;
                NSView *viewUnderMouse;

                contentView = [[event window] contentView];
                viewUnderMouse = [contentView hitTest:[contentView convertPoint:[event locationInWindow] fromView:nil]];
                [[viewUnderMouse enclosingScrollView] scrollDownByPercentage:[event deltaY] * -OAScrollViewPageDownPercentage];
            } else {
                [super sendEvent:event];
            }
        } else {
            [super sendEvent:event];
        }
    } NS_HANDLER {
        if ([[localException name] isEqualToString:NSAbortModalException] || [[localException name] isEqualToString:NSAbortPrintingException])
            [localException raise];
        [self handleRunException:localException];
    } NS_ENDHANDLER;
}

- (void)handleInitException:(NSException *)anException;
{
    id delegate;

    delegate = [self delegate];
    if ([delegate respondsToSelector:@selector(handleInitException:)]) {
        [delegate handleInitException:anException];
    } else {
        NSLog(@"%@", [anException reason]);
    }
}

- (void)handleRunException:(NSException *)anException;
{
    id delegate;

    delegate = [self delegate];
    if ([delegate respondsToSelector:@selector(handleRunException:)]) {
        [delegate handleRunException:anException];
    } else {
        NSLog(@"%@", [anException reason]);
        NSRunAlertPanel(nil, @"%@", nil, nil, nil, [anException reason]);
    }
}

- (NSTimeInterval)lastEventTimeInterval;
{
    return lastEventTimeInterval;
}

- (BOOL)mouseButtonIsDownAtIndex:(unsigned int)mouseButtonIndex;
{
    return (mouseButtonState & (1 << mouseButtonIndex)) != 0;
}

- (BOOL)scrollWheelButtonIsDown;
{
    return [self mouseButtonIsDownAtIndex:2];
}

@end

@implementation OAApplication (Private)

- (void)processMouseButtonsChangedEvent:(NSEvent *)event;
{
    mouseButtonState = [event data2];
}

@end
