// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAPreferenceController.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OmniAppKit/OAPreferenceClient.h>
#import <OmniAppKit/NSView-OAExtensions.h>

#import <OmniAppKit/OAPreferenceCategory.h>
#import <OmniAppKit/OAPreferenceClientRecord.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Preferences.subproj/OAPreferenceController.m,v 1.31 2000/08/13 21:11:13 bungi Exp $")

@interface OAPreferenceController (Private)
- (void)loadInterface;
- (void)resetWindowTitle;
- (OAPreferenceCategory *) categoryNamed: (NSString *) name;
- (void)setCurrentClient:(OAPreferenceClientRecord *)clientRecord;
@end


@implementation OAPreferenceController

static NSString *windowFrameSaveName = @"Preferences";
static OAPreferenceController *sharedPreferenceController = nil;


+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
        return;
    initialized = YES;
}

// OFBundleRegistryTarget informal protocol

+ (void)registerItemName:(NSString *)itemName bundle:(NSBundle *)bundle description:(NSDictionary *)description;
{
    NSString *categoryName;
    OAPreferenceCategory *category;
    
    [OFBundledClass createBundledClassWithName:itemName bundle:bundle description:description];

    if (!(categoryName = [description objectForKey:@"category"]))
        categoryName = @"UNKNOWN";

    category = [[self sharedPreferenceController] categoryNamed: categoryName];
    [category registerClassName:itemName description: description];
}


// Init and dealloc

+ (OAPreferenceController *)sharedPreferenceController;
{
    if (!sharedPreferenceController)
        sharedPreferenceController = [[self alloc] init];
    
    return sharedPreferenceController;
}

- init;
{
    [super init];
	
    categories = [[NSMutableArray alloc] init];
    
    // This is lame.  We should really think up some way to specify the ordering
    // of preferences in the plists.  But this is difficult since preferences
    // can come from many places.  For now, we pre-create the categories we know
    // we are going to use, in the order we want them listed in the tab view.
    //[self categoryNamed: @"User"];
    //[self categoryNamed: @"Administrator"];
    
    return self;
}

- (void)dealloc;
{
    OBPRECONDITION(NO);
}


// API

- (void)selectCategory: (OAPreferenceCategory *) category;
{
    NSArray *clientRecords;
    unsigned int clientIndex, clientCount;
    
    if (category == nonretained_currentCategory)
        return;
    nonretained_currentCategory = category;
    
    if ([[categoryTabView selectedTabViewItem] identifier] != category)
        [categoryTabView selectTabViewItemWithIdentifier: category];
        
    clientRecords = [nonretained_currentCategory clientRecords];
    [preferencesMatrix renewRows: [clientRecords count] columns: 1];
    
    clientCount = [clientRecords count];
    
    for (clientIndex = 0; clientIndex < clientCount; clientIndex++) {
        NSButtonCell *cell;
        OAPreferenceClientRecord *clientRecord;
        
        clientRecord = [clientRecords objectAtIndex: clientIndex];
        cell = [preferencesMatrix cellAtRow: clientIndex column: 0];
        [cell setImage: [clientRecord iconImage]];
        [cell setTitle: [clientRecord shortTitle]];
        [cell setRepresentedObject: clientRecord];
    }

#if USE_SMART_PREF_TABS
    // If there's only one preference client registered, don't show the scrolling preference matrix
    if (clientCount <= 1) {
        [matrixBox morphToFrame:NSMakeRect([matrixBox frame].origin.x, [matrixBox frame].origin.y, 0, [matrixBox frame].size.height) overTimeInterval: 0.1];
        [box morphToFrame:NSMakeRect([matrixBox frame].origin.x, [box frame].origin.y, oldBoxFrame.size.width + oldMatrixFrame.size.width + 11, [box frame].size.height) overTimeInterval: 0.1];
        [box display];
    } else {
        NSView *contentView, *oldView;
        
        // Remove old client box
        contentView = [box contentView];
        oldView = [[contentView subviews] lastObject];
        [oldView fadeOutAndRemoveFromSuperviewOverTimeInterval: 0.075];
        OBASSERT([[contentView subviews] count] == 0);
        
        //show scrolling matrix and shrink client box
        [box morphToFrame:oldBoxFrame overTimeInterval: 0.1];
        [matrixBox morphToFrame:oldMatrixFrame overTimeInterval: 0.075];
    }
#endif
    
    preferredBoxSize = [[box contentView] bounds].size;
    windowExtraSize = [window frame].size;
    windowExtraSize.width -= preferredBoxSize.width;
    windowExtraSize.height -= preferredBoxSize.height;
    
    [preferencesMatrix sizeToCells];
    [preferencesMatrix setNeedsDisplay: YES];

    // Select the first entry in this category
    clientRecords = [nonretained_currentCategory clientRecords];
    if ([clientRecords count])
        [self setCurrentClient:[clientRecords objectAtIndex:0]];
}

- (void)close;
{
    if ([window isVisible])
        [window performClose:nil];
}

- (void)setTitle: (NSString *) title;
{
    [window setTitle: title];
}

- (void)setCurrentClientByName: (NSString *) name;
{
    unsigned int categoryIndex, clientRecordIndex;
    OAPreferenceCategory *category;
    OAPreferenceClientRecord *clientRecord;
    NSArray *clientRecords;
    
    categoryIndex = [categories count];
    while (categoryIndex--) {
        category = [categories objectAtIndex: categoryIndex];
        clientRecords = [category clientRecords];
        clientRecordIndex = [clientRecords count];
        while (clientRecordIndex--) {
            clientRecord = [clientRecords objectAtIndex: clientRecordIndex];
            if ([[clientRecord className] isEqualToString: name]) {
                if (nonretained_currentCategory != category)
                    [self selectCategory: category];
                [self setCurrentClient: clientRecord];
                return;
            }
        }
    }
}

// Actions

- (IBAction)showPreferencesPanel:(id)sender;
{
    [self loadInterface];
    [self resetWindowTitle];
    
    // Let the current client know that it is about to be displayed.
    [nonretained_currentClient becomeCurrentPreferenceClient];
    [window makeKeyAndOrderFront:sender];
}

- (IBAction)selectPreference:(NSTableView *)sender;
{
    [self setCurrentClient:[[preferencesMatrix selectedCell] representedObject]];
}

- (IBAction)restoreDefaults:(id)sender;
{
    [nonretained_currentClient restoreDefaults:sender];
}

// NSTabView delegate

- (void)tabView:(NSTabView *)tabView didSelectTabViewItem:(NSTabViewItem *)tabViewItem;
{
    [self selectCategory: [tabViewItem identifier]];
}

// NSWindow delegate

- (void)windowWillClose:(NSNotification *)notification;
{
    [[notification object] makeFirstResponder:nil];
    [nonretained_currentClient resignCurrentPreferenceClient];
}

- (void)windowDidResignKey:(NSNotification *)notification;
{
    [[notification object] makeFirstResponder:nil];
}

@end


@implementation OAPreferenceController (Private)

- (void)loadInterface;
{
    unsigned int categoryIndex, categoryCount;
    
    if (window)
        return;

    [NSBundle loadNibNamed:@"OAPreferences.nib" owner:self];

    if (![window setFrameUsingName:windowFrameSaveName])
        [window center];
    [window setFrameAutosaveName:windowFrameSaveName];
    
    categoryCount = [categories count];
    
#if USE_SMART_PREF_TABS
    //get rid of the tabs if we only have one category
    if (categoryCount <= 1) {
        [box setFrame:NSMakeRect([box frame].origin.x, [box frame].origin.y, [box frame].size.width, [box frame].size.height + [categoryTabView frame].size.height - 5)];
        [matrixBox setFrame:NSMakeRect([matrixBox frame].origin.x, [matrixBox frame].origin.y, [matrixBox frame].size.width, [matrixBox frame].size.height + [categoryTabView frame].size.height - 5)];
        [categoryTabView setFrame:NSMakeRect([box frame].origin.x, [box frame].origin.y, 0, 0)];
    }
#endif
    
    //save scrolly matrix's size
    oldMatrixFrame = [matrixBox frame];
    oldBoxFrame = [box frame];
    
    preferredBoxSize = [[box contentView] bounds].size;
    windowExtraSize = [window frame].size;
    windowExtraSize.width -= preferredBoxSize.width;
    windowExtraSize.height -= preferredBoxSize.height;
    
    if (categoryCount) {
        while ([categoryTabView numberOfTabViewItems])
            [categoryTabView removeTabViewItem: [categoryTabView tabViewItemAtIndex: 0]];
            
        for (categoryIndex = 0; categoryIndex < categoryCount; categoryIndex++) {
            NSTabViewItem *item;
            OAPreferenceCategory *category;
            
            category = [categories objectAtIndex:[categories count] - categoryIndex - 1];
            item = [[NSTabViewItem alloc] initWithIdentifier: category];
            [item setLabel: [category name]];
            [categoryTabView addTabViewItem:item];
            [item release];
        }
    }
    

    // Select the first category
    if ([categories count])
        [self selectCategory: [[categoryTabView tabViewItemAtIndex:0] identifier]];
}

- (void)resetWindowTitle;
{
    NSString *name;
    
    name = [nonretained_currentClientRecord title];
    if ([name length] > 0)
        [window setTitle:[name stringByAppendingString:@" Preferences"]];
    else
        [window setTitle:@"Preferences"];
}

- (OAPreferenceCategory *) categoryNamed: (NSString *) name;
{
    unsigned int categoryIndex;
    OAPreferenceCategory *category;
    
    categoryIndex = [categories count];
    while (categoryIndex--) {
        category = [categories objectAtIndex: categoryIndex];
        if ([name isEqualToString: [category name]])
            return category;
    }
    
    category = [[OAPreferenceCategory alloc] initWithName: name];
    [categories addObject: category];
    [category release];
    
    return category;
}

// Setting the current preference client

- (void)setCurrentClient:(OAPreferenceClientRecord *)clientRecord
{
    unsigned int clientIndex;
    NSView *contentView, *controlBox;
    NSRect newBoxFrame, windowFrame;
    NSSize newWindowSize;
    NSView *oldView;

    if (nonretained_currentClientRecord == clientRecord)
        return;
        
    // Save changes in any editing text fields
    [window setInitialFirstResponder:nil];
    [window makeFirstResponder:nil];
    
    // Only do this when we are on screen to avoid sending become/resign twice.
    // If we are off screen, the client got resigned when it went off and the
    // new one will get a become when it goes on screen.
    if ([window isVisible])
        [nonretained_currentClient resignCurrentPreferenceClient];
        
    nonretained_currentClientRecord = clientRecord;
    nonretained_currentClient = [clientRecord clientInstanceInController: self];
    
    [self resetWindowTitle];
    
    clientIndex = [[nonretained_currentCategory clientRecords] indexOfObjectIdenticalTo:nonretained_currentClientRecord];
    if (clientIndex != NSNotFound)
        [preferencesMatrix scrollCellToVisibleAtRow: clientIndex column: 0];
    
    
    // Remove old client box
    contentView = [box contentView];
    oldView = [[contentView subviews] lastObject];
    [oldView fadeOutAndRemoveFromSuperviewOverTimeInterval: 0.08];
    OBASSERT([[contentView subviews] count] == 0);

    // Compute the new window size for the new client box
    controlBox = [nonretained_currentClient controlBox];
    newBoxFrame.size = [controlBox frame].size;
    if (newBoxFrame.size.width < preferredBoxSize.width)
        newBoxFrame.size.width = preferredBoxSize.width;
    if (newBoxFrame.size.height < preferredBoxSize.height)
        newBoxFrame.size.height = preferredBoxSize.height;
    newWindowSize.width = newBoxFrame.size.width + windowExtraSize.width;
    newWindowSize.height = newBoxFrame.size.height + windowExtraSize.height;
    
    // Resize the window
    // We don't just tell the window to resize, because that tends to move the upper left corner (which will confuse the user)
    windowFrame = [window frame];
    windowFrame.origin.y += windowFrame.size.height - newWindowSize.height;
    windowFrame.size = newWindowSize;
    [window setFrame:windowFrame display:NO];
    
    // As above, don't do this unless we are onscreen to avoid double become/resigns.
    // Do this before putting the view in the view heirarchy to avoid flashiness
    // in the controls.
    if ([window isVisible])
        [nonretained_currentClient becomeCurrentPreferenceClient];
    [nonretained_currentClient updateUI];

    // Add the new client box to the view hierarchy
    newBoxFrame.origin.x = 0;
    newBoxFrame.origin.y = 0;
    [controlBox setFrame:newBoxFrame];
    [contentView fadeInSubview: controlBox overTimeInterval: 0.08];

    // Highlight the initial first responder, and also tell the window what it should be because I think there is some voodoo with nextKeyView not working unless the window has an initial first responder.
    [window setInitialFirstResponder:[nonretained_currentClient initialFirstResponder]];
    [window makeFirstResponder:[nonretained_currentClient initialFirstResponder]];
}

@end
