// Copyright 2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OAExtendedOutlineView.h"

#import <OmniBase/OmniBase.h>
#import <AppKit/AppKit.h>

#import "NSImage-OAExtensions.h"
#import "NSOutlineView-OAExtensions.h"
#import "NSString-OAExtensions.h"
#import "NSView-OAExtensions.h"
#import "OAInspector.h"
#import "OAOutlineViewEnumerator.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OAExtendedOutlineView.m,v 1.49 2000/10/31 23:05:54 bungi Exp $")

@interface NSOutlineView (PrivateAPIThatWeShouldNotHaveToCallDammit)
- (void)_collapseItem: (id) item collapseChildren:(BOOL)collapseChildren clearExpandState: (BOOL)clearExpandState;
@end

@interface OAExtendedOutlineView (Private)
- (void)startDrag:(NSEvent *)event;
- (void)registerForDragging;
- (void)expandItemAndChildren:(id)item;
- (void)collapseItemAndChildren:(id)item;
@end

#define DISCLOSURE_TRIANGLE_WIDTH 17.0
#define TRANSITION_TIME (0.25)

@implementation OAExtendedOutlineView

- (id)initWithFrame:(NSRect)rect;
{
    if (![super initWithFrame:rect])
        return nil;

    shouldEditNextItemWhenEditingEnds = YES;
// TJW -- Still debugging this    
//    flags.smoothExpandCollapse = YES;
    
    return self;
}

- initWithCoder:(NSCoder *)coder;
{
    if (!(self = [super initWithCoder:coder]))
        return nil;

    shouldEditNextItemWhenEditingEnds = YES;
// TJW -- Still debugging this    
//    flags.smoothExpandCollapse = YES;
    
    return self;
}

- (void)dealloc;
{
    [autoExpandedItems release];
    [self unregisterDraggedTypes];
    [super dealloc];
}

// API

- (id)parentItemForRow:(int)row child:(unsigned int *)childIndexPointer;
{
    int originalLevel;
    
    originalLevel = [self levelForRow:row];
    return [self parentItemForRow:row indentLevel:originalLevel child:childIndexPointer];
}

- (id)parentItemForRow:(int)row indentLevel:(int)childLevel child:(unsigned int *)childIndexPointer;
{
    unsigned int childIndex;

    childIndex = 0;

    while (row-- >= 0) {
        int currentLevel;
        
        currentLevel = [self levelForRow:row];
        if (currentLevel < childLevel) {
            if (childIndexPointer)
                *childIndexPointer = childIndex;
            return [self itemAtRow:row];
        } else if (currentLevel == childLevel)
            childIndex++;
    }
    if (childIndexPointer)
        *childIndexPointer = childIndex;
    return nil;
}

- (BOOL)shouldEditNextItemWhenEditingEnds;
{
    return shouldEditNextItemWhenEditingEnds;
}

- (void)setShouldEditNextItemWhenEditingEnds:(BOOL)value;
{
    shouldEditNextItemWhenEditingEnds = value;
}

- (void) setSmoothExpandCollapseEnabled: (BOOL) yn;
{
    flags.smoothExpandCollapse = yn;
}

- (BOOL) smoothExpandCollapseEnabled;
{
    return flags.smoothExpandCollapse;
}

- (void) autoExpandItems: (NSArray *) items;
{
    unsigned int itemIndex, itemCount;
    id item;
    
    // First, close any previously auto-expanded items
    itemCount = [autoExpandedItems count];
    for (itemIndex = 0; itemIndex < itemCount; itemIndex++) {
        item = [autoExpandedItems objectAtIndex: itemIndex];
        [self collapseItem: item];
    }
    
    // Then, open up the new items and remember them to auto close later.
    itemCount = [items count];
    [autoExpandedItems release];
    autoExpandedItems = [[NSMutableArray alloc] initWithCapacity: itemCount];

    for (itemIndex = 0; itemIndex < itemCount; itemIndex++) {
        item = [items objectAtIndex: itemIndex];
        if ([self isExpandable: item] && ![self isItemExpanded: item]) {
            [self expandItem: item];
            [(NSMutableArray *)autoExpandedItems addObject: item];
        }
    }
}

- (float)rowOffset:(int)row;
{
    return 0;
}


// Actions

- (IBAction)expandAll:(id)sender;
{
    NSArray *selectedItems;

    selectedItems = [self selectedItems];
    [self expandItemAndChildren:nil];
    [self setSelectedItems:selectedItems];
}

- (IBAction)contractAll:(id)sender;
{
    NSArray *selectedItems;

    selectedItems = [self selectedItems];
    [self collapseItemAndChildren:nil];
    [self setSelectedItems:selectedItems];
}

- (IBAction)expandSelection:(id)sender;
{
    NSArray *selectedItems;
    NSEnumerator *itemEnumerator;
    id item;
        
    selectedItems = [self selectedItems];
    itemEnumerator = [selectedItems objectEnumerator];
    while ((item = [itemEnumerator nextObject]))
        [self expandItemAndChildren:item];
    
    [self setSelectedItems:selectedItems];
}

- (IBAction)contractSelection:(id)sender;
{
    NSArray *selectedItems;
    NSEnumerator *itemEnumerator;
    id item;
    
    selectedItems = [self selectedItems];
    itemEnumerator = [selectedItems reverseObjectEnumerator];
    while ((item = [itemEnumerator nextObject]))
        [self collapseItemAndChildren:item];
    
    [self setSelectedItems:selectedItems];
}

// Just like NSText

- (IBAction)copy:(id)sender;
{
    if ([_dataSource respondsToSelector:@selector(outlineView:copyItem:toPasteboard:)])
        [_dataSource outlineView:self copyItem:[self selectedItem] toPasteboard:[NSPasteboard generalPasteboard]];
}

- (IBAction)cut:(id)sender;
{
    [self copy:sender];
    [self delete:sender];

    if ([_dataSource respondsToSelector:@selector(undoManagerForOutlineView:)])
        [[_dataSource undoManagerForOutlineView:self] setActionName:@"Cut"];
}

- (IBAction)delete:(id)sender;
{
    if ([_dataSource respondsToSelector:@selector(outlineView:deleteItem:parentItem:)]) {
        id selectedItem;
        int selectedRow;    
        int numberOfRows;
            
        selectedItem = [self selectedItem];
        selectedRow = [self selectedRow];
        
        [_dataSource outlineView:self deleteItem:selectedItem parentItem:[self parentItemForRow:[self selectedRow] child:NULL]];
        [self reloadData];
        
        // Maintain the selection after deletions
        numberOfRows = [self numberOfRows];
        if (selectedRow > (numberOfRows - 1))
            [self selectRow:(numberOfRows - 1) byExtendingSelection:NO];
        else
            [self selectRow:selectedRow byExtendingSelection:NO];
    }
}

- (IBAction)paste:(id)sender;
{
    if ([_dataSource respondsToSelector:@selector(outlineView:pasteItemFromPasteboard:parentItem:child:)]) {
        int selectedRow, numberOfRows;
        id parentItem;
        int childIndex;
    
        numberOfRows = [self numberOfRows];
        selectedRow = [self selectedRow];
        if (numberOfRows == 0 || selectedRow == -1) {
            parentItem = nil;
            childIndex = 0;
        } else {
            parentItem = [self parentItemForRow:selectedRow child:&childIndex];
            childIndex++; // Paste after current line
        }
            
        [_dataSource outlineView:self pasteItemFromPasteboard:[NSPasteboard generalPasteboard] parentItem:parentItem child:childIndex];
        [self reloadData];
    }
}


// NSResponder

- (BOOL)becomeFirstResponder;
{
    BOOL willBecome;

    willBecome = [super becomeFirstResponder];
    if (willBecome)
        [[NSNotificationCenter defaultCenter] postNotificationName:OAInspectorSelectionDidChangeNotification object:self];
    return willBecome;
}

- (BOOL)resignFirstResponder;
{
    BOOL willResign;

    willResign = [super resignFirstResponder];
    if (willResign)
        [[NSNotificationCenter defaultCenter] postNotificationName:OAInspectorSelectionDidChangeNotification object:self];
    return willResign;
}

- (void)keyDown:(NSEvent *)theEvent;
{
    NSString *characters;
    unsigned int modifierFlags;

    characters = [theEvent characters];
    modifierFlags = [theEvent modifierFlags];

    if ([characters hasPrefix:@"e"]) {
        NSTableColumn *column;
        int row;
        id item;

        // Mmmm, arbitrary constants
        column = [[self tableColumns] objectAtIndex:0];
        row = [self selectedRow];
        item = [self itemAtRow:row];
        if ([column isEditable] && [[self delegate] outlineView:self shouldEditTableColumn:column item:item])
            [self editColumn:0 row:row withEvent:nil select:YES];
    } else if ([characters hasPrefix:@"<"]) {
        [self contractAll:nil];
        [self selectRow:0 byExtendingSelection:NO];
    } else if ([characters hasPrefix:@">"]) {
        id selectedItem;
        int row;
        
        selectedItem = [self itemAtRow:[self selectedRow]];
        [self expandAll:nil];

        if ((row = [self rowForItem:selectedItem]) >= 0)
            [self selectRow:row byExtendingSelection:NO];
    } else if ([characters characterAtIndex:0] == NSLeftArrowFunctionKey && modifierFlags & NSShiftKeyMask)
        [self contractSelection:nil];
    else if ([characters characterAtIndex:0] == NSRightArrowFunctionKey && modifierFlags & NSShiftKeyMask)
        [self expandSelection:nil];
    else
        [self interpretKeyEvents:[NSArray arrayWithObject:theEvent]];
}

- (void)deleteForward:(id)sender;
{
    if ([self selectedItem])
        [self delete:nil];
}

- (void)deleteBackward:(id)sender;
{
    if ([self selectedItem])
        [self delete:nil];
}

- (void)insertTab:(id)sender;
{
    id selectedItem;
    int selectedRow;
    int levelOfSelectedItem;
    int currentRow;
    
    // We can't do this if they don't implement outlineView:parentItem:moveChildren:toNewParentItem
    if ([_dataSource respondsToSelector:@selector(outlineView:parentItem:moveChildren:toNewParentItem:)] == NO)
        return;
    
    selectedRow = [self selectedRow];
    if (selectedRow == 0 || selectedRow == -1)
        return;
        
    selectedItem = [self itemAtRow:selectedRow];
    levelOfSelectedItem = [self levelForItem:selectedItem];
    
    currentRow = selectedRow - 1;
    while (currentRow >= 0) {
        id potentialParent;
        int levelOfPotentialParent;
        
        potentialParent = [self itemAtRow:currentRow];
        levelOfPotentialParent = [self levelForItem:potentialParent];
        
        if (levelOfPotentialParent == levelOfSelectedItem) {
            NSArray *movingChildren;
            id previousParent;
            unsigned int childIndex;
            
            movingChildren = [NSArray arrayWithObject:selectedItem];
            previousParent = [self parentItemForRow:selectedRow child:&childIndex];
    
            // If you're the zeroth child of your parent, you cannot be indented any more.
            if (childIndex == 0)
                return;

            [_dataSource outlineView:self parentItem:previousParent moveChildren:movingChildren toNewParentItem:potentialParent];
            [self expandItem:potentialParent];

            // Reload those items which were affected
            if (previousParent == nil) {
                [self reloadData];
            } else {
                [self reloadItem:previousParent reloadChildren:YES];
                [self reloadItem:potentialParent reloadChildren:YES];
            }
            
            [self selectRow:[self rowForItem:selectedItem] byExtendingSelection:NO];

            break;
        }
        
        // Move upwards through the list
        currentRow--;
    }

    if ([_dataSource respondsToSelector:@selector(undoManagerForOutlineView:)])
        [[_dataSource undoManagerForOutlineView:self] setActionName:@"Indent"];
}

- (void)insertBacktab:(id)sender;
{
    NSMutableArray *peersAfterSelectedItem;
    id selectedItem;
    id parent;
    id parentsParent;
    int selectedRow;
    int levelOfSelectedItem;
    int peerIndex;
    int parentChildrenCount;
    int parentIndex;
    
    // Determine if the dataSource supports the required extension methods for this feature, and if so, determine if this operation is valid for the current selection
    if ([_dataSource respondsToSelector:@selector(outlineView:parentItem:moveChildren:toNewParentItem:beforeIndex:)] == NO)
        return;
    
    selectedRow = [self selectedRow];
    selectedItem = [self itemAtRow:selectedRow];
    levelOfSelectedItem = [self levelForItem:selectedItem];

    if (selectedRow == -1 || selectedRow == 0 || levelOfSelectedItem == 0)
        return;

    parent = [self parentItemForRow:selectedRow child:&peerIndex];
    parentChildrenCount = [_dataSource outlineView:self numberOfChildrenOfItem:parent];
    peersAfterSelectedItem = [NSMutableArray array];
    
    // The peers of the selection which come after it will become children of the selected item
    for (peerIndex += 1; peerIndex < parentChildrenCount; peerIndex++) {
        id peer;
        
        peer = [_dataSource outlineView:self child:peerIndex ofItem:parent];
        [peersAfterSelectedItem addObject:peer];
    }

    // If there were any peers after the selection, move them to the end of the selected item's list of children
    if ([peersAfterSelectedItem count] > 0) {
        [_dataSource outlineView:self parentItem:parent moveChildren:peersAfterSelectedItem toNewParentItem:selectedItem beforeIndex:-1];
    
        // Make sure the selected item is expanded
        if ([_dataSource outlineView:self isItemExpandable:selectedItem])
            [self expandItem:selectedItem];
    }

    parentsParent = [self parentItemForRow:[self rowForItem:parent] child:&parentIndex];

    // Make the selection become a peer of its parent
    [_dataSource outlineView:self parentItem:parent moveChildren:[NSArray arrayWithObject:selectedItem] toNewParentItem:parentsParent beforeIndex:(parentIndex + 1)];
    
    [self reloadData];

    // Reselect row since, in some situations, NSOutlineView gets confused with regards to what should be selected
    [self selectRow:[self rowForItem:selectedItem] byExtendingSelection:NO];

    if ([_dataSource respondsToSelector:@selector(undoManagerForOutlineView:)])
        [[_dataSource undoManagerForOutlineView:self] setActionName:@"Unindent"];
}

- (void)insertNewline:(id)sender;
{
    if ([_dataSource respondsToSelector:@selector(outlineView:createNewItemAsChild:ofItem:)]) {
        int selectedRow, numberOfRows;
        id parentItem;
        int childIndex;
    
        numberOfRows = [self numberOfRows];
        selectedRow = [self selectedRow];
        if (numberOfRows == 0 || selectedRow == -1) {
            parentItem = nil;
            childIndex = 0;
        } else {
            parentItem = [self parentItemForRow:selectedRow child:&childIndex];
            childIndex++; // Insert after current line
        }

        if ([_dataSource outlineView:self createNewItemAsChild:childIndex ofItem:parentItem]) {
            id item;
            int newItemRow;
            NSArray *tableColumns;
            int columnIndex, columnCount;
        
            [self reloadData];
            item = [_dataSource outlineView:self child:childIndex ofItem:parentItem];
            newItemRow = [self rowForItem:item];
            [self selectRow:newItemRow byExtendingSelection:NO];
            
            // Edit the first editable column            
            tableColumns = [self tableColumns];
            columnCount = [tableColumns count];
            for (columnIndex = 0; columnIndex < columnCount; columnIndex++) {
                NSTableColumn *column;
                
                column = [tableColumns objectAtIndex:columnIndex];
                if ([column isEditable] && [[self delegate] outlineView:self shouldEditTableColumn:column item:item]) {
                    [self editColumn:columnIndex row:newItemRow withEvent:nil select:YES];
                    break;
                }
            }
        }
    }
}

- (void)moveUp:(id)sender;
{
    int selectedRow;
    
    selectedRow = [self selectedRow];
    if (selectedRow == 0 || selectedRow == -1)
        return;
        
    [self selectRow:--selectedRow byExtendingSelection:NO];
    [self scrollRowToVisible:selectedRow];
}

- (void)moveDown:(id)sender;
{
    int selectedRow;
    
    selectedRow = [self selectedRow];
    if (selectedRow == -1 || (selectedRow >= [self numberOfRows] - 1))
        return;
        
    [self selectRow:++selectedRow byExtendingSelection:NO];
    [self scrollRowToVisible:selectedRow];
}

- (void)moveLeft:(id)sender
{
    id selectedItem;
    int selectedRow;

    selectedRow = [self selectedRow];
    selectedItem = [self itemAtRow:selectedRow];

    if ([_dataSource outlineView:self isItemExpandable:selectedItem] ||
        [self isItemExpanded: selectedItem])
        [self collapseItem:selectedItem];
}

- (void)moveRight:(id)sender;
{
    id selectedItem;
    int selectedRow;

    selectedRow = [self selectedRow];
    selectedItem = [self itemAtRow:selectedRow];

    if ([_dataSource outlineView:self isItemExpandable:selectedItem])
        [self expandItem:selectedItem];
}

// NSView

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent;
{
    return YES;
}

- (BOOL)shouldDelayWindowOrderingForEvent:(NSEvent *)anEvent 
{
    return YES;
}

- (void)setFrameSize:(NSSize)newSize;
{
    NSRect contentViewFrame;

    // Normally, our frame size is exactly tall enough to fit all of our rows. However, this causes a problem when the outline view is shorter than the scroll view's content view (when there are not enough rows to fill up the whole area). When we drag to the bottom of the outline, the drag image gets clipped to the last row. To fix this problem, we force ourself to always be at least as tall as the scroll view's content view.
    contentViewFrame = [[[self enclosingScrollView] contentView] frame];
    newSize.height = MAX(newSize.height, contentViewFrame.size.height);

    [super setFrameSize:newSize];
}

- (void)drawRect:(NSRect)rect;
{
    NSRect rowFrame;

    [super drawRect:rect];

/* Commenting this out since it's ugly, blinky, and makes it difficult to see the drag destination indicator.
    if (draggingSourceItem && (!flags.isDraggingDestination || flags.isDraggingDestinationAcceptable)) {
        rowFrame = [self rectOfRow:dragSourceRow];
        [[NSColor colorWithDeviceRed:0.2 green:0.2 blue:0.4 alpha:0.8] set];
        NSRectFillUsingOperation(rowFrame, NSCompositeSourceOver);
    } */

    if (flags.isDraggingDestination && flags.isDraggingDestinationAcceptable) {
        NSRect outlineTableColumnRect;
        NSPoint insertionPoint;
        NSBezierPath *path;
        
        if (dragDestinationRow >= [self numberOfRows]) {
            rowFrame = [self rectOfRow:dragDestinationRow-1];
            rowFrame.origin.y = NSMaxY(rowFrame);
        } else
            rowFrame = [self rectOfRow:dragDestinationRow];
        
        outlineTableColumnRect = [self rectOfColumn:[[self tableColumns] indexOfObject:[self outlineTableColumn]]];
        
        insertionPoint.x = NSMinX(outlineTableColumnRect) + [self indentationPerLevel] * dragDestinationLevel + DISCLOSURE_TRIANGLE_WIDTH;
        insertionPoint.y = rowFrame.origin.y;

        path = [NSBezierPath bezierPath];
        [path appendBezierPathWithArcWithCenter:insertionPoint radius:4.5 startAngle:0 endAngle:360];
        [[NSColor whiteColor] set];
        [path fill];

        path = [NSBezierPath bezierPath];
        [path appendBezierPathWithArcWithCenter:insertionPoint radius:2.5 startAngle:0 endAngle:360];
        [path setLineWidth:1.5];
        [[NSColor blackColor] set];
        [path stroke];
        
        path = [NSBezierPath bezierPath];
        [path moveToPoint:NSMakePoint(insertionPoint.x + 2, insertionPoint.y)];
        [path relativeLineToPoint:NSMakePoint(NSMaxX(rowFrame), 0)];
        [path setLineWidth:2.0];
        [path stroke];
    }
}

// NSControl

- (void)mouseDown:(NSEvent *)event;
{
    NSPoint eventLocationInWindow, eventLocation;
    int columnIndex, rowIndex;
    NSRect slopRect;
    const int dragSlop = 4;
    NSEvent *mouseDragCurrentEvent;

    if (![_dataSource respondsToSelector:@selector(outlineView:copyItem:toPasteboard:)]) {
        [super mouseDown:event];
        return;
    }

    eventLocationInWindow = [event locationInWindow];
    eventLocation = [self convertPoint:eventLocationInWindow fromView:nil];
    columnIndex = [self columnAtPoint:eventLocation];
    rowIndex = [self rowAtPoint:eventLocation];
    if (rowIndex == -1 || columnIndex == -1)
        return;

    if ([[self tableColumns] objectAtIndex:columnIndex] == [self outlineTableColumn]) {
        NSRect cellRect;
        
        cellRect = [self frameOfCellAtColumn:columnIndex row:rowIndex];
        if (eventLocation.x < NSMinX(cellRect)) {
            [super mouseDown:event];
            return;
        }
    }

    slopRect = NSInsetRect(NSMakeRect(eventLocationInWindow.x, eventLocationInWindow.y, 0.0, 0.0), -dragSlop, -dragSlop);
    while (1) {
        NSEvent *nextEvent;

        nextEvent = [NSApp nextEventMatchingMask:NSLeftMouseDraggedMask|NSLeftMouseUpMask untilDate:[NSDate distantFuture] inMode:NSEventTrackingRunLoopMode dequeue:NO];
        mouseDragCurrentEvent = nextEvent;

        if ([nextEvent type] == NSLeftMouseUp) {
            break;
        } else {
            [NSApp nextEventMatchingMask:NSLeftMouseDraggedMask untilDate:[NSDate distantFuture] inMode:NSEventTrackingRunLoopMode dequeue:YES];
            if (!NSMouseInRect([nextEvent locationInWindow], slopRect, NO)) {
                [self startDrag:event];
                return;
            }
        }
    }

    [super mouseDown:event];
}

// NSTableView

- (void)setDataSource:(id)aSource;
{
    [super setDataSource:aSource];
    
    [self registerForDragging];
}

- (void)textDidEndEditing:(NSNotification *)notification;
{
    if (shouldEditNextItemWhenEditingEnds == NO) {
        // This is ugly, but just about the only way to do it. NSTableView is determined to select and edit something else, even the text field that it just finished editing, unless we mislead it about what key was pressed to end editing.
        NSMutableDictionary *newUserInfo;
        NSNotification *newNotification;
        
        newUserInfo = [NSMutableDictionary dictionaryWithDictionary:[notification userInfo]];
        [newUserInfo setObject:[NSNumber numberWithInt:0] forKey:@"NSTextMovement"];
        newNotification = [NSNotification notificationWithName:[notification name] object:[notification object] userInfo:newUserInfo];
        [super textDidEndEditing:newNotification];

        // For some reason we lose firstResponder status when when we do the above.
        [[self window] makeFirstResponder:self];
    } else {
        [super textDidEndEditing:notification];
    }
}

- (void)reloadData;
{
    // Some edit has been made.  We can't trust that any of these items are really still in the outline or that their hierarchy hasn't changed
    [autoExpandedItems release];
    autoExpandedItems = nil;
    [super reloadData];
}

- (void)noteNumberOfRowsChanged;
{
    // Some edit has been made.  We can't trust that any of these items are really still in the outline or that their hierarchy hasn't changed
    [autoExpandedItems release];
    autoExpandedItems = nil;
    [super noteNumberOfRowsChanged];
}

//
// NSOutlineView
//

- (void)expandItem:(id)item expandChildren:(BOOL)expandChildren;
{
    NSRect visibleRect, parentRowRect;
    unsigned int itemRow, rowCount, lastChild;
    NSBitmapImageRep *followingRowsImageRep, *childrenImageRep;
    NSImage *followingImage, *childrenImage;
    id followingItem;
    NSRect followingRowsRect, childrenRect, lastChildRect;
    NSTimeInterval  start, current, elapsed;

    if (!flags.smoothExpandCollapse ||
        ![_dataSource outlineView: self numberOfChildrenOfItem: item]) {
        // Nothing to animate
        [super expandItem: item expandChildren: expandChildren];
        return;
    }
    
    visibleRect = [self visibleRect];
    itemRow = [self rowForItem: item];
    parentRowRect = [self rectOfRow: itemRow];
    if (!NSIntersectsRect(parentRowRect, visibleRect)) {
        // Don't do animations that we can't see
        [super expandItem: item expandChildren: expandChildren];
        return;
    }
    
    rowCount = [self numberOfRows];

    if (itemRow == rowCount - 1) {
        // We are opening the last row.  There are no following rows to cache
        followingRowsImageRep = nil;
        followingImage = nil;
        followingItem = nil;
    } else {
        OBASSERT([self isFlipped]);
        followingRowsRect.origin.x = 0;
        followingRowsRect.origin.y = NSMaxY(parentRowRect);
        followingRowsRect.size.width = NSWidth(visibleRect);
        followingRowsRect.size.height = NSMaxY(visibleRect) - NSMaxY(parentRowRect);
        //NSLog(@"followingRowsRect = %@", NSStringFromRect(followingRowsRect));
        
        [self lockFocus];
        followingRowsImageRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: followingRowsRect] autorelease];
        [self unlockFocus];
        //[[followingRowsImageRep TIFFRepresentation] writeToFile: @"/tmp/following.tiff" atomically: YES];
        
        followingImage = [[[NSImage alloc] initWithSize: followingRowsRect.size] autorelease];
        [followingImage addRepresentation: followingRowsImageRep];
        //NSLog(@"followingImage = %@", followingImage);
        
        followingItem = [self itemAtRow: itemRow + 1];
    }
    
    // Do the expand.  Make sure we get drawn, so that the cache below works, but don't flush to the window
    [super expandItem: item expandChildren: expandChildren];
    [self lockFocus];
    [self drawSelfAndSubviewsInRect: visibleRect];
    [self unlockFocus];
    
    // Now, cache the new exposed children (everything between the opening item and the old followingItem)
    if (followingItem) {
        lastChild = [self rowForItem: followingItem] - 1;
    } else
        lastChild = [self numberOfRows] - 1;
    lastChildRect = [self rectOfRow: lastChild];
    
    childrenRect.origin.x = 0;
    childrenRect.origin.y = NSMaxY(parentRowRect);
    childrenRect.size.width = NSWidth(visibleRect);
    childrenRect.size.height = MIN(NSMaxY(visibleRect), NSMaxY(lastChildRect)) - NSMaxY(parentRowRect);
    //NSLog(@"childrenRect = %@", NSStringFromRect(childrenRect));
    [self lockFocus];
    childrenImageRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: childrenRect] autorelease];
    [self unlockFocus];
    //[[childrenImageRep TIFFRepresentation] writeToFile: @"/tmp/children.tiff" atomically: YES];
    
    childrenImage = [[[NSImage alloc] initWithSize: childrenRect.size] autorelease];
    [childrenImage addRepresentation: childrenImageRep];

    // Do the animation loop.  On each frame, display some percentage of the
    // children pixels (starting from the bottom) and shift the following
    // rows down by that amount.
    start = [NSDate timeIntervalSinceReferenceDate];    
    while (YES) {
        float  ratio;
        unsigned int childrenPixels;
        NSRect childrenSubRect, followingSubRect;
        NSPoint childrenPoint, followingPoint;
        
        current = [NSDate timeIntervalSinceReferenceDate];
        elapsed = current - start;
        if (elapsed >  TRANSITION_TIME)
            break;
        ratio = elapsed / TRANSITION_TIME;
    
        childrenPixels = ratio * NSHeight(childrenRect);
        childrenSubRect.origin.x = 0;
        childrenSubRect.origin.y = 0;
        childrenSubRect.size.width = NSWidth(visibleRect);
        childrenSubRect.size.height = childrenPixels;
        childrenPoint.x = 0;
        childrenPoint.y = NSMaxY(parentRowRect) + NSHeight(childrenSubRect);
        
        followingSubRect.origin.x = 0;
        followingSubRect.origin.y = childrenPixels;
        followingSubRect.size.width = NSWidth(visibleRect);
        followingSubRect.size.height = followingRowsRect.size.height - childrenPixels;
        followingPoint.x = 0;
        followingPoint.y = NSMaxY(visibleRect);
        
        //NSLog(@"followingSubRect = %@", NSStringFromRect(followingSubRect));
        
        [self lockFocus];
        [childrenImage compositeToPoint: childrenPoint fromRect: childrenSubRect operation: NSCompositeCopy];
        [followingImage compositeToPoint: followingPoint fromRect: followingSubRect operation: NSCompositeCopy];
        [self unlockFocus];
        
        [_window flushWindow];
    }
}


#warning TJW -- In DP4 NSOutlineView does not call the public method when collapsing.  Thus, we are subclassing the private method
- (void)_collapseItem: (id) item collapseChildren:(BOOL)collapseChildren clearExpandState: (BOOL)clearExpandState;
{
    NSRect visibleRect, parentRowRect;
    unsigned int itemRow, lastChild;
    NSBitmapImageRep *followingRowsImageRep, *childrenImageRep;
    NSImage *followingImage, *childrenImage;
    NSRect followingRowsRect, childrenRect, lastChildRect;
    NSTimeInterval  start, current, elapsed;
    unsigned int childrenCount;
        
    //check for item
    if (!item) {
        return;
    }
    
    if (!flags.smoothExpandCollapse ||
        !(childrenCount = [_dataSource outlineView: self numberOfChildrenOfItem: item])) {
        // Nothing to animate
        
        // DONT FORGET TO FIX THE SUPER BELOW TOO (when we can subclass the public method)
        [super _collapseItem: item collapseChildren: collapseChildren clearExpandState: clearExpandState];
        return;
    }
    
    visibleRect = [self visibleRect];
    itemRow = [self rowForItem: item];
    parentRowRect = [self rectOfRow: itemRow];
    if (!NSIntersectsRect(parentRowRect, visibleRect)) {
        // Don't do animations that we can't see
        // DONT FORGET TO FIX THE SUPER BELOW TOO (when we can subclass the public method)
        [super _collapseItem: item collapseChildren: collapseChildren clearExpandState: clearExpandState];
        return;
    }
    
    lastChild = itemRow + childrenCount;
    lastChildRect = [self rectOfRow: lastChild];
    
    // Cache the visible children
    childrenRect.origin.x = 0;
    childrenRect.origin.y = NSMaxY(parentRowRect);
    childrenRect.size.width = NSWidth(visibleRect);
    childrenRect.size.height = MIN(NSMaxY(visibleRect), NSMaxY(lastChildRect)) - NSMaxY(parentRowRect);
    //NSLog(@"childrenRect = %@", NSStringFromRect(childrenRect));

    [self lockFocus];
    childrenImageRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: childrenRect] autorelease];
    [self unlockFocus];
    //[[childrenImageRep TIFFRepresentation] writeToFile: @"/tmp/children.tiff" atomically: YES];
    
    childrenImage = [[[NSImage alloc] initWithSize: childrenRect.size] autorelease];
    [childrenImage addRepresentation: childrenImageRep];


    // Do the collapse.  Make sure we get drawn, so that the cache below works, but don't flush to the window
    // DONT FORGET TO FIX THE SUPER ABOVE TOO (when we can subclass the public method)
    [super _collapseItem: item collapseChildren: collapseChildren clearExpandState: clearExpandState];
    [self lockFocus];
    [self drawSelfAndSubviewsInRect: visibleRect];
    [self unlockFocus];

    // Cache the new stuff under the parent row
    followingRowsRect.origin.x = 0;
    followingRowsRect.origin.y = NSMaxY(parentRowRect);
    followingRowsRect.size.width = NSWidth(visibleRect);
    followingRowsRect.size.height = NSMaxY(visibleRect) - NSMaxY(parentRowRect);
    //NSLog(@"followingRowsRect = %@", NSStringFromRect(followingRowsRect));
    
    [self lockFocus];
    followingRowsImageRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: followingRowsRect] autorelease];
    [self unlockFocus];
    //[[followingRowsImageRep TIFFRepresentation] writeToFile: @"/tmp/following.tiff" atomically: YES];
    
    followingImage = [[[NSImage alloc] initWithSize: followingRowsRect.size] autorelease];
    [followingImage addRepresentation: followingRowsImageRep];
    //NSLog(@"followingImage = %@", followingImage);


    // Do the animation loop.  On each frame, remove some percentage of the
    // children pixels (starting from the top) and shift the following
    // rows up by that amount.
    start = [NSDate timeIntervalSinceReferenceDate];    
    while (YES) {
        float  ratio;
        unsigned int childrenPixels;
        NSRect childrenSubRect, followingSubRect;
        NSPoint childrenPoint, followingPoint;
        
        current = [NSDate timeIntervalSinceReferenceDate];
        elapsed = current - start;
        if (elapsed >  TRANSITION_TIME)
            break;
        ratio = elapsed / TRANSITION_TIME;
    
        childrenPixels = ratio * NSHeight(childrenRect);
        childrenSubRect.origin.x = 0;
        childrenSubRect.origin.y = 0;
        childrenSubRect.size.width = NSWidth(visibleRect);
        childrenSubRect.size.height = NSHeight(childrenRect) - childrenPixels;
        childrenPoint.x = 0;
        childrenPoint.y = NSMaxY(parentRowRect) + NSHeight(childrenSubRect);
        
        followingSubRect.origin.x = 0;
        followingSubRect.origin.y = NSHeight(childrenSubRect);
        followingSubRect.size.width = NSWidth(visibleRect);
        followingSubRect.size.height = NSHeight(followingRowsRect) - NSHeight(childrenSubRect);
        followingPoint.x = 0;
        followingPoint.y = NSMaxY(visibleRect);
        
        //NSLog(@"followingSubRect = %@", NSStringFromRect(followingSubRect));
        
        [self lockFocus];
        [childrenImage compositeToPoint: childrenPoint fromRect: childrenSubRect operation: NSCompositeCopy];
        [followingImage compositeToPoint: followingPoint fromRect: followingSubRect operation: NSCompositeCopy];
        [self unlockFocus];
        
        [_window flushWindow];
    }
}

- (void)reloadItem:(id)item reloadChildren:(BOOL)reloadChildren;
{
    // Some edit has been made.  We can't trust that any of these items are really still in the outline or that their hierarchy hasn't changed
    [autoExpandedItems release];
    autoExpandedItems = nil;
    [super reloadItem: item reloadChildren: reloadChildren];
}

//
// NSDraggingDestination
//

- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender;
{
    flags.isDraggingDestination = YES;
    dragDestinationRow = NSNotFound;
    return [self draggingUpdated:sender];
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender;
{
    NSPoint draggedImageLocation;
    int requestedRow;
    int maximumLevel, minimumLevel, requestedLevel;
    BOOL allowDrag;
    
    draggedImageLocation = [self convertPoint:[sender draggedImageLocation] fromView:nil];
    draggedImageLocation.x -= DISCLOSURE_TRIANGLE_WIDTH + NSMinX([self rectOfColumn:[[self tableColumns] indexOfObject:[self outlineTableColumn]]]);

    // Figure out the row at which this drag would drop
    requestedRow = [self rowAtPoint:draggedImageLocation];
    if (requestedRow == -1)
        requestedRow = [self numberOfRows];

    // Kind of a special case: if we are trying to drop right AFTER the row we are dragging, then that's really the same as dropping ON the current row, so we change the requestedRow so when we special-case the drawing we only have to handle one special case.
    if (draggingSourceItem && requestedRow == dragSourceRow+1)
        requestedRow = dragSourceRow;
    
    // Figure out the indention level at which this drag would drop
    if (requestedRow == 0) {
        requestedLevel = 0;
    } else {
        maximumLevel = [self levelForRow:requestedRow-1]+1;
        
        if (draggingSourceItem && requestedRow == dragSourceRow) {
            if (requestedRow >= [self numberOfRows]-1)
                minimumLevel = 0;
            else
                minimumLevel = MAX([self levelForRow:requestedRow+1]-1, 0);
        } else {
            if (requestedRow == [self numberOfRows])
                minimumLevel = 0;
            else
                minimumLevel = MAX([self levelForRow:requestedRow]-1, 0);
        }
        requestedLevel = MAX(MIN(rint((draggedImageLocation.x - [self rowOffset:requestedRow]) / [self indentationPerLevel]), maximumLevel), minimumLevel);
    }

    // Give the dataSource a chance to change or deny which indention level we'll drop to.
    do {
        id parentItem;
        unsigned int childIndex;
        
        // Is it OK to drop where we are right now?
        parentItem = [self parentItemForRow:requestedRow indentLevel:requestedLevel child:&childIndex];
        allowDrag = [_dataSource outlineView:self allowPasteItemFromPasteboard:[sender draggingPasteboard] parentItem:parentItem child:childIndex];
        if (!allowDrag) {
            int normalDestinationLevel;
            
            normalDestinationLevel = [self levelForRow:requestedRow];
            if (requestedLevel == normalDestinationLevel) {
                flags.isDraggingDestinationAcceptable = NO;
                [self setNeedsDisplay:YES];
                return NSDragOperationNone;
            }
            requestedLevel = normalDestinationLevel;
        }
    } while (!allowDrag);
    
    // The drag is allowable (collapse drawing so we don't flicker if we've already this state)!
    if (requestedRow != dragDestinationRow || requestedLevel != dragDestinationLevel) {
        dragDestinationLevel = requestedLevel;
        dragDestinationRow = requestedRow;

        // Don't accept the drag if we're the source and we're dropping right where we started dragging.
        flags.isDraggingDestinationAcceptable = draggingSourceItem == nil || dragDestinationRow != dragSourceRow || dragDestinationLevel != dragSourceLevel;
    
        [self setNeedsDisplay:YES];
    }
    
    if (flags.isDraggingDestinationAcceptable)
        return [sender draggingSourceOperationMask];
    else
        return NSDragOperationNone;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender;
{
    flags.isDraggingDestination = NO;
    [self setNeedsDisplay:YES];
}

//- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender;
- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender;
{
    id parentItem;
    unsigned int childIndex;
    
    if (!flags.isDraggingDestinationAcceptable)
        return NO;

    if ([_dataSource respondsToSelector:@selector(undoManagerForOutlineView:)])
        [[_dataSource undoManagerForOutlineView:self] beginUndoGrouping];

    // If we are the drag source, we paste the item in the new position and later delete it in the old position. We don't want the intermediate step to be displayed, so we turn off the window updating until we're finished.
    if (draggingSourceItem != nil)
        [[self window] setAutodisplay:NO];
        
    parentItem = [self parentItemForRow:dragDestinationRow indentLevel:dragDestinationLevel child:&childIndex];
    [_dataSource outlineView:self pasteItemFromPasteboard:[sender draggingPasteboard] parentItem:parentItem child:childIndex];
    [self reloadData];

    // Must we move some children from our new sibling (who is above us now)?
    if (dragDestinationLevel < [self levelForRow:dragDestinationRow]) {
        id previousParent, newParent;
        NSMutableArray *movingChildren;
        unsigned int movingChildIndex, previousParentChildCount;

        previousParent = [self parentItemForRow:dragDestinationRow child:&movingChildIndex];
        previousParentChildCount = [_dataSource outlineView:self numberOfChildrenOfItem:previousParent];
        newParent = [_dataSource outlineView:self child:childIndex ofItem:parentItem];
        
        movingChildren = [NSMutableArray arrayWithCapacity:previousParentChildCount];
        for (; movingChildIndex < previousParentChildCount; movingChildIndex++) {
            id movingChild;
            
            movingChild = [_dataSource outlineView:self child:movingChildIndex ofItem:previousParent];
            if (movingChild == draggingSourceItem || movingChild == newParent)
                continue;
                
            [movingChildren addObject:movingChild];
        }

        [_dataSource outlineView:self parentItem:previousParent moveChildren:movingChildren toNewParentItem:newParent];
        [self reloadData];
    }
    
    if ([_dataSource respondsToSelector:@selector(undoManagerForOutlineView:)]) {
        [[_dataSource undoManagerForOutlineView:self] setActionName:@"Drag Operation"];
        [[_dataSource undoManagerForOutlineView:self] endUndoGrouping];
    }

    flags.justAcceptedDrag = YES;
    return YES;
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender;
{
    flags.isDraggingDestination = NO;
    if (draggingSourceItem != nil) {
        // We're the source of the drag:  we'll draw ourselves after we delete the source item.
    } else {
        [self setNeedsDisplay:YES];
    }
}


// NSDraggingSource

- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)isLocal;
{
    if (isLocal)
        return NSDragOperationGeneric;
    else
        return NSDragOperationCopy;
}

//- (void)draggedImage:(NSImage *)image beganAt:(NSPoint)screenPoint;
//- (void)draggedImage:(NSImage *)image endedAt:(NSPoint)screenPoint deposited:(BOOL)flag;
//- (BOOL)ignoreModifierKeysWhileDragging;

//
// Informal OmniFindControllerAware protocol
//

- (id <OAFindControllerTarget>)omniFindControllerTarget;
{
    if (![_dataSource respondsToSelector: @selector(outlineView:item:matchesString:ignoreCase:)])
        return nil;
    return self;
}

//
// OAFindControllerTarget protocol
//

- (BOOL)findString:(NSString *)textPattern ignoreCase:(BOOL)ignoreCase backwards:(BOOL)backwards wrap:(BOOL)wrap;
{
    NSArray *selectedItems, *path;
    id item;
    OAOutlineViewEnumerator *outlineEnum;
    BOOL hasWrapped = NO;
    
    // Start at the first selected item, if any.  If not, start at the first item, if any
    selectedItems = [self selectedItems];
    if ([selectedItems count])
        item = [selectedItems objectAtIndex: 0];
    else {
        item = [self firstItem];
        if (!item)
            // Nothing to find...
            return NO;
    }

    outlineEnum = [[[OAOutlineViewEnumerator alloc] initWithOutlineView: self visibleItem: item] autorelease];
    // If we have a selected item, the enumerator will return it first, but we don't
    // want to consider it in a find operation.
    if ([selectedItems count]) {
        if (backwards)
            [outlineEnum previousPath];
        else
            [outlineEnum nextPath];
    } else {
        // If we had nothing selected and we are going backwards, then set the enumerator to the end
        [outlineEnum resetToEnd];
    }
    
    while (YES) {
        if (backwards)
            path = [outlineEnum previousPath];
        else
            path = [outlineEnum nextPath];
            
        if (!path) {
            if (wrap && !hasWrapped) {
                hasWrapped = YES;
                if (backwards)
                    [outlineEnum resetToEnd];
                else
                    [outlineEnum resetToBeginning];
            } else {
                break;
            }
        }
        
        item = [path lastObject];
        
        if ([_dataSource outlineView: self item: item matchesString: textPattern ignoreCase: ignoreCase]) {
            NSMutableArray *ancestors;
            
            // Don't open the last item (the one that got found)
            ancestors = [[[NSMutableArray alloc] initWithArray: path] autorelease];
            OBASSERT([ancestors count]); // we found something, dammit
            [ancestors removeObjectAtIndex: [ancestors count] - 1];
            
            [self autoExpandItems: ancestors];
            [self setSelectedItem: item];
            [self scrollRowToVisible: [self rowForItem: item]];
            return YES;
        }
    }
    
    return NO;
}

@end

@implementation OAExtendedOutlineView (Private)

- (void)startDrag:(NSEvent *)event;
{
    int outlineTableColumnIndex;
    NSRect rowFrame;
    double xOffsetOfFirstColumn;
    NSPoint eventLocation;
    NSCachedImageRep *cachedImageRep;
    NSRect imageFrame;
    NSView *contentView;
    NSImage *dragImage;
    NSPasteboard *pasteboard;
    id parentItem;
    NSArray *selectedItems;
    id item;
    
    eventLocation = [self convertPoint:[event locationInWindow] fromView:nil];
    dragSourceRow = [self rowAtPoint:eventLocation];
    dragSourceLevel = [self levelForRow:dragSourceRow];

    item = [self itemAtRow:dragSourceRow];
    [self selectRow:dragSourceRow byExtendingSelection:NO];
    if ([self isItemExpanded: item])
        [self collapseItem:[self itemAtRow:dragSourceRow]];

    // Cache an image for the current row
    outlineTableColumnIndex = [[self tableColumns] indexOfObject:[self outlineTableColumn]];
    rowFrame = [self rectOfRow:dragSourceRow];
    xOffsetOfFirstColumn = [self frameOfCellAtColumn:outlineTableColumnIndex row:dragSourceRow].origin.x;
    imageFrame = NSMakeRect(0, 0, NSWidth(rowFrame)-xOffsetOfFirstColumn, NSHeight(rowFrame));

    cachedImageRep = [[NSCachedImageRep alloc] initWithSize:imageFrame.size depth:[[NSScreen mainScreen] depth] separate:YES alpha:YES];
    contentView = [[cachedImageRep window] contentView];

    [contentView lockFocus];
    {
        int columnIndex;
    
        [[NSColor colorWithDeviceWhite:0.0 alpha:0.0] set];
        NSRectFillUsingOperation(imageFrame, NSCompositeCopy);
    
        for (columnIndex = outlineTableColumnIndex; columnIndex < [self numberOfColumns]; columnIndex++) {
            NSTableColumn *tableColumn;
            NSCell *cell;
            NSRect cellRect;
            id objectValue;
            
            tableColumn = [[self tableColumns] objectAtIndex:columnIndex];
            objectValue = [_dataSource outlineView:self objectValueForTableColumn:tableColumn byItem:item];

            cellRect = [self frameOfCellAtColumn:columnIndex row:dragSourceRow];
            cellRect.origin = NSMakePoint(NSMinX(cellRect) - xOffsetOfFirstColumn, 0);
            cell = [tableColumn dataCellForRow:dragSourceRow];
            
            if ([objectValue isKindOfClass:[NSString class]])
                [objectValue drawOutlinedWithFont:nil color:nil backgroundColor:nil rectangle:[cell titleRectForBounds:cellRect]];
            else {
                [cell setCellAttribute:NSCellHighlighted to:0];
                [cell setObjectValue:objectValue];
        
                [cell drawWithFrame:cellRect inView:contentView];
            }
        }
    }
    [contentView unlockFocus];

    dragImage = [[NSImage alloc] init];
    [dragImage addRepresentation:cachedImageRep];
    [cachedImageRep release];

    // Let's start the drag.
    if ([_dataSource respondsToSelector:@selector(undoManagerForOutlineView:)])
        [[_dataSource undoManagerForOutlineView:self] beginUndoGrouping];
        
    selectedItems = [self selectedItems];

    flags.justAcceptedDrag = NO;
    pasteboard = [NSPasteboard pasteboardWithName: NSDragPboard];

    draggingSourceItem = [[self itemAtRow:dragSourceRow] retain];
    parentItem = [self parentItemForRow:[self rowForItem:draggingSourceItem] child:NULL];
    [_dataSource outlineView:self copyItem:draggingSourceItem toPasteboard:pasteboard];
    
    [self dragImage:dragImage at:NSMakePoint(NSMinX(rowFrame)+xOffsetOfFirstColumn, NSMaxY(rowFrame) - 1) offset:NSMakeSize(0, 0) event:event pasteboard:pasteboard source:self slideBack:YES];
    [self setNeedsDisplay:YES];

    [dragImage release];
    
    // Only delete if the drag was accepted in THIS outlineView
    if (flags.justAcceptedDrag) {
        [_dataSource outlineView:self deleteItem:draggingSourceItem parentItem:parentItem];

        // NOW we can display again.
        [[self window] setAutodisplay:YES];
        [self setNeedsDisplay:YES];
    }
            
    [draggingSourceItem release];
    draggingSourceItem = nil; // Do this here so reloadData, below, draws correctly
    
    if (flags.justAcceptedDrag)
        [self reloadData];
    
    [self setSelectedItems:selectedItems];

    if ([_dataSource respondsToSelector:@selector(undoManagerForOutlineView:)]) {
        [[_dataSource undoManagerForOutlineView:self] setActionName:@"Drag Operation"];
        [[_dataSource undoManagerForOutlineView:self] endUndoGrouping];
    }

    flags.justAcceptedDrag = NO;
}

- (void)registerForDragging;
{
    if ([_dataSource respondsToSelector:@selector(outlineViewAcceptedPasteboardTypes:)])
        [self registerForDraggedTypes:[_dataSource outlineViewAcceptedPasteboardTypes:self]];
}

- (void)expandItemAndChildren:(id)item;
{
    if (item == nil || [_dataSource outlineView:self isItemExpandable:item]) {
        unsigned int childIndex, childCount;

        if (item != nil)
            [self expandItem:item];
    
        childCount = [_dataSource outlineView:self numberOfChildrenOfItem:item];
        for (childIndex = 0; childIndex < childCount; childIndex++)
            [self expandItemAndChildren:[_dataSource outlineView:self child:childIndex ofItem:item]];
    }
}

- (void)collapseItemAndChildren:(id)item;
{
    if (item == nil || [_dataSource outlineView:self isItemExpandable:item]) {
        unsigned int childIndex;

        // Collapse starting from the bottom.  This makes it feasible to have
        // the smooth scrolling on when doing this (since most of the collapsing
        // then happens off screen and thus doesn't get animated).
        childIndex = [_dataSource outlineView:self numberOfChildrenOfItem:item];
        while (childIndex--)
            [self collapseItemAndChildren:[_dataSource outlineView:self child:childIndex ofItem:item]];
            
        if (item != nil)
            [self collapseItem:item];
    }
}

@end


