// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAApplication.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import "NSView-OAExtensions.h"
#import "OAAppkitQueueProcessor.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/OAApplication.m,v 1.26 2000/11/27 23:42:34 kc Exp $")

@interface OAApplication (Private)
- (void)processMouseButtonsChangedEvent:(NSEvent *)event;
@end

@interface NSWindow (OAApplicationExtensions)
- (id)_addToOrderedWindowsArray;
@end

@implementation OAApplication

+ (void)setupOmniApplication;
{
    [OBPostLoader processClasses];
}

+ (NSApplication *)sharedApplication;
{
    static OAApplication *omniApplication = nil;

    if (omniApplication)
        return omniApplication;

    omniApplication = (id)[super sharedApplication];
    [OAApplication setupOmniApplication];
    return omniApplication;
}

- (void)run;
{
    BOOL stillRunning = YES;

    exceptionCount = 0;
    exceptionCheckpointDate = [[NSDate alloc] init];
    do {
        NS_DURING {
            [super run];
            stillRunning = NO;
        } NS_HANDLER {
            if (++exceptionCount % 300 == 0) {
                if ([exceptionCheckpointDate timeIntervalSinceNow] > -3.0)
                    stillRunning = NO; // 300 exceptions in 3 seconds: abort
                [exceptionCheckpointDate release];
                exceptionCheckpointDate = [[NSDate alloc] init];
                exceptionCount = 0;
            }
            if (localException) {
                if (_appFlags._hasBeenRun)
                    [self handleRunException:localException];
                else
                    [self handleInitException:localException];
            }
        } NS_ENDHANDLER;
    } while (stillRunning && _appFlags._hasBeenRun);
}

#define OASystemDefinedEvent_MouseButtonsChangedSubType 7

- (void)sendEvent:(NSEvent *)event;
{
    // The -timestamp method on NSEvent doesn't seem to return an NSTimeInterval based off the same reference date as NSDate (which is what we want).
    lastEventTimeInterval = [NSDate timeIntervalSinceReferenceDate];

    NS_DURING {
        if ([event type] == NSSystemDefined && [event subtype] == OASystemDefinedEvent_MouseButtonsChangedSubType) {
            [self processMouseButtonsChangedEvent:event];
            [super sendEvent:event];
        } else if ([event type] == NSScrollWheel) {
            if ([self scrollWheelButtonIsDown]) {
                NSView *contentView;
                NSView *viewUnderMouse;

                contentView = [[event window] contentView];
                viewUnderMouse = [contentView hitTest:[contentView convertPoint:[event locationInWindow] fromView:nil]];
                [[viewUnderMouse enclosingScrollView] scrollDownByPages:-[event deltaY]];
            } else {
                [super sendEvent:event];
            }
        } else {
            [super sendEvent:event];
        }
    } NS_HANDLER {
        if ([[localException name] isEqualToString:NSAbortModalException] || [[localException name] isEqualToString:NSAbortPrintingException])
            [localException raise];
        [self handleRunException:localException];
    } NS_ENDHANDLER;
}

- (void)handleInitException:(NSException *)anException;
{
    id delegate;

    delegate = [self delegate];
    if ([delegate respondsToSelector:@selector(handleInitException:)]) {
        [delegate handleInitException:anException];
    } else {
        NSLog(@"%@", [anException reason]);
    }
}

- (void)handleRunException:(NSException *)anException;
{
    id delegate;

    delegate = [self delegate];
    if ([delegate respondsToSelector:@selector(handleRunException:)]) {
        [delegate handleRunException:anException];
    } else {
        NSLog(@"%@", [anException reason]);
        NSRunAlertPanel(nil, @"%@", nil, nil, nil, [anException reason]);
    }
}

- (NSTimeInterval)lastEventTimeInterval;
{
    return lastEventTimeInterval;
}

- (BOOL)mouseButtonIsDownAtIndex:(unsigned int)mouseButtonIndex;
{
    return (mouseButtonState & (1 << mouseButtonIndex)) != 0;
}

- (BOOL)scrollWheelButtonIsDown;
{
    return [self mouseButtonIsDownAtIndex:2];
}

static NSMutableArray *_orderedWindows = nil;

- (NSArray *)orderedWindows;
    // Note:  this method would need a lock to be thread-safe
{
    NSArray *orderedWindows;

    _orderedWindows = [[NSMutableArray alloc] init];
    [self makeWindowsPerform:@selector(_addToOrderedWindowsArray) inOrder:YES];
    orderedWindows = [[NSArray alloc] initWithArray:_orderedWindows];
    [_orderedWindows release];
    _orderedWindows = nil;
    return [orderedWindows autorelease];
}

// Actions

- (IBAction)closeAllMainWindows:(id)sender;
{
    NSArray *windows;
    unsigned int windowIndex, windowCount;

    windows = [[NSArray alloc] initWithArray:[self orderedWindows]];
    windowCount = [windows count];
    for (windowIndex = 0; windowIndex < windowCount; windowIndex++) {
        NSWindow *window;

        window = [windows objectAtIndex:windowIndex];
        if ([window canBecomeMainWindow])
            [window performClose:nil];
    }
}

- (IBAction)cycleToNextMainWindow:(id)sender;
{
    NSWindow *mainWindow;
    NSArray *orderedWindows;
    unsigned int windowIndex, windowCount;
    
    mainWindow = [NSApp mainWindow];
    orderedWindows = [NSApp orderedWindows];
    windowCount = [orderedWindows count];
    for (windowIndex = 0; windowIndex < windowCount; windowIndex++) {
        NSWindow *window;

        window = [orderedWindows objectAtIndex:windowIndex];
        if (window != mainWindow && [window canBecomeMainWindow]) {
            [window makeKeyAndOrderFront:nil];
            [mainWindow orderBack:nil];
            return;
        }
    }
    // There's one (or less) window which can potentially be main, make it key and bring it forward.
    [mainWindow makeKeyAndOrderFront:nil];
}

- (IBAction)cycleToPreviousMainWindow:(id)sender;
{
    NSWindow *mainWindow;
    NSArray *orderedWindows;
    unsigned int windowIndex;
    
    mainWindow = [NSApp mainWindow];
    orderedWindows = [NSApp orderedWindows];
    windowIndex = [orderedWindows count];
    while (windowIndex--) {
        NSWindow *window;

        window = [orderedWindows objectAtIndex:windowIndex];
        if (window != mainWindow && [window canBecomeMainWindow]) {
            [window makeKeyAndOrderFront:nil];
            return;
        }
    }
    // There's one (or less) window which can potentially be main, make it key and bring it forward.
    [mainWindow makeKeyAndOrderFront:nil];
}

@end

@implementation OAApplication (Private)

- (void)processMouseButtonsChangedEvent:(NSEvent *)event;
{
    mouseButtonState = [event data2];
}

@end

@implementation NSWindow (OAApplicationExtensions)

- (id)_addToOrderedWindowsArray;
{
    [_orderedWindows addObject:self];
    return nil; // Returns nil so -[NSApplication makeWindowsPerform:inOrder:] continues to the next window
}

@end
