// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/NSString-OAExtensions.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/OpenStepExtensions.subproj/NSString-OAExtensions.m,v 1.20 2000/11/16 11:49:23 wjs Exp $")

@implementation NSString (OAExtensions)

+ (NSString *)possiblyAbbreviatedStringForBytes:(unsigned long long)bytes inTableView:(NSTableView *)tableView tableColumn:(NSTableColumn *)tableColumn;
{
    NSCell *dataCell;
    NSString *bytesString;

    bytesString = [NSString stringWithFormat:@"%@", [NSNumber numberWithUnsignedLongLong:bytes]];
    dataCell = [tableColumn dataCell];
    if ([[dataCell font] widthOfString:bytesString] + 5 <= [dataCell titleRectForBounds:NSMakeRect(0, 0, [tableColumn width], [tableView rowHeight])].size.width)
        return [bytesString stringByAppendingString:NSLocalizedStringFromTableInBundle(@" bytes", @"OmniAppKit", [self bundle], last word of abbreviated bytes string if no abbreviation is necessary)];
    else
        return [NSString abbreviatedStringForBytes:bytes];
}


// String drawing

- (void)drawWithFontAttributes:(NSDictionary *)attributes alignment:(int)alignment rectangle:(NSRect)rectangle;
{
    static NSTextStorage *showStringTextStorage = nil;
    static NSLayoutManager *showStringLayoutManager = nil;
    static NSTextContainer *showStringTextContainer = nil;

    NSAttributedString *attributedString;
    unsigned int originalGlyphCount;
    NSRange drawGlyphRange, drawGlyphRangeWithoutEllipsis;
    NSRect *rectArray;
    unsigned int rectCount;
    NSSize size;
    NSString *ellipsisString;
    NSSize ellipsisSize;
    BOOL drawEllipsisIfTruncated;

    OBPRECONDITION(attributes);

    if (!showStringTextStorage) {
        showStringTextStorage = [[NSTextStorage alloc] init];

        showStringLayoutManager = [[NSLayoutManager alloc] init];
        [showStringTextStorage addLayoutManager:showStringLayoutManager];

        showStringTextContainer = [[NSTextContainer alloc] initWithContainerSize:NSMakeSize(1e7, 1e7)];
        [showStringTextContainer setLineFragmentPadding:0];
        [showStringLayoutManager addTextContainer:showStringTextContainer];
    }
    
    attributedString = [[NSAttributedString alloc] initWithString:self attributes:attributes];
    [showStringTextStorage setAttributedString:attributedString];
    [attributedString release];

    drawGlyphRange = [showStringLayoutManager glyphRangeForTextContainer:showStringTextContainer];
    if (drawGlyphRange.length == 0)
        return;
    drawGlyphRangeWithoutEllipsis = NSMakeRange(0, 0);
    originalGlyphCount = drawGlyphRange.length;

    ellipsisString = nil;
    ellipsisSize = NSMakeSize(0, 0);

    rectArray = [showStringLayoutManager rectArrayForGlyphRange:drawGlyphRange withinSelectedGlyphRange:NSMakeRange(NSNotFound, 0) inTextContainer:showStringTextContainer rectCount:&rectCount];
    if (rectCount < 1)
        return;

    size = rectArray[0].size;

    if (size.width > NSWidth(rectangle)) {
        NSSize testSize;
        unsigned int lowerCount, upperCount;

        lowerCount = 0;
        upperCount = originalGlyphCount;

        ellipsisString = [NSString horizontalEllipsisString];
        ellipsisSize = [ellipsisString sizeWithAttributes:attributes];

        while (lowerCount + 1 < upperCount) {
            unsigned int middleCount;

            middleCount = (upperCount + lowerCount) / 2;

#warning WJS: This is slow, I found out.  Use the same algorithm OHLine uses.
            rectArray = [showStringLayoutManager rectArrayForGlyphRange:NSMakeRange(0, middleCount) withinSelectedGlyphRange:NSMakeRange(NSNotFound, 0) inTextContainer:showStringTextContainer rectCount:&rectCount];
            if (rectCount < 1)
                return;

            testSize = rectArray[0].size;

            // DEBUGGING
//            rectArray[0].origin.x += rectangle.origin.x;
//            rectArray[0].origin.y += rectangle.origin.y;
//            NSDottedFrameRect(rectArray[0]);

            if (testSize.width <= NSWidth(rectangle) && drawGlyphRangeWithoutEllipsis.length < middleCount)
                drawGlyphRangeWithoutEllipsis = NSMakeRange(0, middleCount);
            testSize.width += ellipsisSize.width;

            if (testSize.width <= NSWidth(rectangle)) {
                lowerCount = middleCount;
                size = testSize;
            } else
                upperCount = middleCount;
        }
        
        drawGlyphRange.length = lowerCount;
    }

    if (drawGlyphRange.length != 0) {
        drawEllipsisIfTruncated = YES;
    } else {
        // If we couldn't fit ANY characters with the ellipsis, try drawing some without it (better than drawing nothing)
        drawEllipsisIfTruncated = NO;
        drawGlyphRange = drawGlyphRangeWithoutEllipsis;
    }
    
    if (drawGlyphRange.length) {
        NSPoint drawPoint;

        // determine drawPoint based on alignment
        drawPoint.y = NSMinY(rectangle);
        switch (alignment) {
            default:
            case NSLeftTextAlignment:
                drawPoint.x = NSMinX(rectangle);
                break;
            case NSCenterTextAlignment:
                drawPoint.x = NSMidX(rectangle) - size.width / 2.0;
                break;
            case NSRightTextAlignment:
                drawPoint.x = NSMaxX(rectangle) - size.width;
                break;
        }

        [showStringLayoutManager drawGlyphsForGlyphRange:drawGlyphRange atPoint:drawPoint];
        if (drawGlyphRange.length < originalGlyphCount && drawEllipsisIfTruncated) {
            // draw only part of string, then maybe ellipsis if they fit
            drawPoint.x += size.width - ellipsisSize.width;
            [ellipsisString drawAtPoint:drawPoint withAttributes:attributes];
        }
    }
}

- (void)drawWithFont:(NSFont *)font color:(NSColor *)color alignment:(int)alignment rectangle:(NSRect)rectangle;
{
    NSMutableDictionary *attributes;

    attributes = [[NSMutableDictionary alloc] initWithCapacity:2];
    if (font)
        [attributes setObject:font forKey:NSFontAttributeName];
    if (color)
        [attributes setObject:color forKey:NSForegroundColorAttributeName];

    [self drawWithFontAttributes:attributes alignment:alignment rectangle:rectangle];
    [attributes release];
}

- (void)drawInRect:(NSRect)rectangle xOffset:(float)xOffset yOffset:(float)yOffset attributes:(NSDictionary *)attributes;
{
    rectangle.origin.x += xOffset;
    rectangle.origin.y += yOffset;
    [self drawInRect:rectangle withAttributes:attributes];
}

- (void)drawOutlineInRect:(NSRect)rectangle radius:(float)radius step:(float)step attributes:(NSDictionary *)attributes;
{
    float offset;

    for (offset = -radius; offset <= radius; offset += step) {
        [self drawInRect:rectangle xOffset:offset yOffset:-radius attributes:attributes];
        [self drawInRect:rectangle xOffset:offset yOffset:radius attributes:attributes];
    }
    for (offset = -radius + step; offset <= radius - step; offset += step) {
        [self drawInRect:rectangle xOffset:-radius yOffset:offset attributes:attributes];
        [self drawInRect:rectangle xOffset:radius yOffset:offset attributes:attributes];
    }
}

- (void)drawFillInRect:(NSRect)rectangle radius:(float)radius step:(float)step attributes:(NSDictionary *)attributes;
{
    float xOffset, yOffset;

    for (xOffset = -radius; xOffset <= radius; xOffset += step)
        for (yOffset = -radius; yOffset <= radius; yOffset += step)
            [self drawInRect:rectangle xOffset:xOffset yOffset:yOffset attributes:attributes];
}

- (void)drawOutlinedWithFont:(NSFont *)font color:(NSColor *)color backgroundColor:(NSColor *)backgroundColor rectangle:(NSRect)rectangle;
{
    NSMutableDictionary *attributes;

    // WJS: OS X DP3 Hack -- munge the rectangle slightly so we draw just like a cell will draw, so that if we're using this in place of a cell the text won't wiggle
    rectangle.origin.x +=2;
    rectangle.size.width -=2;

    attributes = [[NSMutableDictionary alloc] initWithCapacity:2];
    if (font)
        [attributes setObject:font forKey:NSFontAttributeName];
    if (!color) // Default is inverted, white on black field
        color = [NSColor textBackgroundColor];
    if (!backgroundColor)
        backgroundColor = [NSColor textColor];

    // Draw the background
    [attributes setObject:[backgroundColor colorWithAlphaComponent:0.8] forKey:NSForegroundColorAttributeName];
    [self drawFillInRect:rectangle radius:1.0 step:1.0 attributes:attributes];

    [attributes setObject:[backgroundColor colorWithAlphaComponent:0.2] forKey:NSForegroundColorAttributeName];
    [self drawOutlineInRect:rectangle radius:2.0 step:1.0 attributes:attributes];

    // Draw it a final time, in the real color
    [attributes setObject:color forKey:NSForegroundColorAttributeName];
    [self drawInRect:rectangle withAttributes:attributes];
    [attributes release];
}

- (NSImage *)outlinedImageWithColor:(NSColor *)color;
{
    NSImage *image;
    NSSize size;
    
    size = [self sizeWithAttributes:nil];
    size.width += 4.0; // Because the text is fuzzy
    size.height += 4.0;

    image = [[NSImage alloc] initWithSize:size];
    [image lockFocus];
    [self drawOutlinedWithFont:nil color:nil backgroundColor:nil rectangle:(NSRect){NSMakePoint(0, 2), size}];
    [image unlockFocus];
    
    return [image autorelease];
}

@end
