// Copyright 2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OADockStatusItem.h"

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/rcsid.h>
#import <OmniAppKit/NSString-OAExtensions.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OADockStatusItem.m,v 1.4 2001/02/15 15:12:55 kc Exp $")

@interface OADockStatusItem (private)
+ (void)redisplay;
+ (void)registerItem:(OADockStatusItem *)item;
+ (void)unregisterItem:(OADockStatusItem *)item;
- (void)drawDockStatusItem;
@end

@implementation OADockStatusItem

- initWithIcon:(NSImage *)anIcon andTextColor:(NSColor *)color atPosition:(NSPoint)aPosition;
{
    [super init];
    icon = [anIcon retain];
    textColor = [color retain];
    textBackgroundColor = [[NSColor blackColor] retain];
    position = aPosition;
    count = NSNotFound;
    isHidden = YES;
    [isa registerItem:self];
    return self;
}

- (void)dealloc;
{
    [isa unregisterItem:self];
    if (!isHidden)
        [isa redisplay];
    [icon release];
    [textColor release];
    [textBackgroundColor release];
}

- (void)setTextBackgroundColor: (NSColor *) aTextBackgroundColor;
{
    if (textBackgroundColor != aTextBackgroundColor) {
        [textBackgroundColor release];
        textBackgroundColor = [aTextBackgroundColor retain];
        if (!isHidden)
            [isa redisplay];
    }
}

- (void)setCount:(unsigned int)aCount;
{
    if (count != aCount) {
        count = aCount;
        if (!isHidden)
            [isa redisplay];
    }
}

- (void)setNoCount;
{
    [self setCount:NSNotFound];
}

- (void)hide;
{
    if (!isHidden) {
        isHidden = YES;
        [isa redisplay];
    }
}

- (void)show;
{
    if (isHidden) {
        isHidden = NO;
        [isa redisplay];
    }
}

- (BOOL)isHidden;
{
    return isHidden;
}

@end

@implementation OADockStatusItem (private)

static NSImage *appImage, *bufferImage;
static NSMutableArray *dockStatusItems;
static NSTextView *sharedText;

+ (void)redisplay;
{
    unsigned int index;
    
    if (!appImage) {
        NSRect rect;
        
        appImage = [[NSImage imageNamed:@"NSApplicationIcon"] retain];
        rect.origin = NSMakePoint(0,0);
        rect.size = [appImage size];
        sharedText = [[NSTextView alloc] initWithFrame:rect];
    }

    bufferImage = [[NSImage alloc] initWithSize:[appImage size]];
    [bufferImage setFlipped:YES];
    [bufferImage lockFocus];
    [appImage compositeToPoint:NSMakePoint(0, [bufferImage size].height) operation:NSCompositeSourceOver];
    
    index = [dockStatusItems count];
    while(index--)
        [[[dockStatusItems objectAtIndex:index] nonretainedObjectValue] drawDockStatusItem];
    [bufferImage unlockFocus];
    
    [NSApp setApplicationIconImage:bufferImage];
    [bufferImage autorelease];
}

+ (void)registerItem:(OADockStatusItem *)item;
{    
    if (!dockStatusItems)
        dockStatusItems = [[NSMutableArray alloc] init];

    [dockStatusItems addObject:[NSValue valueWithNonretainedObject:item]]; 
}

+ (void)unregisterItem:(OADockStatusItem *)item;
{
    unsigned int index;
    
    index = [dockStatusItems count];
    while(index--) {
        if ([[dockStatusItems objectAtIndex:index] nonretainedObjectValue] == item) {
            [dockStatusItems removeObjectAtIndex:index];
            break;
        }
    }
}

- (void)drawDockStatusItem;
{
    NSRect rect;
    NSRect textRect;
    NSString *string;
    NSAttributedString *attrString;
    NSRange glyphRange;
    NSFont *font;
    
    if (isHidden)
        return;
        
    [icon compositeToPoint:position operation:NSCompositeSourceOver];
    
    if (count == NSNotFound)
        return;
        
    rect.size = [icon size];
    rect.origin.x = position.x;
    rect.origin.y = position.y - rect.size.height;

    string = [NSString stringWithFormat:@"%d", count];
    font = [NSFont boldSystemFontOfSize:rect.size.height / 2.0];
    attrString = [[NSAttributedString alloc] initWithString:string attributes:[NSDictionary dictionaryWithObjectsAndKeys:textColor, NSForegroundColorAttributeName, font, NSFontAttributeName, nil]];
    [[sharedText textStorage] setAttributedString:attrString];
    glyphRange = NSMakeRange(0, [[sharedText layoutManager] numberOfGlyphs]);
    textRect = [[sharedText layoutManager] boundingRectForGlyphRange:glyphRange inTextContainer:[sharedText textContainer]];
    textRect.origin.x = rect.origin.x + (rect.size.width - textRect.size.width) / 2;
    textRect.origin.y = rect.origin.y + (rect.size.height - textRect.size.height) / 2;

    if (count > 1) {
#if 0
    [[sharedText layoutManager] drawGlyphsForGlyphRange:glyphRange atPoint:textRect.origin];
#else
    [string drawOutlinedWithFont:font color:textColor backgroundColor:textBackgroundColor rectangle: textRect];
#endif
    }
    
    [attrString release];
}

@end
