// Copyright 2000-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OAInternetConfig.h"

#import <AppKit/NSPanel.h>
#import <OmniFoundation/OmniFoundation.h>
#import <Foundation/Foundation.h>
#import <CoreFoundation/CoreFoundation.h>
#import <Carbon/Carbon.h>
#import <OmniBase/OmniBase.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Carbon/OAInternetConfig.m,v 1.5 2001/03/03 21:34:22 bungi Exp $")

// The InternetConfig documentation can be found at http://www.quinn.echidna.id.au/Quinn/Config/.

@interface OAInternetConfig (Private)
- (NSString *)_helperKeyNameForScheme:(NSString *)scheme;
@end

@implementation OAInternetConfig

+ (OAInternetConfig *)internetConfig;
{
    return [[[self alloc] init] autorelease];
}

+ (unsigned long)applicationSignature;
{
    NSString *bundleSignature;
    NSData *signatureBytes;

    bundleSignature = [[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleSignature"];
    if (!bundleSignature)
        return FOUR_CHAR_CODE('????');
    
    signatureBytes = [bundleSignature dataUsingEncoding:[NSString defaultCStringEncoding]];
    if (!signatureBytes || ([signatureBytes length] < 4))
        return FOUR_CHAR_CODE('????');
    
    return *(const OSType *)[signatureBytes bytes];
}

// Init and dealloc

- init;
{
    ICInstance anInstance;

    if (![super init])
        return nil;
    if (ICStart(&anInstance, [[self class] applicationSignature]) != noErr)
        return nil;
    internetConfigInstance = anInstance;
    return self;
}

- (void)dealloc;
{
    ICStop(internetConfigInstance);
    [super dealloc];
}

//

- (OAInternetConfigInfo)getInfoForScheme:(NSString *)scheme;
{
    OAInternetConfigInfo info;
    NSString *helperKeyString;
    Str255 helperKeyPascalString;
    ICAppSpec helperApplication;
    ICAttr attribute;
    long helperApplicationSize;
    OSStatus error;

    helperKeyString = [self _helperKeyNameForScheme:scheme];
    CFStringGetPascalString((CFStringRef)helperKeyString, helperKeyPascalString, sizeof(helperKeyPascalString) - 1, kCFStringEncodingMacRoman);

    helperApplicationSize = sizeof(helperApplication);
    error = ICGetPref(internetConfigInstance, helperKeyPascalString, &attribute, &helperApplication, &helperApplicationSize);
    switch (error) {
        case noErr:
            info.valid = YES;
            info.applicationName = [(NSString *)CFStringCreateWithPascalString(NULL, helperApplication.name, kCFStringEncodingMacRoman) autorelease];
            info.creatorCode = helperApplication.fCreator;
            break;
        case icPrefNotFoundErr:
            info.valid = NO;
            NSLog(@"OAInternetConfig: ICGetPref found no helper application for %@", scheme);
        default:
            info.valid = NO;
            NSLog(@"OAInternetConfig: ICGetPref returned an error while getting key %@: 0x%x (%d)", helperKeyString, error, error);
            break;
    }
    
    return info;
}

- (void)setApplicationCreatorCode:(long)applicationCreatorCode name:(NSString *)applicationName forScheme:(NSString *)scheme;
{
    NSString *helperKeyString;
    Str255 helperKeyPascalString;
    ICAppSpec helperApplication;
    long helperApplicationSize;
    OSStatus error;

    helperKeyString = [self _helperKeyNameForScheme:scheme];
    CFStringGetPascalString((CFStringRef)helperKeyString, helperKeyPascalString, sizeof(helperKeyPascalString) - 1, kCFStringEncodingMacRoman);
    helperApplication.fCreator = applicationCreatorCode;
    CFStringGetPascalString((CFStringRef)applicationName, helperApplication.name, sizeof(helperApplication.name) - 1, kCFStringEncodingMacRoman);
    helperApplicationSize = sizeof(helperApplication);
    error = ICSetPref(internetConfigInstance, helperKeyPascalString, kICAttrNoChange, &helperApplication, helperApplicationSize);
    switch (error) {
        case noErr:
            break;
        case icPermErr:
            [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"ICSetPref returned permissions when attempting setting key %@", internet config error)];
        default:
            [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"ICSetPref returned an error while setting key %@: 0x%x (%d)", internet config error), helperKeyString, error, error];
    }
}

- (void)launchURL:(NSString *)urlString;
{
    const char *urlCString;
    long start, length;
    OSStatus error;

    urlCString = [urlString UTF8String];
    start = 0;
    length = strlen(urlCString);
    error = ICLaunchURL(internetConfigInstance, NULL, (Ptr)urlCString, length, &start, &length);
    if (error != noErr)
        [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"ICLaunchURL returned an error while launching URL %@: 0x%x (%d)", internet config error), urlString, error, error];
}

- (BOOL)omniWebIsDefaultBrowser;
{
    OAInternetConfigInfo info;
    
    info = [self getInfoForScheme:@"http"];
    return info.valid && info.creatorCode == FOUR_CHAR_CODE('OWEB');
}

- (void)makeOmniWebDefaultBrowser;
{
    long applicationCreatorCode;
    NSString *applicationName;

    applicationCreatorCode = FOUR_CHAR_CODE('OWEB');
    // applicationName = [[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleName"];
    applicationName = @"OmniWeb";
#warning Get URL schemes from OmniWeb's content type registrations
    // maybe this method should be moved back into OW (as a category?)
    [self setApplicationCreatorCode:applicationCreatorCode name:applicationName forScheme:@"http"];
    [self setApplicationCreatorCode:applicationCreatorCode name:applicationName forScheme:@"https"];
    [self setApplicationCreatorCode:applicationCreatorCode name:applicationName forScheme:@"file"];
    [self setApplicationCreatorCode:applicationCreatorCode name:applicationName forScheme:@"ftp"];
    [self setApplicationCreatorCode:applicationCreatorCode name:applicationName forScheme:@"gopher"];
}

// Download folder

- (NSString *)downloadFolderPath;
{
    // NSData *data;
    ICAttr attribute;
    struct ICFileSpec *downloadFolderICFileSpecPtr;
    long downloadFolderICFileSpecSize;
    AliasPtr downloadFolderAliasPointer;
    Boolean aliasUpdated;
    FSSpec downloadFolderFSSpec;
    OSStatus error;
    UInt8 path[PATH_MAX + 1];
    FSRef downloadFolderReference;

    // data = [self dataForPreferenceKey:kICDownloadFolder];
    error = ICGetPref(internetConfigInstance, kICDownloadFolder, &attribute, NULL, &downloadFolderICFileSpecSize);
    switch (error) {
        case noErr:
            break;
        case icPrefNotFoundErr:
            // No preference found
            return nil;
        default:
            NSLog(@"OAInternetConfig: ICGetPref returned an error while getting key %s: 0x%x (%d)", kICDownloadFolder + 1, error, error);
            // No preference found
            return nil;
    }
    downloadFolderICFileSpecPtr = malloc(downloadFolderICFileSpecSize);
    error = ICGetPref(internetConfigInstance, kICDownloadFolder, &attribute, downloadFolderICFileSpecPtr, &downloadFolderICFileSpecSize);
    switch (error) {
        case noErr:
            break;
        case icPrefNotFoundErr:
            // No preference found
            return nil;
        default:
            NSLog(@"OAInternetConfig: ICGetPref returned an error while getting key %s: 0x%x (%d)", kICDownloadFolder + 1, error, error);
            // No preference found
            return nil;
    }

    downloadFolderAliasPointer = &downloadFolderICFileSpecPtr->alias;
    error = ResolveAlias(NULL, &downloadFolderAliasPointer, &downloadFolderFSSpec, &aliasUpdated);
    if (error != noErr) {
        NSLog(@"-[OAInternetConfig downloadFolderPath]: Error resolving alias: ResolveAlias() failed: 0x%x (%d)", error, error);
        free(downloadFolderICFileSpecPtr);
        return nil;
    }
    error = FSpMakeFSRef(&downloadFolderFSSpec, &downloadFolderReference);
    if (error != noErr) {
        NSString *name;

        name = (NSString *)CFStringCreateWithPascalString(NULL, downloadFolderFSSpec.name, kCFStringEncodingMacRoman);
        NSLog(@"-[OAInternetConfig downloadFolderPath]: Error converting FSSpec (%d,%d,\"%@\") to FSRef: FSpMakeFSRef() failed: 0x%x (%d)", downloadFolderFSSpec.vRefNum, downloadFolderFSSpec.parID, name, error, error);
        free(downloadFolderICFileSpecPtr);
        return nil;
    }
    error = FSRefMakePath(&downloadFolderReference, path, sizeof(path));
    if (error != noErr) {
        NSLog(@"-[OAInternetConfig downloadFolderPath]: Error converting FSRef to Path: FSRefMakePath() failed: 0x%x (%d)", error, error);
        free(downloadFolderICFileSpecPtr);
        return nil;
    }
    return [NSString stringWithCString:path];
}

// Mappings between type/creator codes and filename extensions

- (NSArray *)mapEntries;
{
#warning -mapEntries not finished
    return nil;
}

- (OAInternetConfigMapEntry *)mapEntryForFilename:(NSString *)filename;
{
    OSStatus error;
    ICMapEntry mapEntry;
    Str255 filenamePascalString;

    CFStringGetPascalString((CFStringRef)filename, filenamePascalString, sizeof(filenamePascalString) - 1, kCFStringEncodingMacRoman);
    error = ICMapFilename(internetConfigInstance, filenamePascalString, &mapEntry);
    if (error != noErr)
        [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"ICMapFilename returned an error for file %@: 0x%x (%d)", internet config error), filename, error, error];
#warning -mapEntryForFilename: not finished
    return nil;
}

- (OAInternetConfigMapEntry *)mapEntryForTypeCode:(long)fileTypeCode creatorCode:(long)fileCreatorCode hintFilename:(NSString *)filename;
{
#warning -mapEntryForTypeCode:creatorCode:hintFilename: not finished
    return nil;
}

- (void)editPreferencesFocusOnKey:(NSString *)keyString;
{
    OSStatus error;
    Str255 keyPascalString;

    if (keyString == nil)
        keyString = @"";
    CFStringGetPascalString((CFStringRef)keyString, keyPascalString, sizeof(keyPascalString) - 1, kCFStringEncodingMacRoman);
    error = ICEditPreferences(internetConfigInstance, keyPascalString);
    if (error != noErr)
        [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"Error editing InternetConfig preferences for key %@: 0x%x (%d)", internet config error), keyString, error, error];
}

// Low-level access

- (void)beginReadOnlyAccess;
{
    OSStatus error;

    error = ICBegin(internetConfigInstance, icReadOnlyPerm);
    if (error != noErr)
        [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"Error opening InternetConfig for  read-only access: 0x%x (%d)", internet config error), error, error];
}

- (void)beginReadWriteAccess;
{
    OSStatus error;

    error = ICBegin(internetConfigInstance, icReadWritePerm);
    if (error != noErr)
        [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"Error opening InternetConfig for  read-write access: 0x%x (%d)", internet config error), error, error];
}

- (void)endAccess;
{
    OSStatus error;

    error = ICEnd(internetConfigInstance);
    if (error != noErr)
        [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"ICEnd returned an error: 0x%x (%d)", internet config error), error, error];
}

- (NSArray *)allPreferenceKeys;
{
    OSStatus error;
    NSMutableArray *keys;
    long preferenceIndex, preferenceCount;

    [self beginReadOnlyAccess];
    error = ICCountPref(internetConfigInstance, &preferenceCount);
    if (error != noErr)
        [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"ICCountPref returned an error: 0x%x (%d)", internet config error), error, error];
    keys = [[NSMutableArray alloc] initWithCapacity:preferenceCount];
    for (preferenceIndex = 0; preferenceIndex < preferenceCount; preferenceIndex++) {
        Str255 keyPascalString;
        NSString *keyString;

        error = ICGetIndPref(internetConfigInstance, preferenceIndex + 1, keyPascalString);
        if (error != noErr)
            [NSException raise:@"OAInternetConfigException" format:NSLocalizedString(@"ICGetIndPref %d/%d returned an error: 0x%x (%d)", internet config error), preferenceIndex + 1, preferenceCount, error, error];
            keyString = (NSString *)CFStringCreateWithPascalString(NULL, keyPascalString, kCFStringEncodingMacRoman);
            [keys addObject:keyString];
    }
    [self endAccess];
    return keys;
}

- (NSData *)dataForPreferenceKey:(NSString *)preferenceKey;
{
#warning -dataForPreferenceKey: not finished
    return nil;
}

- (void) launchMailTo:(NSString *) receiver
           carbonCopy:(NSString *) carbonCopy
              subject:(NSString *) subject
                 body:(NSString *) body;
{
    NSMutableString *url;
    NSMutableArray *arguments;
    NSString *urlCodedString;
    
    OBPRECONDITION(receiver);
    
    url = [NSMutableString stringWithString:@"mailto:"];
    urlCodedString = [NSString encodeURLString:receiver asQuery:NO leaveSlashes:NO leaveColons:NO];
    [url appendString: urlCodedString];
    
    arguments = [NSMutableArray array];
    if (carbonCopy) {
        urlCodedString = [NSString encodeURLString:carbonCopy asQuery:NO leaveSlashes:NO leaveColons:NO];
        [arguments addObject:[NSString stringWithFormat:@"cc=%@", urlCodedString]];
    }
    
    if (subject) {
        urlCodedString = [NSString encodeURLString:subject asQuery:NO leaveSlashes:NO leaveColons:NO];
        [arguments addObject:[NSString stringWithFormat:@"subject=%@", urlCodedString]];
    }
    
    if (body) {
        urlCodedString = [NSString encodeURLString:body asQuery:NO leaveSlashes:NO leaveColons:NO];
        [arguments addObject:[NSString stringWithFormat:@"body=%@", urlCodedString]];
    }
    
        
    if ([arguments count]) {
        [url appendString:@"?"];
        [url appendString:[arguments componentsJoinedByString:@"&"]];
    }

    NS_DURING {
        NSLog(@"url=%@", url);
        [self launchURL:url];
    } NS_HANDLER {
        NSString *message;
        NSString *sendWithMail;
        NSString *dontSendMail;
        int choice;
        
        message = NSLocalizedStringFromTableInBundle(@"Error sending mail via InternetConfig.\nYou may need to select your prefered Mail application in System Preferences.", @"OmniAppKit", [self bundle], InternetConfig mail error);
        sendWithMail = NSLocalizedStringFromTableInBundle(@"Use built-in Mail application", @"OmniAppKit", [self bundle], Send with Mail button);
        dontSendMail = NSLocalizedStringFromTableInBundle(@"Don't send mail", @"OmniAppKit", [self bundle], Do not send mail button);
        
        choice = NSRunAlertPanel(nil, message, sendWithMail, dontSendMail, nil);
        if (choice == NSAlertDefaultReturn) {
            extern BOOL MVLaunchComposeWindow(NSAttributedString *messageText, NSDictionary *headerDictionary);
            NSMutableDictionary *headerDictionary;
            NSAttributedString *bodyAttr;
            
            headerDictionary = [NSMutableDictionary dictionary];
            [headerDictionary setObject:receiver forKey:@"to"];
            [headerDictionary setObject:subject forKey:@"subject"];
        
            bodyAttr = [[[NSAttributedString alloc] initWithString:body] autorelease];
            if (!MVLaunchComposeWindow(bodyAttr, headerDictionary)) {
                [NSException raise:@"Mail Error" format:NSLocalizedStringFromTableInBundle(@"Unable to open a Compose window in Mail to send mail to %@.", @"OmniAppKit", [self bundle], error message), receiver];
            }
        }
    } NS_ENDHANDLER;
}

@end

@implementation OAInternetConfig (Private)

- (NSString *)_helperKeyNameForScheme:(NSString *)scheme;
{
    return [@"HELPER" stringByAppendingString:scheme];
}


@end
