// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAApplication.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <Carbon/Carbon.h>

#import "NSView-OAExtensions.h"
#import "NSImage-OAExtensions.h"
#import "OAAppkitQueueProcessor.h"

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/OAApplication.m,v 1.36 2001/02/28 21:12:55 kc Exp $")

@interface OAApplication (Private)
- (void)processMouseButtonsChangedEvent:(NSEvent *)event;
- (void)setupDynamicMenus:(NSMenu *)aMenu;
- (void)makeDynamicItemGroupInMenu:(NSMenu *)aMenu forRangeOfItems:(NSRange)aRange;
@end

#if 0
// not needed any more?
@interface NSWindow (OAApplicationExtensions)
- (id)_addToOrderedWindowsArray;
@end
#endif

@implementation OAApplication

+ (void)setupOmniApplication;
{
    [OBPostLoader processClasses];
    
    // make this image available to client nibs and whatnot
    [NSImage imageNamed:@"OAHelpIcon" inBundleForClass:self];
}

+ (NSApplication *)sharedApplication;
{
    static OAApplication *omniApplication = nil;

    if (omniApplication)
        return omniApplication;

    omniApplication = (id)[super sharedApplication];
    [OAApplication setupOmniApplication];
    return omniApplication;
}

- (void)finishLaunching;
{
    [super finishLaunching];
    [self setupDynamicMenus:[self mainMenu]];
}

- (void)run;
{
    BOOL stillRunning = YES;

    exceptionCount = 0;
    exceptionCheckpointDate = [[NSDate alloc] init];
    do {
        NS_DURING {
            [super run];
            stillRunning = NO;
        } NS_HANDLER {
            if (++exceptionCount % 300 == 0) {
                if ([exceptionCheckpointDate timeIntervalSinceNow] > -3.0)
                    stillRunning = NO; // 300 exceptions in 3 seconds: abort
                [exceptionCheckpointDate release];
                exceptionCheckpointDate = [[NSDate alloc] init];
                exceptionCount = 0;
            }
            if (localException) {
                if (_appFlags._hasBeenRun)
                    [self handleRunException:localException];
                else
                    [self handleInitException:localException];
            }
        } NS_ENDHANDLER;
    } while (stillRunning && _appFlags._hasBeenRun);
}

#define OASystemDefinedEvent_MouseButtonsChangedSubType 7

- (void)sendEvent:(NSEvent *)event;
{
    // The -timestamp method on NSEvent doesn't seem to return an NSTimeInterval based off the same reference date as NSDate (which is what we want).
    lastEventTimeInterval = [NSDate timeIntervalSinceReferenceDate];

    NS_DURING {
        switch ([event type]) {
            case NSSystemDefined:
                if ([event subtype] == OASystemDefinedEvent_MouseButtonsChangedSubType)
                    [self processMouseButtonsChangedEvent:event];
                [super sendEvent:event];
                break;
            case NSScrollWheel:
            {
                NSView *contentView;
                NSView *viewUnderMouse;
                NSScrollView *scrollView;
                BOOL scrollWheelButtonIsDown;
    
                contentView = [[event window] contentView];
                viewUnderMouse = [contentView hitTest:[contentView convertPoint:[event locationInWindow] fromView:nil]];
                scrollView = [viewUnderMouse enclosingScrollView];
                scrollWheelButtonIsDown = [self scrollWheelButtonIsDown];
                if (scrollView == nil) {
                    // Punt
                    [super sendEvent:event];
                    if (scrollWheelButtonIsDown) {
                        // Multiply the strength of the original event
                        [super sendEvent:event];
                        [super sendEvent:event];
                        [super sendEvent:event];
                        [super sendEvent:event];
                    }
                } else {
                    if (scrollWheelButtonIsDown) {
                        [scrollView scrollDownByPages:-[event deltaY]];
                    } else {
                        [scrollView scrollDownByLines:2.0 * -[event deltaY]];
                    }
                }
                break;
            }
            case 25: // Other mouse button down
            case 26: // Other mouse button up
            case 27: // Other mouse button dragged
                // Other mouse buttons currently generate these events, but the AppKit doesn't know how to deal with them and logs a message for each one.  Rather than getting lots of log messages, we'll filter them.
                break;
            default:
                [super sendEvent:event];
                break;
        }
    } NS_HANDLER {
        if ([[localException name] isEqualToString:NSAbortModalException] || [[localException name] isEqualToString:NSAbortPrintingException])
            [localException raise];
        [self handleRunException:localException];
    } NS_ENDHANDLER;
}

- (void)handleInitException:(NSException *)anException;
{
    id delegate;

    delegate = [self delegate];
    if ([delegate respondsToSelector:@selector(handleInitException:)]) {
        [delegate handleInitException:anException];
    } else {
        NSLog(@"%@", [anException reason]);
    }
}

- (void)handleRunException:(NSException *)anException;
{
    id delegate;

    delegate = [self delegate];
    if ([delegate respondsToSelector:@selector(handleRunException:)]) {
        [delegate handleRunException:anException];
    } else {
        NSLog(@"%@", [anException reason]);
        NSRunAlertPanel(nil, @"%@", nil, nil, nil, [anException reason]);
    }
}

- (NSTimeInterval)lastEventTimeInterval;
{
    return lastEventTimeInterval;
}

- (BOOL)mouseButtonIsDownAtIndex:(unsigned int)mouseButtonIndex;
{
    return (mouseButtonState & (1 << mouseButtonIndex)) != 0;
}

- (BOOL)scrollWheelButtonIsDown;
{
    return [self mouseButtonIsDownAtIndex:2];
}

#if 0
// This method is now supplied by the AppKit's applescript support category

static NSMutableArray *_orderedWindows = nil;

- (NSArray *)orderedWindows;
    // Note:  this method would need a lock to be thread-safe
{
    NSArray *orderedWindows;

    _orderedWindows = [[NSMutableArray alloc] init];
    [self makeWindowsPerform:@selector(_addToOrderedWindowsArray) inOrder:YES];
    orderedWindows = [[NSArray alloc] initWithArray:_orderedWindows];
    [_orderedWindows release];
    _orderedWindows = nil;
    return [orderedWindows autorelease];
}
#endif

// Actions

- (IBAction)closeAllMainWindows:(id)sender;
{
    NSArray *windows;
    unsigned int windowIndex, windowCount;
    
    windows = [[NSArray alloc] initWithArray:[self orderedWindows]];
    windowCount = [windows count];
    for (windowIndex = 0; windowIndex < windowCount; windowIndex++) {
        NSWindow *window;

        window = [windows objectAtIndex:windowIndex];
        if ([window canBecomeMainWindow]) 
            if ([[window delegate] windowShouldClose:nil])
                [window close];
    }
}

- (IBAction)miniaturizeAll:(id)sender;
{
    NSArray *windows;
    unsigned int windowIndex, windowCount;
    
    windows = [[NSArray alloc] initWithArray:[self orderedWindows]];
    windowCount = [windows count];
    for (windowIndex = 0; windowIndex < windowCount; windowIndex++) {
        NSWindow *window;

        window = [windows objectAtIndex:windowIndex];
        if ([window styleMask] & NSMiniaturizableWindowMask) 
            [window miniaturize:nil];
    }
}

- (IBAction)cycleToNextMainWindow:(id)sender;
{
    NSWindow *mainWindow;
    NSArray *orderedWindows;
    unsigned int windowIndex, windowCount;
    
    mainWindow = [NSApp mainWindow];
    orderedWindows = [NSApp orderedWindows];
    windowCount = [orderedWindows count];
    for (windowIndex = 0; windowIndex < windowCount; windowIndex++) {
        NSWindow *window;

        window = [orderedWindows objectAtIndex:windowIndex];
        if (window != mainWindow && [window canBecomeMainWindow]) {
            [window makeKeyAndOrderFront:nil];
            [mainWindow orderBack:nil];
            return;
        }
    }
    // There's one (or less) window which can potentially be main, make it key and bring it forward.
    [mainWindow makeKeyAndOrderFront:nil];
}

- (IBAction)cycleToPreviousMainWindow:(id)sender;
{
    NSWindow *mainWindow;
    NSArray *orderedWindows;
    unsigned int windowIndex;
    
    mainWindow = [NSApp mainWindow];
    orderedWindows = [NSApp orderedWindows];
    windowIndex = [orderedWindows count];
    while (windowIndex--) {
        NSWindow *window;

        window = [orderedWindows objectAtIndex:windowIndex];
        if (window != mainWindow && [window canBecomeMainWindow]) {
            [window makeKeyAndOrderFront:nil];
            return;
        }
    }
    // There's one (or less) window which can potentially be main, make it key and bring it forward.
    [mainWindow makeKeyAndOrderFront:nil];
}

@end

@implementation OAApplication (Private)

- (void)processMouseButtonsChangedEvent:(NSEvent *)event;
{
    mouseButtonState = [event data2];
}

- (void)setupDynamicMenus:(NSMenu *)aMenu;
{
    int i;
    NSRange range = NSMakeRange(0,0);
    BOOL inRange = NO;
    NSMenuItem *currentItem;
    
    for (i = 0; i < [aMenu numberOfItems]; i++) {
        currentItem = [aMenu itemAtIndex:i];
        if ([currentItem hasSubmenu])
            [self setupDynamicMenus:[currentItem submenu]];
        if ([currentItem tag] == 1) {
            if (!inRange) {
                range.location = i;
                inRange = YES;
            }
            range.length++;
        } else if ([currentItem tag] == 0) {
            if (inRange) {
                //NSLog(@"dynamic range: location %i, length %i", range.location, range.length);
                [self makeDynamicItemGroupInMenu:aMenu forRangeOfItems:range];
                inRange = NO;
                range = NSMakeRange(0,0);
            }
        }
    }

}

- (void)makeDynamicItemGroupInMenu:(NSMenu *)aMenu forRangeOfItems:(NSRange)aRange;
{
    extern MenuRef GetCarbonMenu(NSMenu *);
    MenuRef menu;
    int item;
    
    menu = GetCarbonMenu(aMenu);
    aRange.location += 1; // Carbon menu indexes include the menu itself at index 0 
    for (item = aRange.location; item < aRange.location+aRange.length; item++) {
        //NSLog(@"making item %i dynamic", item);
        ChangeMenuItemAttributes( menu, item, kMenuItemAttrDynamic, 0 );
    }
}

@end

#if 0
// not needed any more?
@implementation NSWindow (OAApplicationExtensions)

- (id)_addToOrderedWindowsArray;
{
    [_orderedWindows addObject:self];
    return nil; // Returns nil so -[NSApplication makeWindowsPerform:inOrder:] continues to the next window
}

@end
#endif

