// Copyright 2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OACalendarView.h"

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OACalendarView.m,v 1.6 2001/03/26 11:11:43 kc Exp $")

// View size seems to be 155x133
// Buttons aren't properly positioned if the view size changes.

@implementation OACalendarView

// We need to have an NSActionCell (or subclass of that) to handle the target/action stuff, otherwise you just can't
// set these values.  We could subclass NSActionCell and have that do most of the calendar drawing.

+ (Class)cellClass;
{
    return [NSActionCell class];
}

- (id)initWithFrame:(NSRect)frameRect;
{
    NSDateFormatter *monthFormatter, *yearFormatter;
    int index;
    NSUserDefaults *defaults;
    NSArray *shortWeekDays;
    NSRect buttonFrame;
    NSButton *button;
    float monthHeight;

    if ([super initWithFrame:frameRect] == nil)
        return nil;

    monthTextFieldCell = [[NSTextFieldCell alloc] init];
    monthFormatter = [[NSDateFormatter alloc] initWithDateFormat:@"%B" allowNaturalLanguage:NO];
    [monthTextFieldCell setFormatter:monthFormatter];
    [monthFormatter release];

    yearTextFieldCell = [[NSTextFieldCell alloc] init];
    yearFormatter = [[NSDateFormatter alloc] initWithDateFormat:@"%Y" allowNaturalLanguage:NO];
    [yearTextFieldCell setFormatter:yearFormatter];
    [yearFormatter release];

    defaults = [NSUserDefaults standardUserDefaults];
    shortWeekDays = [defaults objectForKey:NSShortWeekDayNameArray];
    for (index = 0; index < 7; index++) {
        dayOfWeekCell[index] = [[NSTableHeaderCell alloc] init];
        [dayOfWeekCell[index] setAlignment:NSCenterTextAlignment];
        [dayOfWeekCell[index] setStringValue:[[shortWeekDays objectAtIndex:index] substringToIndex:1]];
    }

    dayOfMonthCell = [[NSTextFieldCell alloc] init];
    [dayOfMonthCell setAlignment:NSCenterTextAlignment];
    [dayOfMonthCell setFont:[NSFont controlContentFontOfSize:11]];

    columnWidth = [self _columnWidth];
    rowHeight = [self _rowHeight];

    [self setVisibleMonth:[NSCalendarDate calendarDate]];

    // Add left/right buttons to month and year

    buttonFrame = NSMakeRect(0, 0, 15, 15);
    button = [[NSButton alloc] initWithFrame:buttonFrame];
    [button setBezelStyle:NSShadowlessSquareBezelStyle];
    [button setImagePosition:NSImageOnly];
    [button setTarget:self];
    [button setAction:@selector(previousMonth:)];
    [button setContinuous:YES];
    [self addSubview:button];
    [button release];

    buttonFrame = NSMakeRect(frameRect.size.width - 15, 0, 15, 15);
    button = [[NSButton alloc] initWithFrame:buttonFrame];
    [button setBezelStyle:NSShadowlessSquareBezelStyle];
    [button setImagePosition:NSImageOnly];
    [button setTarget:self];
    [button setAction:@selector(nextMonth:)];
    [button setContinuous:YES];
    [self addSubview:button];
    [button release];

    monthHeight = [monthTextFieldCell cellSize].height;
    //NSLog(@"monthHeight: %f", monthHeight); // Is 16

    buttonFrame = NSMakeRect(0, monthHeight, 15, 15);
    button = [[NSButton alloc] initWithFrame:buttonFrame];
    [button setBezelStyle:NSShadowlessSquareBezelStyle];
    [button setImagePosition:NSImageOnly];
    [button setTarget:self];
    [button setAction:@selector(previousYear:)];
    [button setContinuous:YES];
    [self addSubview:button];
    [button release];

    buttonFrame = NSMakeRect(frameRect.size.width - 15, monthHeight, 15, 15);
    button = [[NSButton alloc] initWithFrame:buttonFrame];
    [button setBezelStyle:NSShadowlessSquareBezelStyle];
    [button setImagePosition:NSImageOnly];
    [button setTarget:self];
    [button setAction:@selector(nextYear:)];
    [button setContinuous:YES];
    [self addSubview:button];
    [button release];

    [self setSelectedDay:[NSCalendarDate calendarDate]];

    //[self sizeToFit];
    //NSLog(@"frame: %@", NSStringFromRect([self frame]));

    return self;
}

- (void)dealloc;
{
    int index;

    [dayOfMonthCell release];

    for (index = 0; index < 7; index++)
        [dayOfWeekCell[index] release];

    [yearTextFieldCell release];
    [monthTextFieldCell release];
    [selectedDay release];
    [visibleMonth release];

    [super dealloc];
}

- (NSCalendarDate *)visibleMonth;
{
    return visibleMonth;
}

- (void)setVisibleMonth:(NSCalendarDate *)aDate;
{
    [visibleMonth release];
    visibleMonth = [[aDate firstDayOfMonth] retain]; // [firstDayOfMonth isEqual:[firstDayOfMonth safeReferenceDate]];
    [monthTextFieldCell setObjectValue:visibleMonth];
    [yearTextFieldCell setObjectValue:visibleMonth];

    [self updateHighlightMask];
    [self setNeedsDisplay:YES];
}

- (NSCalendarDate *)selectedDay;
{
    return selectedDay;
}

- (void)setSelectedDay:(NSCalendarDate *)newSelectedDay;
{
    [selectedDay release];
    selectedDay = [newSelectedDay retain];
    [self setNeedsDisplay:YES];
}

- (int)dayHighlightMask;
{
    return dayHighlightMask;
}

- (void)setDayHighlightMask:(int)newMask;
{
    dayHighlightMask = newMask;
    [self setNeedsDisplay:YES];
}

- (void)updateHighlightMask;
{
    if ([[self target] respondsToSelector:@selector(calendarView:highlightMaskForVisibleMonth:)] == YES) {
        int mask;
        mask = [(id <OACalendarViewDataSourceProtocol>)[self target] calendarView:self highlightMaskForVisibleMonth:visibleMonth];
        [self setDayHighlightMask:mask];
    } else
        [self setDayHighlightMask:0];

    [self setNeedsDisplay:YES];
}

- (void)sizeToFit;
{
    NSSize monthSize, yearSize;
    NSSize headerSize;
    NSSize minimumSize;

    monthSize = [monthTextFieldCell cellSize];
    yearSize = [yearTextFieldCell cellSize];
    headerSize = [dayOfWeekCell[0] cellSize];

    minimumSize.height = monthSize.height + yearSize.height + headerSize.height + 6 * rowHeight;
    // This should really check the lengths of the months, and include space for the buttons.
    minimumSize.width = [self _calendarWidth];

    [self setFrameSize:minimumSize];
    [self setNeedsDisplay:YES];
}

- (BOOL)isFlipped;
{
    return YES;
}

- (void)drawRect:(NSRect)rect;
{
    NSSize cellSize;
    NSRect viewBounds;
    NSRect topRect, bottomRect;
    NSRect centerRect;
    NSRect discardRect;
    NSRect leftRect;
    int index;
    NSRect grayRect;

    [[NSColor windowBackgroundColor] set];
    //[[NSColor blueColor] set];
    NSRectFill(rect);

    viewBounds = [self bounds];

    cellSize = [monthTextFieldCell cellSize];
    NSDivideRect(viewBounds, &topRect, &bottomRect, ceil(cellSize.height), NSMinYEdge);
    NSDivideRect(topRect, &discardRect, &centerRect, floor((viewBounds.size.width - cellSize.width) / 2), NSMinXEdge);
    centerRect.size.width = cellSize.width;
    [monthTextFieldCell drawWithFrame:centerRect inView:self];

    cellSize = [yearTextFieldCell cellSize];
    NSDivideRect(bottomRect, &topRect, &bottomRect, ceil(cellSize.height), NSMinYEdge);
    NSDivideRect(topRect, &discardRect, &centerRect, floor((viewBounds.size.width - cellSize.width) / 2), NSMinXEdge);
    centerRect.size.width = cellSize.width;
    [yearTextFieldCell drawWithFrame:centerRect inView:self];

    NSDivideRect(bottomRect, &discardRect, &bottomRect, floor((viewBounds.size.width - [self _calendarWidth]) / 2), NSMinXEdge);
    bottomRect.size.width = [self _calendarWidth];

    grayRect = bottomRect;
    NSDivideRect(bottomRect, &discardRect, &bottomRect, 1, NSMinXEdge);

    cellSize = [dayOfWeekCell[0] cellSize];
    NSDivideRect(bottomRect, &topRect, &bottomRect, ceil(cellSize.height), NSMinYEdge);

    grayRect.size.height = topRect.size.height + 6 * rowHeight; // Don't need extra pixel for grid
    [[NSColor gridColor] set];
    NSRectFill(grayRect);

    for (index = 0; index < 7; index++) {
        NSDivideRect(topRect, &leftRect, &topRect, columnWidth, NSMinXEdge);
        [dayOfWeekCell[index] drawWithFrame:leftRect inView:self];
    }

    // The table header cells draw 7 extra pixels of shadow below, but this isn't part of their size.
    // We just overwrite this, since we don't want the shadow anyway.

    bottomRect.size.height = 6 * rowHeight - 1;

    [[NSColor controlBackgroundColor] set];
    //[[NSColor blueColor] set];
    NSRectFill(bottomRect);

    [self drawDaysOfMonthInRect:bottomRect];
    [self drawGridInRect:bottomRect];
}

- (void)drawDaysOfMonthInRect:(NSRect)rect;
{
    int row, column;
    int count, index;
    NSRect cellFrame;
    int day;

    cellFrame.size.height = rowHeight;
    cellFrame.size.width = columnWidth;

    if (selectedDay != nil && [selectedDay yearOfCommonEra] == [visibleMonth yearOfCommonEra] && [selectedDay monthOfYear] == [visibleMonth monthOfYear])
        day = [selectedDay dayOfMonth];
    else
        day = -1;

    row = 0;
    column = [visibleMonth dayOfWeek];
    count = [visibleMonth numberOfDaysInMonth];

    for (index = 0; index < count; index++) {
        cellFrame.origin.x = rect.origin.x + column * columnWidth;
        cellFrame.origin.y = rect.origin.y + row * rowHeight;

        [dayOfMonthCell setIntValue:index + 1];
        if (index + 1 == day) {
            //[[NSColor controlHighlightColor] set];
            [[NSColor selectedControlColor] set];
            NSRectFill(cellFrame);
        }
        if ((dayHighlightMask & (1 << index)) == 0) {
            [dayOfMonthCell setTextColor:[NSColor blackColor]];
        } else {
            [dayOfMonthCell setTextColor:[NSColor blueColor]];
        }

        [dayOfMonthCell drawWithFrame:cellFrame inView:self];

        column++;
        if (column > 6) {
            column = 0;
            row++;
        }
    }
}

- (void)drawGridInRect:(NSRect)rect;
{
    NSRect rightRect, discardRect;
    int index;

    rightRect = rect;
    NSDivideRect(rightRect, &rightRect, &discardRect, 1, NSMaxXEdge);

    [[NSColor controlHighlightColor] set];
    rect.size.height = 1;

    for (index = 0; index < 5; index++) {
        rect.origin.y += rowHeight;
        NSRectFill(rect);
    }

    [[NSColor gridColor] set];
    NSRectFill(rightRect);
}

- (float)_maxDayOfWeekWidth;
{
    float maxWidth;
    int index;
    NSSize cellSize;

    maxWidth = 0;
    for (index = 0; index < 7; index++) {
        cellSize = [dayOfWeekCell[index] cellSize];
        if (maxWidth < cellSize.width)
            maxWidth = cellSize.width;
    }

    return ceil(maxWidth);
}

- (NSSize)_maxDayOfMonthSize;
{
    NSSize maxSize;
    int index;
    NSString *str;
    NSSize cellSize;

    maxSize = NSZeroSize; // I'm sure the height doesn't change, but I need to know the height anyway.
    for (index = 1; index <= 31; index++) {
        str = [NSString stringWithFormat:@"%d", index];
        [dayOfMonthCell setStringValue:str];
        cellSize = [dayOfMonthCell cellSize];
        if (maxSize.width < cellSize.width)
            maxSize.width = cellSize.width;
        if (maxSize.height < cellSize.height)
            maxSize.height = cellSize.height;
    }

    maxSize.width = ceil(maxSize.width);
    maxSize.height = ceil(maxSize.height);

    return maxSize;
}

- (float)_columnWidth;
{
    float a, b;
    float width;

    a = [self _maxDayOfWeekWidth];
    b = [self _maxDayOfMonthSize].width;

    width = (a > b) ? a : b;
    //width += 5; // 2 pixels on either side, plus an extra one on the right for the grid

    return width;
}

- (float)_calendarWidth;
{
    return 7 * columnWidth + 1; // 1 pixel on the left for the border.  The last pixel of the last column is used for the right border.
}

- (float)_rowHeight;
{
    return [self _maxDayOfMonthSize].height;
}

- (NSRect)_calendarDaysRect;
{
    NSSize monthSize, yearSize, headerSize;
    NSRect calendarDaysRect, viewFrame;

    monthSize = [monthTextFieldCell cellSize];
    yearSize = [yearTextFieldCell cellSize];
    headerSize = [dayOfWeekCell[0] cellSize];

    viewFrame = [self frame];
    calendarDaysRect.origin.x = floor((viewFrame.size.width - [self _calendarWidth]) / 2);
    calendarDaysRect.origin.y = monthSize.height + yearSize.height + headerSize.height;
    calendarDaysRect.size.width = [self _calendarWidth];
    calendarDaysRect.size.height = 6 * rowHeight;

    return calendarDaysRect;
}

- (NSCalendarDate *)hitDateWithLocation:(NSPoint)targetPoint;
{
    NSRect calendarDaysRect;
    int hitRow, hitColumn;
    int firstDayOfWeek, targetDayOfMonth;

    calendarDaysRect = [self _calendarDaysRect];
    if (NSPointInRect(targetPoint, calendarDaysRect) == NO)
        return nil;

    firstDayOfWeek = [[visibleMonth firstDayOfMonth] dayOfWeek];

    hitRow = (targetPoint.y - calendarDaysRect.origin.y) / rowHeight;
    hitColumn = (targetPoint.x - calendarDaysRect.origin.x) / columnWidth;

    targetDayOfMonth = hitRow * 7 + hitColumn - firstDayOfWeek + 1;
    if (targetDayOfMonth < 1 || targetDayOfMonth > [visibleMonth numberOfDaysInMonth])
        return nil;

    return [NSCalendarDate dateWithYear:[visibleMonth yearOfCommonEra] month:[visibleMonth monthOfYear] day:targetDayOfMonth
                           hour:12 minute:0 second:0 timeZone:nil];
}

- (void)mouseDown:(NSEvent *)mouseEvent;
{
    NSCalendarDate *hitDate;
    NSPoint location;

    location = [self convertPoint:[mouseEvent locationInWindow] fromView:nil];
    hitDate = [self hitDateWithLocation:location];
    [self setSelectedDay:hitDate];
    [self sendAction:[self action] to:[self target]];
}

//
// Actions
//

- (IBAction)previousMonth:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:0 months:-1 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

- (IBAction)nextMonth:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:0 months:1 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

- (IBAction)previousYear:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:-1 months:0 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

- (IBAction)nextYear:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:1 months:0 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

@end
