// Copyright 2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OACalendarView.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import <OmniAppKit/NSImage-OAExtensions.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Widgets.subproj/OACalendarView.m,v 1.13 2001/06/19 05:38:03 krevis Exp $")


@interface OACalendarView (Private)

- (NSButton *)_createButtonWithFrame:(NSRect)buttonFrame;

- (void)_drawDaysOfMonthInRect:(NSRect)rect;
- (void)_drawGridInRect:(NSRect)rect;

- (float)_maxDayOfWeekWidth;
- (NSSize)_maxDayOfMonthSize;
- (float)_columnWidth;
- (float)_calendarWidth;
- (float)_rowHeight;

- (NSRect)_calendarDaysRect;
- (NSCalendarDate *)_hitDateWithLocation:(NSPoint)targetPoint;

- (NSRect)_headerDaysRect;
- (NSCalendarDate *)_hitWeekdayWithLocation:(NSPoint)targetPoint;

@end


@implementation OACalendarView

const float OACalendarViewButtonWidth = 15.0;
const float OACalendarViewButtonHeight = 15.0;
const float OACalendarViewSpaceBetweenMonthYearAndGrid = 6.0;

//
// Init / dealloc
//

- (id)initWithFrame:(NSRect)frameRect;
{
    // The view size comes out to be 155 x 123.
    // Note that the buttons aren't repositioned if we are resized after being instantiated.

    NSDateFormatter *monhAndYearFormatter;
    int index;
    NSUserDefaults *defaults;
    NSArray *shortWeekDays;
    NSRect buttonFrame;
    NSButton *button;

    if ([super initWithFrame:frameRect] == nil)
        return nil;

    monthAndYearTextFieldCell = [[NSTextFieldCell alloc] init];
    monhAndYearFormatter = [[NSDateFormatter alloc] initWithDateFormat:@"%B %Y" allowNaturalLanguage:NO];
    [monthAndYearTextFieldCell setFormatter:monhAndYearFormatter];
    [monhAndYearFormatter release];

    defaults = [NSUserDefaults standardUserDefaults];
    shortWeekDays = [defaults objectForKey:NSShortWeekDayNameArray];
    for (index = 0; index < 7; index++) {
        dayOfWeekCell[index] = [[NSTableHeaderCell alloc] init];
        [dayOfWeekCell[index] setAlignment:NSCenterTextAlignment];
        [dayOfWeekCell[index] setStringValue:[[shortWeekDays objectAtIndex:index] substringToIndex:1]];
    }

    dayOfMonthCell = [[NSTextFieldCell alloc] init];
    [dayOfMonthCell setAlignment:NSCenterTextAlignment];
    [dayOfMonthCell setFont:[NSFont controlContentFontOfSize:11]];

    columnWidth = [self _columnWidth];
    rowHeight = [self _rowHeight];

    buttons = [[NSMutableArray alloc] initWithCapacity:2];

    // Add left/right buttons

    buttonFrame = NSMakeRect(0, 0, OACalendarViewButtonWidth, OACalendarViewButtonHeight);
    button = [self _createButtonWithFrame:buttonFrame];
    [button setImage:[NSImage imageNamed:@"OALeftArrow" inBundleForClass:isa]];
    [button setAlternateImage:[NSImage imageNamed:@"OALeftArrowPressed" inBundleForClass:isa]];
    [button setAction:@selector(previousMonth:)];

    buttonFrame = NSMakeRect(frameRect.size.width - OACalendarViewButtonWidth, 0, OACalendarViewButtonWidth, OACalendarViewButtonHeight);
    button = [self _createButtonWithFrame:buttonFrame];
    [button setImage:[NSImage imageNamed:@"OARightArrow" inBundleForClass:isa]];
    [button setAlternateImage:[NSImage imageNamed:@"OARightArrowPressed" inBundleForClass:isa]];
    [button setAction:@selector(nextMonth:)];

    // [self sizeToFit];
    // NSLog(@"frame: %@", NSStringFromRect([self frame]));

    [self setVisibleMonth:[NSCalendarDate calendarDate]];
    [self setSelectedDay:[NSCalendarDate calendarDate]];

    return self;
}

- (void)dealloc;
{
    int index;

    [dayOfMonthCell release];

    for (index = 0; index < 7; index++)
        [dayOfWeekCell[index] release];

    [monthAndYearTextFieldCell release];
    [buttons release];
    [selectedDay release];
    [visibleMonth release];

    [super dealloc];
}


//
// NSControl overrides
//

+ (Class)cellClass;
{
    // We need to have an NSActionCell (or subclass of that) to handle the target and action; otherwise, you just can't set those values.
    return [NSActionCell class];
}

- (void)setEnabled:(BOOL)flag;
{
    unsigned int buttonIndex;

    [super setEnabled:flag];
    
    buttonIndex = [buttons count];
    while (buttonIndex--)
        [[buttons objectAtIndex:buttonIndex] setEnabled:flag];
}

- (void)sizeToFit;
{
    NSSize monthAndYearSize;
    NSSize headerSize;
    NSSize minimumSize;

    monthAndYearSize = [monthAndYearTextFieldCell cellSize];
    headerSize = [dayOfWeekCell[0] cellSize];

    minimumSize.height = monthAndYearSize.height + OACalendarViewSpaceBetweenMonthYearAndGrid + headerSize.height + 6 * rowHeight;
    // This should really check the lengths of the months, and include space for the buttons.
    minimumSize.width = [self _calendarWidth];

    [self setFrameSize:minimumSize];
    [self setNeedsDisplay:YES];
}


//
// NSView overrides
//

- (BOOL)isFlipped;
{
    return YES;
}

- (void)drawRect:(NSRect)rect;
{
    NSSize cellSize;
    NSRect viewBounds;
    NSRect topRect, bottomRect;
    NSRect centerRect;
    NSRect discardRect;
    NSRect leftRect;
    int index;
    NSRect grayRect;

    viewBounds = [self bounds];

    cellSize = [monthAndYearTextFieldCell cellSize];
    NSDivideRect(viewBounds, &topRect, &bottomRect, ceil(cellSize.height + OACalendarViewSpaceBetweenMonthYearAndGrid), NSMinYEdge);
    NSDivideRect(topRect, &discardRect, &centerRect, floor((viewBounds.size.width - cellSize.width) / 2), NSMinXEdge);
    centerRect.size.width = cellSize.width;
    [monthAndYearTextFieldCell drawWithFrame:centerRect inView:self];

    NSDivideRect(bottomRect, &discardRect, &bottomRect, floor((viewBounds.size.width - [self _calendarWidth]) / 2), NSMinXEdge);
    bottomRect.size.width = [self _calendarWidth];

    grayRect = bottomRect;
    NSDivideRect(bottomRect, &discardRect, &bottomRect, 1, NSMinXEdge);

    cellSize = [dayOfWeekCell[0] cellSize];
    NSDivideRect(bottomRect, &topRect, &bottomRect, ceil(cellSize.height), NSMinYEdge);

    grayRect.size.height = topRect.size.height + 6 * rowHeight; // Don't need extra pixel for grid
    [[NSColor gridColor] set];
    NSRectFill(grayRect);

    for (index = 0; index < 7; index++) {
        NSDivideRect(topRect, &leftRect, &topRect, columnWidth, NSMinXEdge);
        [dayOfWeekCell[index] drawWithFrame:leftRect inView:self];
    }

    // The table header cells draw 7 extra pixels of shadow below, but this isn't part of their size.
    // We just overwrite this, since we don't want the shadow anyway.

    bottomRect.size.height = 6 * rowHeight - 1;

    [[NSColor controlBackgroundColor] set];
    NSRectFill(bottomRect);

    [self _drawDaysOfMonthInRect:bottomRect];
    [self _drawGridInRect:bottomRect];
}

- (void)mouseDown:(NSEvent *)mouseEvent;
{
    if ([self isEnabled]) {
        NSCalendarDate *hitDate;
        NSPoint location;
    
        location = [self convertPoint:[mouseEvent locationInWindow] fromView:nil];
        hitDate = [self _hitDateWithLocation:location];
        if (hitDate) {
            [self setSelectedDay:hitDate];
            [self setVisibleMonth:hitDate];
            [self sendAction:[self action] to:[self target]];
        } else if (selectionType == OACalendarViewSelectByWeekday) {
            NSCalendarDate *hitWeekday;
            
            hitWeekday = [self _hitWeekdayWithLocation:location];
            if (hitWeekday) {
                [self setSelectedDay:hitWeekday];
                [self sendAction:[self action] to:[self target]];
            }
        }
    }
}


//
// API
//

- (NSCalendarDate *)visibleMonth;
{
    return visibleMonth;
}

- (void)setVisibleMonth:(NSCalendarDate *)aDate;
{
    [visibleMonth release];
    visibleMonth = [[aDate firstDayOfMonth] retain];
    [monthAndYearTextFieldCell setObjectValue:visibleMonth];

    [self updateHighlightMask];
    [self setNeedsDisplay:YES];
}

- (NSCalendarDate *)selectedDay;
{
    return selectedDay;
}

- (void)setSelectedDay:(NSCalendarDate *)newSelectedDay;
{
    [selectedDay release];
    selectedDay = [newSelectedDay retain];
    [self setNeedsDisplay:YES];
}

- (int)dayHighlightMask;
{
    return dayHighlightMask;
}

- (void)setDayHighlightMask:(int)newMask;
{
    dayHighlightMask = newMask;
    [self setNeedsDisplay:YES];
}

- (void)updateHighlightMask;
{
    if ([[self target] respondsToSelector:@selector(calendarView:highlightMaskForVisibleMonth:)] == YES) {
        int mask;
        mask = [(id <OACalendarViewDataSourceProtocol>)[self target] calendarView:self highlightMaskForVisibleMonth:visibleMonth];
        [self setDayHighlightMask:mask];
    } else
        [self setDayHighlightMask:0];

    [self setNeedsDisplay:YES];
}

- (BOOL)showsDaysForOtherMonths;
{
    return flags.showsDaysForOtherMonths;
}

- (void)setShowsDaysForOtherMonths:(BOOL)value;
{
    if (value != flags.showsDaysForOtherMonths) {
        flags.showsDaysForOtherMonths = value;

        [self setNeedsDisplay:YES];
    }
}

- (OACalendarViewSelectionType)selectionType;
{
    return selectionType;
}

- (void)setSelectionType:(OACalendarViewSelectionType)value;
{
    if (selectionType != value) {
        selectionType = value;

        [self setNeedsDisplay:YES];
    }
}

- (NSArray *)selectedDays;
{
    if (!selectedDay)
        return nil;

    switch (selectionType) {
        case OACalendarViewSelectByDay:
            return [NSArray arrayWithObject:selectedDay];
            break;
            
        case OACalendarViewSelectByWeek:
            {
                NSMutableArray *days;
                NSCalendarDate *day;
                int index;
                
                days = [NSMutableArray arrayWithCapacity:7];
                day = [selectedDay dateByAddingYears:0 months:0 days:-[selectedDay dayOfWeek] hours:0 minutes:0 seconds:0];
                for (index = 0; index < 7; index++) {
                    NSCalendarDate *nextDay;

                    nextDay = [day dateByAddingYears:0 months:0 days:index hours:0 minutes:0 seconds:0];
                    if (flags.showsDaysForOtherMonths || [nextDay monthOfYear] == [selectedDay monthOfYear])
                        [days addObject:nextDay];                    
                }
            
                return days;
            }            
            break;

        case OACalendarViewSelectByWeekday:
            {
                NSMutableArray *days;
                NSCalendarDate *day;
                int index;
                
                days = [NSMutableArray arrayWithCapacity:6];
                day = [selectedDay dateByAddingYears:0 months:0 days:-(([selectedDay weekOfMonth] - 1) * 7) hours:0 minutes:0 seconds:0];
                for (index = 0; index < 6; index++) {
                    NSCalendarDate *nextDay;

                    nextDay = [day dateByAddingYears:0 months:0 days:(index * 7) hours:0 minutes:0 seconds:0];
                    if (flags.showsDaysForOtherMonths || [nextDay monthOfYear] == [selectedDay monthOfYear])
                        [days addObject:nextDay];
                }

                return days;
            }
            break;
            
        default:
            NSLog(@"OACalendarView: Unknown selection type: %d", selectionType);
            return nil;
            break;
    }
}


//
// Actions
//

- (IBAction)previousMonth:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:0 months:-1 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

- (IBAction)nextMonth:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:0 months:1 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

- (IBAction)previousYear:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:-1 months:0 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

- (IBAction)nextYear:(id)sender;
{
    NSCalendarDate *newDate;

    newDate = [visibleMonth dateByAddingYears:1 months:0 days:0 hours:0 minutes:0 seconds:0];
    [self setVisibleMonth:newDate];
}

@end


@implementation OACalendarView (Private)

- (NSButton *)_createButtonWithFrame:(NSRect)buttonFrame;
{
    NSButton *button;
    
    button = [[NSButton alloc] initWithFrame:buttonFrame];
    [button setBezelStyle:NSShadowlessSquareBezelStyle];
    [button setBordered:NO];
    [button setImagePosition:NSImageOnly];
    [button setTarget:self];
    [button setContinuous:YES];
    [self addSubview:button];
    [buttons addObject:button];
    [button release];

    return button;
}

- (void)_drawDaysOfMonthInRect:(NSRect)rect;
{
    NSRect cellFrame;
    int visibleMonthIndex;
    NSCalendarDate *thisDay;
    int index, row, column;

    cellFrame.size.height = rowHeight;
    cellFrame.size.width = columnWidth;

    visibleMonthIndex = [visibleMonth monthOfYear];

    thisDay = [visibleMonth dateByAddingYears:0 months:0 days:-[visibleMonth dayOfWeek] hours:0 minutes:0 seconds:0];

    for (row = column = index = 0; index < 6 * 7; index++) {
        NSColor *textColor;
        BOOL isVisibleMonth;

        cellFrame.origin.x = rect.origin.x + column * columnWidth;
        cellFrame.origin.y = rect.origin.y + row * rowHeight;

        [dayOfMonthCell setIntValue:[thisDay dayOfMonth]];
        isVisibleMonth = ([thisDay monthOfYear] == visibleMonthIndex);

        if (flags.showsDaysForOtherMonths || isVisibleMonth) {
            if (selectedDay) {
                BOOL shouldHighlightThisDay = NO;

                // We could just check if thisDay is in [self selectedDays]. However, that makes the selection look somewhat weird when we
                // are selecting by weekday, showing days for other months, and the visible month is the previous/next from the selected day.
                // (Some of the weekdays are shown as highlighted, and later ones are not.)
                // So, we fib a little to make things look better.
                switch (selectionType) {
                    case OACalendarViewSelectByDay:
                        shouldHighlightThisDay = ([selectedDay dayOfCommonEra] == [thisDay dayOfCommonEra]);
                        break;
                        
                    case OACalendarViewSelectByWeek:
                        shouldHighlightThisDay = [selectedDay isInSameWeekAsDate:thisDay];
                        break;
                        
                    case OACalendarViewSelectByWeekday:
                        shouldHighlightThisDay = ([selectedDay monthOfYear] == visibleMonthIndex && [selectedDay dayOfWeek] == [thisDay dayOfWeek]);
                        break;
                        
                    default:
                        NSLog(@"OACalendarView: Unknown selection type: %d", selectionType);
                        break;
                }
                
                if (shouldHighlightThisDay) {
                    [[NSColor selectedControlColor] set];
                    NSRectFill(cellFrame);
                }
            }
    
            if ((dayHighlightMask & (1 << index)) == 0) {
                textColor = (isVisibleMonth ? [NSColor blackColor] : [NSColor grayColor]);
            } else {
                textColor = [NSColor blueColor];
            }
            [dayOfMonthCell setTextColor:textColor];
            [dayOfMonthCell drawWithFrame:cellFrame inView:self];
        }
        
        thisDay = [thisDay dateByAddingYears:0 months:0 days:1 hours:0 minutes:0 seconds:0];
        column++;
        if (column > 6) {
            column = 0;
            row++;
        }
    }
}

- (void)_drawGridInRect:(NSRect)rect;
{
    NSRect rightRect, discardRect;
    int index;

    rightRect = rect;
    NSDivideRect(rightRect, &rightRect, &discardRect, 1, NSMaxXEdge);

    [[NSColor controlHighlightColor] set];
    rect.size.height = 1;

    for (index = 0; index < 5; index++) {
        rect.origin.y += rowHeight;
        NSRectFill(rect);
    }

    [[NSColor gridColor] set];
    NSRectFill(rightRect);
}

- (float)_maxDayOfWeekWidth;
{
    float maxWidth;
    int index;

    maxWidth = 0;
    for (index = 0; index < 7; index++) {
        NSSize cellSize;

        cellSize = [dayOfWeekCell[index] cellSize];
        if (maxWidth < cellSize.width)
            maxWidth = cellSize.width;
    }

    return ceil(maxWidth);
}

- (NSSize)_maxDayOfMonthSize;
{
    NSSize maxSize;
    int index;

    maxSize = NSZeroSize; // I'm sure the height doesn't change, but I need to know the height anyway.
    for (index = 1; index <= 31; index++) {
        NSString *str;
        NSSize cellSize;

        str = [NSString stringWithFormat:@"%d", index];
        [dayOfMonthCell setStringValue:str];
        cellSize = [dayOfMonthCell cellSize];
        if (maxSize.width < cellSize.width)
            maxSize.width = cellSize.width;
        if (maxSize.height < cellSize.height)
            maxSize.height = cellSize.height;
    }

    maxSize.width = ceil(maxSize.width);
    maxSize.height = ceil(maxSize.height);

    return maxSize;
}

- (float)_columnWidth;
{
    float a, b;
    float width;

    a = [self _maxDayOfWeekWidth];
    b = [self _maxDayOfMonthSize].width;

    width = (a > b) ? a : b;
    //width += 5; // 2 pixels on either side, plus an extra one on the right for the grid

    return width;
}

- (float)_calendarWidth;
{
    return 7 * columnWidth + 1; 
    // 1 pixel on the left for the border.  The last pixel of the last column is used for the right border.
}

- (float)_rowHeight;
{
    return [self _maxDayOfMonthSize].height;
}

- (NSRect)_calendarDaysRect;
{
    NSSize monthAndYearSize, headerSize;
    NSRect calendarDaysRect, viewFrame;

    monthAndYearSize = [monthAndYearTextFieldCell cellSize];
    headerSize = [dayOfWeekCell[0] cellSize];

    viewFrame = [self frame];
    calendarDaysRect.origin.x = floor((viewFrame.size.width - [self _calendarWidth]) / 2);
    calendarDaysRect.origin.y = monthAndYearSize.height + OACalendarViewSpaceBetweenMonthYearAndGrid + headerSize.height;
    calendarDaysRect.size.width = [self _calendarWidth];
    calendarDaysRect.size.height = 6 * rowHeight;

    return calendarDaysRect;
}

- (NSCalendarDate *)_hitDateWithLocation:(NSPoint)targetPoint;
{
    NSRect calendarDaysRect;
    int hitRow, hitColumn;
    int firstDayOfWeek, targetDayOfMonth;

    calendarDaysRect = [self _calendarDaysRect];
    if (NSPointInRect(targetPoint, calendarDaysRect) == NO)
        return nil;

    firstDayOfWeek = [[visibleMonth firstDayOfMonth] dayOfWeek];

    hitRow = (targetPoint.y - calendarDaysRect.origin.y) / rowHeight;
    hitColumn = (targetPoint.x - calendarDaysRect.origin.x) / columnWidth;

    targetDayOfMonth = hitRow * 7 + hitColumn - firstDayOfWeek + 1;
    if (!flags.showsDaysForOtherMonths && (targetDayOfMonth < 1 || targetDayOfMonth > [visibleMonth numberOfDaysInMonth]))
        return nil;

    return [visibleMonth dateByAddingYears:0 months:0 days:targetDayOfMonth-1 hours:0 minutes:0 seconds:0];
}

- (NSRect)_headerDaysRect;
{
    NSSize monthAndYearSize, headerSize;
    NSRect headerDaysRect, viewFrame;

    monthAndYearSize = [monthAndYearTextFieldCell cellSize];
    headerSize = [dayOfWeekCell[0] cellSize];

    viewFrame = [self frame];
    headerDaysRect.origin.x = floor((viewFrame.size.width - [self _calendarWidth]) / 2);
    headerDaysRect.origin.y = monthAndYearSize.height + OACalendarViewSpaceBetweenMonthYearAndGrid;
    headerDaysRect.size.width = [self _calendarWidth];
    headerDaysRect.size.height = headerSize.height;

    return headerDaysRect;
}

- (NSCalendarDate *)_hitWeekdayWithLocation:(NSPoint)targetPoint;
{
    NSRect headerDaysRect;
    int hitDayOfWeek;
    int firstDayOfWeek, targetDayOfMonth;

    headerDaysRect = [self _headerDaysRect];
    if (NSPointInRect(targetPoint, headerDaysRect) == NO)
        return nil;

    hitDayOfWeek = (targetPoint.x - headerDaysRect.origin.x) / columnWidth;

    firstDayOfWeek = [[visibleMonth firstDayOfMonth] dayOfWeek];
    if (hitDayOfWeek >= firstDayOfWeek)
        targetDayOfMonth = hitDayOfWeek - firstDayOfWeek + 1;
    else
        targetDayOfMonth = hitDayOfWeek + 7 - firstDayOfWeek + 1;

    return [visibleMonth dateByAddingYears:0 months:0 days:targetDayOfMonth-1 hours:0 minutes:0 seconds:0];
}

@end
