// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAPreferenceController.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import "NSView-OAExtensions.h"
#import "OAPreferenceClient.h"
#import "OAPreferenceClientRecord.h"
#import "OAPreferencesIconView.h"
#import "OAPreferencesShowAllIconView.h"

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Preferences.subproj/OAPreferenceController.m,v 1.45.4.3 2001/05/31 00:00:08 kc Exp $")

@interface OAPreferenceController (Private)
- (void)_loadInterface;
- (void)_resetWindowTitle;
- (void)_setCurrentClientRecord:(OAPreferenceClientRecord *)clientRecord;
- (void)_showAllIcons;
- (void)_defaultsDidChange:(NSNotification *)notification;
- (void)_validateRestoreDefaultsButton;
//
- (NSArray *)_categoryNames;
- (void)_registerCategoryName:(NSString *)categoryName localizedName:(NSString *)localizedCategoryName priorityNumber:(NSNumber *)priorityNumber;
- (NSString *)_localizedCategoryNameForCategoryName:(NSString *)categoryName;
- (float)_priorityForCategoryName:(NSString *)categoryName;
//
- (void)_registerClassName:(NSString *)className inCategoryNamed:(NSString *)categoryName description:(NSDictionary *)description;
@end


@implementation OAPreferenceController

static NSString *windowFrameSaveName = @"Preferences";
static OAPreferenceController *sharedPreferenceController = nil;

// OFBundleRegistryTarget informal protocol

+ (void)registerItemName:(NSString *)itemName bundle:(NSBundle *)bundle description:(NSDictionary *)description;
{
    NSString *categoryName;
    OAPreferenceController *controller;
    
    [OFBundledClass createBundledClassWithName:itemName bundle:bundle description:description];

    if ((categoryName = [description objectForKey:@"category"]) == nil)
        categoryName = @"UNKNOWN";

    controller = [self sharedPreferenceController];
    [controller _registerCategoryName:categoryName localizedName:[bundle localizedStringForKey:categoryName value:@"" table:@"Preferences"] priorityNumber:[description objectForKey:@"categoryPriority"]];
    [controller _registerClassName:itemName inCategoryNamed:categoryName description:description];
}


// Init and dealloc

+ (OAPreferenceController *)sharedPreferenceController;
{
    if (sharedPreferenceController == nil)
        sharedPreferenceController = [[self alloc] init];
    
    return sharedPreferenceController;
}

- init;
{
    [super init];
	
    categoryNamesToClientRecordsArrays = [[NSMutableDictionary alloc] init];
    localizedCategoryNames = [[NSMutableDictionary alloc] init];
    categoryPriorities = [[NSMutableDictionary alloc] init];
    allClientRecords = [[NSMutableArray alloc] init];
    preferencesIconViews = [[NSMutableArray alloc] init];
    
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(_defaultsDidChange:) name:NSUserDefaultsDidChangeNotification object:nil];
    return self;
}

- (void)dealloc;
{
    OBPRECONDITION(NO);
}


// API

- (void)close;
{
    if ([window isVisible])
        [window performClose:nil];
}

- (void)setTitle:(NSString *)title;
{
    [window setTitle:title];
}

- (void)setCurrentClientByClassName:(NSString *)name;
{
    unsigned int clientRecordIndex;
    
    clientRecordIndex = [allClientRecords count];
    while (clientRecordIndex--) {
        OAPreferenceClientRecord *clientRecord;

        clientRecord = [allClientRecords objectAtIndex:clientRecordIndex];
        if ([[clientRecord className] isEqualToString:name]) {
            [self _setCurrentClientRecord:clientRecord];
            return;
        }
    }
}

- (NSArray *)allClientRecords;
{
    return allClientRecords;
}

- (OAPreferenceClientRecord *)clientRecordWithShortTitle:(NSString *)shortTitle;
{
    unsigned int clientRecordIndex;

    OBPRECONDITION(shortTitle != nil);
    clientRecordIndex = [allClientRecords count];
    while (clientRecordIndex--) {
        OAPreferenceClientRecord *clientRecord;

        clientRecord = [allClientRecords objectAtIndex:clientRecordIndex];
        if ([shortTitle isEqualToString:[clientRecord shortTitle]])
            return clientRecord;
    }
    return nil;
}

- (OAPreferenceClientRecord *)clientRecordWithIdentifier:(NSString *)identifier;
{
    unsigned int clientRecordIndex;

    OBPRECONDITION(identifier != nil);
    clientRecordIndex = [allClientRecords count];
    while (clientRecordIndex--) {
        OAPreferenceClientRecord *clientRecord;

        clientRecord = [allClientRecords objectAtIndex:clientRecordIndex];
        if ([identifier isEqualToString:[clientRecord identifier]])
            return clientRecord;
    }
    return nil;
}

- (void)iconView:(OAPreferencesIconView *)iconView buttonHitAtIndex:(unsigned int)index;
{
    if (iconView == showAllIconView)
        [self _showAllIcons];
    else
        [self _setCurrentClientRecord:[[iconView preferenceClientRecords] objectAtIndex:index]];
}

// Actions

- (IBAction)showPreferencesPanel:(id)sender;
{
    [self _loadInterface];
    [self _resetWindowTitle];
    
    // Let the current client know that it is about to be displayed.
    [nonretained_currentClient becomeCurrentPreferenceClient];
    [self _validateRestoreDefaultsButton];
    [window makeKeyAndOrderFront:sender];
}

- (IBAction)restoreDefaults:(id)sender;
{
    [nonretained_currentClient restoreDefaults:sender];
}

- (IBAction)showNextClient:(id)sender;
{
    NSMutableArray *sortedClientRecords;
    NSEnumerator *enumerator;
    NSString *key;
    int newIndex;

    sortedClientRecords = [[NSMutableArray alloc] init];
    enumerator = [[self _categoryNames] objectEnumerator];
    while ((key = [enumerator nextObject])) {
        [sortedClientRecords addObjectsFromArray:[categoryNamesToClientRecordsArrays objectForKey:key]];
    }
    
    newIndex = [sortedClientRecords indexOfObject:nonretained_currentClientRecord] + 1;
    if (newIndex < [sortedClientRecords count])
        [self _setCurrentClientRecord:[sortedClientRecords objectAtIndex:newIndex]];
    else
        [self _setCurrentClientRecord:[sortedClientRecords objectAtIndex:0]];
    
    [sortedClientRecords release];
}

- (IBAction)showPreviousClient:(id)sender;
{
    NSMutableArray *sortedClientRecords;
    NSEnumerator *enumerator;
    NSString *key;
    int newIndex;

    sortedClientRecords = [[NSMutableArray alloc] init];
    enumerator = [[self _categoryNames] objectEnumerator];
    while ((key = [enumerator nextObject])) {
        [sortedClientRecords addObjectsFromArray:[categoryNamesToClientRecordsArrays objectForKey:key]];
    }

    newIndex = [sortedClientRecords indexOfObject:nonretained_currentClientRecord] - 1;
    if (newIndex < 0)
        [self _setCurrentClientRecord:[sortedClientRecords objectAtIndex:([sortedClientRecords count] - 1)]];
    else
        [self _setCurrentClientRecord:[sortedClientRecords objectAtIndex:newIndex]];
    
    [sortedClientRecords release];
}

- (IBAction)showHelpForClient:(id)sender;
{
    NSString *helpURL = [nonretained_currentClientRecord helpURL];
    
    if (helpURL) {
        id applicationDelegate;
        
        applicationDelegate = [NSApp delegate];
        if ([applicationDelegate respondsToSelector:@selector(openAddressWithString:)]) {
            // We're presumably in OmniWeb, in which case we should display our help internally
            [applicationDelegate openAddressWithString:helpURL];
        } else {
            // We can let the system decide who to open the URL with
            [[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:helpURL]];
        }
    }
            
}


// NSWindow delegate

- (void)windowWillClose:(NSNotification *)notification;
{
    [[notification object] makeFirstResponder:nil];
    [nonretained_currentClient resignCurrentPreferenceClient];
}

- (void)windowDidResignKey:(NSNotification *)notification;
{
    [[notification object] makeFirstResponder:nil];
}

@end


@implementation OAPreferenceController (Private)

- (void)_loadInterface;
{
    NSEnumerator *categoryNameEnumerator;
    NSString *categoryName;
    unsigned int boxHeight = 12;
    
    if (window)
        return;    

    [NSBundle loadNibNamed:@"OAPreferences.nib" owner:self];
    [globalControlsView retain];

    if (![window setFrameUsingName:windowFrameSaveName])
        [window center];
    [window setFrameAutosaveName:windowFrameSaveName];
    
    if ([allClientRecords count] == 1) {
        [self _setCurrentClientRecord:[allClientRecords lastObject]];
        return;
    }

    categoryNameEnumerator = [[self _categoryNames] reverseObjectEnumerator];

    showAllIconsView = [[NSView alloc] initWithFrame:NSZeroRect];

    // This is lame.  We should really think up some way to specify the ordering of preferences in the plists.  But this is difficult since preferences can come from many places.
    while ((categoryName = [categoryNameEnumerator nextObject])) {
        NSArray *categoryClientRecords;
        OAPreferencesIconView *preferencesIconView;
        NSTextField *categoryHeaderTextField;
        const unsigned int verticalSpaceBelowTextField = 8, verticalSpaceAboveTextField = 13;
        
        categoryClientRecords = [categoryNamesToClientRecordsArrays objectForKey:categoryName];
        
        // category preferences view
        preferencesIconView = [[OAPreferencesIconView alloc] initWithFrame:[preferenceBox bounds]];
        [preferencesIconView setPreferenceController:self];
        [preferencesIconView setPreferenceClientRecords:categoryClientRecords];
        
        [showAllIconsView addSubview:preferencesIconView];
        [preferencesIconView setFrameOrigin:NSMakePoint(0, boxHeight)];
        [preferencesIconViews addObject:preferencesIconView];

        boxHeight += NSHeight([preferencesIconView frame]);
        [preferencesIconView release];
        
        // category header
        categoryHeaderTextField = [[NSTextField alloc] initWithFrame:NSZeroRect];
        [categoryHeaderTextField setDrawsBackground:NO];
        [categoryHeaderTextField setBordered:NO];
        [categoryHeaderTextField setEditable:NO];
        [categoryHeaderTextField setSelectable:NO];
        [categoryHeaderTextField setTextColor:[NSColor darkGrayColor]];
        [categoryHeaderTextField setFont:[NSFont systemFontOfSize:11.0]];
        [categoryHeaderTextField setAlignment:NSCenterTextAlignment];
        [categoryHeaderTextField setStringValue:[NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"%@ preferences; drag your favorites to the toolbar above.", @"OmniAppKit", [self bundle], preferences category title format), [self _localizedCategoryNameForCategoryName:categoryName]]];
        [categoryHeaderTextField sizeToFit];
        [showAllIconsView addSubview:categoryHeaderTextField];
        [categoryHeaderTextField setFrame:NSMakeRect(0, boxHeight + verticalSpaceBelowTextField, NSWidth([preferenceBox bounds]), NSHeight([categoryHeaderTextField frame]))];
        
        boxHeight += NSHeight([categoryHeaderTextField frame]) + verticalSpaceBelowTextField + verticalSpaceAboveTextField;
        [categoryHeaderTextField release];
    }

    [showAllIconsView setFrameSize:NSMakeSize(NSWidth([preferenceBox bounds]), boxHeight)];

    [self _showAllIcons];
}

- (void)_resetWindowTitle;
{
    NSString *name;
    
    name = [nonretained_currentClientRecord title];
    if (name == nil || [name isEqualToString:@""])
        name = [[NSProcessInfo processInfo] processName];
    [window setTitle:[NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"%@ Preferences", @"OmniAppKit", [self bundle], preferences panel title format), name]];
}

// Setting the current preference client

- (void)_setCurrentClientRecord:(OAPreferenceClientRecord *)clientRecord
{
    NSView *contentView, *controlBox;
    unsigned int newWindowHeight;
    NSRect controlBoxFrame, windowFrame, newWindowFrame;
    NSView *oldView;
    NSTimeInterval animationResizeTime;

    if (nonretained_currentClientRecord == clientRecord)
        return;
        
    // Save changes in any editing text fields
    [window setInitialFirstResponder:nil];
    [window makeFirstResponder:nil];
    
    // Select correct icon (if any) in bar at top of window
    [favoritesIconView setSelectedClientRecord:clientRecord];
    [showAllIconView setSelectedClientRecord:clientRecord];
    
    // Only do this when we are on screen to avoid sending become/resign twice.  If we are off screen, the client got resigned when it went off and the new one will get a become when it goes on screen.
    if ([window isVisible])
        [nonretained_currentClient resignCurrentPreferenceClient];
        
    nonretained_currentClientRecord = clientRecord;
    nonretained_currentClient = [clientRecord clientInstanceInController:self];
    
    [self _resetWindowTitle];
    
    // Remove old client box
    contentView = [preferenceBox contentView];
    oldView = [[contentView subviews] lastObject];
    [oldView fadeOutAndRemoveFromSuperviewOverTimeInterval:0.08];

    controlBox = [nonretained_currentClient controlBox];
    // It's an error for controlBox to be nil, but it's pretty unfriendly to resize our window to be infinitely high when that happens.
    controlBoxFrame = controlBox != nil ? [controlBox frame] : NSZeroRect;
    
    // Resize the window
    // We don't just tell the window to resize, because that tends to move the upper left corner (which will confuse the user)
    windowFrame = [NSWindow contentRectForFrameRect:[window frame] styleMask:[window styleMask]];
    if ([allClientRecords count] == 1) {
        newWindowHeight = NSHeight(controlBoxFrame) + NSHeight([globalControlsView frame]);
        [preferenceBox setFrameSize:NSMakeSize(NSWidth([preferenceBox frame]), NSHeight([preferenceBox frame]) + NSHeight([showAllIconView frame]))];
        [iconsBox removeFromSuperview];
    } else {
        newWindowHeight = NSHeight([showAllIconView frame]) + NSHeight(controlBoxFrame) + NSHeight([globalControlsView frame]);
    }
    newWindowFrame = [NSWindow frameRectForContentRect:NSMakeRect(NSMinX(windowFrame), NSMaxY(windowFrame) - newWindowHeight, NSWidth(windowFrame), newWindowHeight) styleMask:[window styleMask]];
    animationResizeTime = [window animationResizeTime:newWindowFrame];
    [window setFrame:newWindowFrame display:YES animate:[window isVisible]];
    
    // As above, don't do this unless we are onscreen to avoid double become/resigns.
    // Do this before putting the view in the view hierarchy to avoid flashiness in the controls.
    if ([window isVisible]) {
        [nonretained_currentClient becomeCurrentPreferenceClient];
        [self _validateRestoreDefaultsButton];
    }
    [nonretained_currentClient updateUI];

    // set up the global controls view
    [helpButton setEnabled:([nonretained_currentClientRecord helpURL] != nil)];
    if ([allClientRecords count] == 1) {
        [nextButton removeFromSuperview];
        [previousButton removeFromSuperview];
    }
        
    [contentView addSubview:globalControlsView];
    
    // Add the new client box to the view hierarchy
    [controlBox setFrameOrigin:NSMakePoint(floor((NSWidth([contentView frame]) - NSWidth(controlBoxFrame)) / 2.0), NSHeight([globalControlsView frame]))];
    if ([window isVisible])
        [contentView fadeInSubview:controlBox overTimeInterval:animationResizeTime];
    else
        [contentView addSubview:controlBox];

    // Highlight the initial first responder, and also tell the window what it should be because I think there is some voodoo with nextKeyView not working unless the window has an initial first responder.
    [window setInitialFirstResponder:[nonretained_currentClient initialFirstResponder]];
    [window makeFirstResponder:[nonretained_currentClient initialFirstResponder]];
}

- (void)_showAllIcons;
{
    NSRect windowFrame, newWindowFrame;
    unsigned int newWindowHeight;
    NSTimeInterval animationResizeTime;

    // Are we already showing?
    if ([[[preferenceBox contentView] subviews] lastObject] == showAllIconsView)
        return;

    // Save changes in any editing text fields
    [window setInitialFirstResponder:nil];
    [window makeFirstResponder:nil];

    // Clear out current preference and reset window title
    nonretained_currentClientRecord = nil;
    nonretained_currentClient = nil;
    [[[preferenceBox contentView] subviews] makeObjectsPerformSelector:@selector(removeFromSuperview)];
    [self _resetWindowTitle];
    
    // Highlight appropriate button on button bar at top of window
    [favoritesIconView setSelectedClientRecord:nil];
    [showAllIconView setSelectedClientRecord:nil];
    
    // Resize window
    windowFrame = [NSWindow contentRectForFrameRect:[window frame] styleMask:[window styleMask]];
    newWindowHeight = NSHeight([showAllIconView frame]) + NSHeight([showAllIconsView frame]);
    newWindowFrame = [NSWindow frameRectForContentRect:NSMakeRect(NSMinX(windowFrame), NSMaxY(windowFrame) - newWindowHeight, NSWidth(windowFrame), newWindowHeight) styleMask:[window styleMask]];
    animationResizeTime = [window animationResizeTime:newWindowFrame];
    [window setFrame:newWindowFrame display:YES animate:[window isVisible]];

    // Add new icons view
    if ([window isVisible])
        [preferenceBox fadeInSubview:showAllIconsView overTimeInterval:animationResizeTime];
    else
        [preferenceBox addSubview:showAllIconsView];
}

- (void)_defaultsDidChange:(NSNotification *)notification;
{
    if ([window isVisible])
        [self _validateRestoreDefaultsButton];
}

- (void)_validateRestoreDefaultsButton;
{
    [returnToOriginalValuesButton setEnabled:[nonretained_currentClient haveAnyDefaultsChanged]];
}

// 

static int _OAPreferenceControllerCompareCategoryNames(id name1, id name2, void *context)
{
    OAPreferenceController *self = context;
    float priority1, priority2;

    priority1 = [self _priorityForCategoryName:name1];
    priority2 = [self _priorityForCategoryName:name2];
    if (priority1 == priority2)
        return [[self _localizedCategoryNameForCategoryName:name1] caseInsensitiveCompare:[self _localizedCategoryNameForCategoryName:name2]];
    else if (priority1 > priority2)
        return NSOrderedAscending;
    else // priority1 < priority2
        return NSOrderedDescending;
}

- (NSArray *)_categoryNames;
{
    return [[categoryNamesToClientRecordsArrays allKeys] sortedArrayUsingFunction:_OAPreferenceControllerCompareCategoryNames context:self];
}

- (void)_registerCategoryName:(NSString *)categoryName localizedName:(NSString *)localizedCategoryName priorityNumber:(NSNumber *)priorityNumber;
{
    if (localizedCategoryName != nil && ![localizedCategoryName isEqualToString:categoryName])
        [localizedCategoryNames setObject:localizedCategoryName forKey:categoryName];
    if (priorityNumber != nil)
        [categoryPriorities setObject:priorityNumber forKey:categoryName];
}

- (NSString *)_localizedCategoryNameForCategoryName:(NSString *)categoryName;
{
    return [localizedCategoryNames objectForKey:categoryName defaultObject:categoryName];
}

- (float)_priorityForCategoryName:(NSString *)categoryName;
{
    NSNumber *priority;

    priority = [categoryPriorities objectForKey:categoryName];
    if (priority != nil)
        return [priority floatValue];
    else
        return 0.0;
}

- (void)_registerClassName:(NSString *)className inCategoryNamed:(NSString *)categoryName description:(NSDictionary *)description;
{
    NSMutableArray *categoryClientRecords;
    OAPreferenceClientRecord *newRecord;
    NSDictionary *defaultsDictionary;
    NSString *title, *iconName, *nibName, *identifier, *shortTitle;
    
    
    categoryClientRecords = [categoryNamesToClientRecordsArrays objectForKey:categoryName];
    if (categoryClientRecords == nil) {
        categoryClientRecords = [NSMutableArray array];
        [categoryNamesToClientRecordsArrays setObject:categoryClientRecords forKey:categoryName];
    }
    
    defaultsDictionary = [description objectForKey:@"defaultsDictionary"];
    if (defaultsDictionary)
        [[NSUserDefaults standardUserDefaults] registerDefaults:defaultsDictionary];
    else
        defaultsDictionary = [NSDictionary dictionary]; // placeholder

    title = [[OFBundledClass bundleForClassNamed:className] localizedStringForKey:[description objectForKey:@"title"] value:@"" table:@"Preferences"];
    iconName = [description objectForKey:@"icon"];
    nibName = [description objectForKey:@"nib"];
    
    if (!(className && title && iconName && nibName))
        return;

    identifier = [description objectForKey:@"identifier"];
    if (identifier == nil) {
        // Before we introduced a separate notion of identifiers, we simply used the short title (which defaulted to the title)
        identifier = [description objectForKey:@"shortTitle" defaultObject:[description objectForKey:@"title"]];
    }
    shortTitle = [description objectForKey:@"shortTitle"];

    newRecord = [[OAPreferenceClientRecord alloc] initWithCategoryName:categoryName];
    [newRecord setIdentifier:identifier];
    [newRecord setClassName:className];
    [newRecord setTitle:title];
    [newRecord setShortTitle:shortTitle != nil ? [[OFBundledClass bundleForClassNamed:className] localizedStringForKey:shortTitle value:@"" table:@"Preferences"] : nil];
    [newRecord setIconName:iconName];
    [newRecord setNibName:nibName];
    [newRecord setHelpURL:[description objectForKey:@"helpURL"]];
    [newRecord setDefaultsDictionary:defaultsDictionary];
    [newRecord setDefaultsArray:[description objectForKey:@"defaultsArray"]];

    [categoryClientRecords addObject:newRecord];
    [categoryClientRecords sortUsingSelector:@selector(compare:)];

    [allClientRecords addObject:newRecord];
    [newRecord release];
}

@end
