// Copyright 2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAAppleScript.h>

#import <OmniAppKit/OAInternetConfig.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OmniFoundation/OFResourceFork.h>
#import <OmniBase/OmniBase.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Carbon/OAAppleScript.m,v 1.4 2001/09/11 01:53:23 rick Exp $")

static ComponentInstance OAAppleScriptComponent;

static id aedesc_to_id(AEDesc *desc)
{
    OSErr err;
    
    if (desc->descriptorType == typeChar)
    {
        NSMutableData *outBytes;
        NSString *txt;
        
        outBytes = [[NSMutableData alloc] initWithLength:AEGetDescDataSize(desc)];
        err = AEGetDescData(desc, [outBytes mutableBytes], [outBytes length]);
        if (err) {
            NSLog(@"AECreateDesc -> %d", err);
            return nil;
        }
    
        txt = [[NSString alloc] initWithData:outBytes encoding:[NSString defaultCStringEncoding]];
        [outBytes release];
        [txt autorelease];
        
        return txt;
    }
    
    if (desc->descriptorType == typeSInt16)
    {
        SInt16 buf;
        
        AEGetDescData(desc, &buf, sizeof(buf));
        
        return [NSNumber numberWithShort:buf];
    }
    
    return [NSString stringWithFormat:@"[unconverted AEDesc, type=\"%c%c%c%c\"]", ((char *)&(desc->descriptorType))[0], ((char *)&(desc->descriptorType))[1], ((char *)&(desc->descriptorType))[2], ((char *)&(desc->descriptorType))[3]];

}

@implementation OAAppleScript

+ (void) initialize;
{
    OBINITIALIZE;
    OAAppleScriptComponent = OpenDefaultComponent(kOSAComponentType, kOSAGenericScriptingComponentSubtype);
}

- initWithContentsOfFile:(NSString *)filename;
{
    OAAppleScript *newScript;
    OFResourceFork *resourceFork;
    NSData *possibleScriptData;
    NSString *possibleScriptString;

    if (filename == nil || [filename isEqualToString:@""] || ![[NSFileManager defaultManager] fileExistsAtPath:filename]) {
        NSLog(@"OAAppleScript: No file at %@", filename);
        [self release];
        return nil;
    }
    
    // First, find out if it's got a 'scpt' 128 resource
    NS_DURING {
        resourceFork = [[[OFResourceFork alloc] initWithContentsOfFile:filename] autorelease];
    } NS_HANDLER {
        resourceFork = nil;
    } NS_ENDHANDLER;
    if (resourceFork != nil) {
        possibleScriptData = [resourceFork dataForResourceType:FOUR_CHAR_CODE('scpt') atIndex:0];
        if (possibleScriptData != nil) {
            newScript = [[OAAppleScript alloc] initWithScriptData:possibleScriptData];
            if (newScript != nil)
                return newScript;
        }
    }
    
    // No dice? See if it's a data-fork .scpt file
    possibleScriptData = [NSData dataWithContentsOfFile:filename];
    if (possibleScriptData != nil) {
        newScript = [[OAAppleScript alloc] initWithScriptData:possibleScriptData];
        if (newScript != nil)
            return newScript;
    } else {
        NSLog(@"OAAppleScript: No file data at %@", filename);
        [self release];
        return nil;
    }

    // Finally, try it as a text file to compile
    possibleScriptString = [[[NSString alloc] initWithData:possibleScriptData encoding:[NSString defaultCStringEncoding]] autorelease];
    newScript = [[OAAppleScript alloc] initWithScriptString:possibleScriptString];
    if (newScript != nil)
        return newScript;

    NSLog(@"OAAppleScript: No script in file %@", filename);
    [self release];
    return nil;
}

- initWithScriptData:(NSData *)scriptData;
{
    AEDesc script;
    OSErr err;
    
    err = AECreateDesc(typeOSAGenericStorage, [scriptData bytes], [scriptData length], &script);
    if (err) {
        NSLog(@"AECreateDesc -> %d", err);
        [self release];
        return nil;
    }

    _scriptId = kOSANullScript;
    err = OSALoad(OAAppleScriptComponent, &script, kOSAModeNull, &_scriptId);
    AEDisposeDesc(&script);

    if (err) {
        NSLog(@"OSALoad -> %d", err);
        [self release];
        return nil;
    }

    return self;
}

- initWithScriptString: (NSString *) scriptString;
{
    NSData *scriptChars;
    AEDesc source;
    OSErr err;
    
    scriptChars = [scriptString dataUsingEncoding:[NSString defaultCStringEncoding]];
    err = AECreateDesc(typeChar, [scriptChars bytes], [scriptChars length], &source);
    if (err) {
        NSLog(@"AECreateDesc -> %d", err);
        [self release];
        return nil;
    }
    
    _scriptId = kOSANullScript;
    err = OSACompile(OAAppleScriptComponent, &source, kOSAModeNull, &_scriptId);
    AEDisposeDesc(&source);
    
    if (err) {
        NSLog(@"OSACompile -> %d", err);
        [self release];
        return nil;
    }
    
    return self;
}

- (void)dealloc;
{
    OSErr err;

    err = OSADispose(OAAppleScriptComponent, _scriptId);
    if (err) {
        NSLog(@"OSADispose -> %d", err);
    }
    
    [super dealloc];
}

- (NSString *) execute;
{
    OSAID resultId;
    OSErr err;
    NSString *resultString;
    AEDesc resultText;
    
    resultId = 0;
    err = OSAExecute(OAAppleScriptComponent, _scriptId, kOSANullScript, kOSAModeNull, &resultId);
    if (err)
        NSLog(@"OSADispose -> %d", err);
    
    if (err == errOSAScriptError) {
        AEDesc ernum, erstr;
        id ernumobj, erstrobj;
        
        // Extract the error number and error message from our scripting component.
        err = OSAScriptError(OAAppleScriptComponent, kOSAErrorNumber, typeShortInteger, &ernum);
        if (err) {
            NSLog(@"OSAScriptError -> %d", err);
            return nil;
        }
        err = OSAScriptError(OAAppleScriptComponent, kOSAErrorMessage, typeChar, &erstr);
        if (err) {
            NSLog(@"OSAScriptError -> %d", err);
            return nil;
        }
        
        // Convert them to ObjC types.
        ernumobj = aedesc_to_id(&ernum);
        AEDisposeDesc(&ernum);
        erstrobj = aedesc_to_id(&erstr);
        AEDisposeDesc(&erstr);
        
        resultString = [NSString stringWithFormat:@"Error, number=%@, message=%@", ernumobj, erstrobj];
    } else {
        // If no error, extract the result, and convert it to a string for display
        
        if (resultId != 0) { // apple doesn't mention that this can be 0?
            err = OSADisplay(OAAppleScriptComponent, resultId, typeChar, kOSAModeNull, &resultText);
            if (err) {
                NSLog(@"OSADisplay -> %d", err);
                return nil;
            }
        
            //NSLog(@"result thingy type = \"%c%c%c%c\"", ((char *)&(resultText.descriptorType))[0], ((char *)&(resultText.descriptorType))[1], ((char *)&(resultText.descriptorType))[2], ((char *)&(resultText.descriptorType))[3]);

            resultString = aedesc_to_id(&resultText);
            AEDisposeDesc(&resultText);
        } else {
            resultString = @"[no value returned]";
        }
        OSADispose(OAAppleScriptComponent, resultId);
    }
    
    return resultString;
}

// Utility methods build on OAAppleScript

+ (NSString *) executeScriptString: (NSString *) scriptString;
{
    OAAppleScript *script;
    
    script = [[[self alloc] initWithScriptString: scriptString] autorelease];
    return [script execute];
}

+ (void) sendMailTo:(NSString *) receiver
         carbonCopy:(NSString *) carbonCopy
            subject:(NSString *) subject
               body:(NSString *) body;
{
    NSMutableString *script;
    NSMutableArray *arguments;
    NSString *urlCodedString, *mailApp;
    BOOL firstArgument = YES;
    OAInternetConfigInfo appInfo;
    
    OBPRECONDITION(receiver);
    
    // Look up the user mail app in InternetConfig (LaunchServices doesn't return app names for mailto urls for some reason).
    appInfo = [[OAInternetConfig internetConfig] getInfoForScheme: @"mailto"];
    if (!appInfo.valid)
        mailApp = @"Mail";
    else
        mailApp = appInfo.applicationName;
    
    script = [NSMutableString stringWithFormat:@"tell application \"%@\"\n %@event GURLGURL%@ \"mailto:", mailApp, [NSString stringWithCharacter:0x00AB], [NSString stringWithCharacter:0x00BB]];
    
    urlCodedString = [NSString encodeURLString:receiver asQuery:NO leaveSlashes:NO leaveColons:NO];
    [script appendString: urlCodedString];

#define NEW_ARG do { \
    [script appendString: firstArgument ? @"?" : @"&"]; \
    firstArgument = NO; \
} while (0)
    
    arguments = [NSMutableArray array];
    if (carbonCopy) {
        NEW_ARG;
        urlCodedString = [NSString encodeURLString:carbonCopy asQuery:NO leaveSlashes:NO leaveColons:NO];
        [script appendFormat: @"cc=%@", urlCodedString];
    }
    
    if (subject) {
        NEW_ARG;
        urlCodedString = [NSString encodeURLString:subject asQuery:NO leaveSlashes:NO leaveColons:NO];
        [script appendFormat: @"subject=%@", urlCodedString];
    }
    
    if (body) {
        unsigned int bodyIndex, bodyLength, fragmentLength;
        NEW_ARG;
        
        [script appendString: @"body=\" "];
        // AppleScript does not handle string constants longer than 32K.  This can be avoided by using the string concatenation operator.  We'll concatenate the body text with the rest of the URL string (to make sure that the rest of the URL doesn't blow the 32K limit in conjunction with the first chunk of body text.
        // We'll be a little more conservative and only use 16k character strings.
#define APPLE_SCRIPT_MAX_STRING_LENGTH (16*1024)

        // URL encode before fragmenting since URL encoding will return longer strings
        urlCodedString = [NSString encodeURLString:body asQuery:NO leaveSlashes:NO leaveColons:NO];
        bodyLength = [urlCodedString length];
        bodyIndex = 0;
        while (bodyIndex < bodyLength) {
            NSString *fragment;
            
            fragmentLength = bodyLength - bodyIndex;
            if (fragmentLength > APPLE_SCRIPT_MAX_STRING_LENGTH)
                fragmentLength = APPLE_SCRIPT_MAX_STRING_LENGTH;
                
            fragment = [urlCodedString substringWithRange: (NSRange){bodyIndex, fragmentLength}];
            [script appendFormat: @"& \"%@\" ", fragment];
            bodyIndex += fragmentLength;
        }
    }
    
        
    [script appendString: @"\nend tell"];
    
//    NSLog(@"script = %@", script);
    
    [self executeScriptString: script];
}

@end
