// Copyright 1997-2001 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniAppKit/OAInspector.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OmniAppKit/NSBundle-OAExtensions.h>

RCS_ID("$Header: /NetworkDisk/Source/CVS/OmniGroup/Frameworks/OmniAppKit/Inspector.subproj/OAInspector.m,v 1.27 2001/08/01 23:50:43 wiml Exp $")

@interface OAInspector (Private)
- (void)_showInspector;
- (void)_updateInspector;
- (void)_inspectMainWindow;
- (void)_inspectWindow:(NSWindow *)window;
- (void)_selectionMightHaveChangedNotification:(NSNotification *)notification;
- (void)_inspectWindowNotification:(NSNotification *)notification;
- (void)loadInterface;
@end


NSString *OAInspectorSelectionDidChangeNotification = @"OAInspectorSelectionDidChangeNotification";
NSString *OAInspectorShowInspectorDefaultKey = @"OAShowInspector";

static NSString *windowFrameSaveName = @"Info";

static OAInspector *sharedInspector = nil;
static id multipleSelectionObject;
static NSMutableDictionary *registeredInspectors = nil;

@implementation OAInspector

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
        return;
    initialized = YES;

    // Allocate shared instance on first messsage to class.
    registeredInspectors = [[NSMutableDictionary alloc] init];
    multipleSelectionObject = [[NSObject alloc] init];
}

+ (void)didLoad;
{
    // Allows us to bring up the Inspector panel if it was up when the app closed previously.
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(controllerStartRunning:) name:NSApplicationDidFinishLaunchingNotification object:nil];
}

+ (void)controllerStartRunning:(NSNotification *)notification
{
    if ([[NSUserDefaults standardUserDefaults] boolForKey:OAInspectorShowInspectorDefaultKey])
        [self showInspector];
}

+ (void)registerInspector:(id <OAInspector>)inspector forClass:(Class)aClass;
{
    [registeredInspectors setObject:inspector forKey:aClass];
}

+  (OAInspector *)sharedInspector;
{
    if (!sharedInspector)
        sharedInspector = [[self alloc] init];
    return sharedInspector;
}

+ (void)showInspector
{
    [[self sharedInspector] _showInspector];
}

+ (void)updateInspector
{
    [[self sharedInspector] _updateInspector];
}

+ (id)multipleSelectionObject;
{
    return multipleSelectionObject;
}

// Init

- init
{
    [super init];

    isOnScreen = NO;
    return self;
}

- (id <OAInspectableController, NSObject>)currentInspectableController;
{
    return currentInspectableController;
}
- (id)inspectedObject;
{
    return inspectedObject;
}
- (BOOL)isInspectorVisible
{
    return isOnScreen;
}


// NSWindow delegate methods.

- (void)windowWillClose:(NSNotification *)notification;
{
    NSNotificationCenter *center;

    // Unsubscribe for the notifications we signed up for in -_showInspector
    center = [NSNotificationCenter defaultCenter];
    [center removeObserver:self name:NSWindowDidBecomeMainNotification object:nil];
    [center removeObserver:self name:NSWindowDidResignMainNotification object:nil];
    [center removeObserver:self name:OAInspectorSelectionDidChangeNotification object:nil];

    [[NSUserDefaults standardUserDefaults] setBool:NO forKey:OAInspectorShowInspectorDefaultKey];
    
    [inspectedObject release];
    inspectedObject = nil;
    [currentInspector inspectObject:nil];
    [currentInspectableController release];
    currentInspectableController = nil;
    isOnScreen = NO;
}

- (NSUndoManager *)windowWillReturnUndoManager:(NSWindow *)aWindow;
{
    NSWindow *window;
    NSResponder *nextResponder;
    NSUndoManager *undoManager = nil;

    window = [NSApp mainWindow];
    nextResponder = [window firstResponder];
    if (nextResponder == nil)
        nextResponder = window;

    do {
        if ([nextResponder respondsToSelector:@selector(undoManager)])
            undoManager = [nextResponder undoManager];
        else if ([nextResponder respondsToSelector:@selector(delegate)] && [[(id)nextResponder delegate] respondsToSelector:@selector(undoManager)])
            undoManager = [[(id)nextResponder delegate] undoManager];
        nextResponder = [nextResponder nextResponder];
    } while (nextResponder && !undoManager);
    
    return undoManager;
}

@end


//
// Private API.
//

@implementation OAInspector (Private)

- (void)_showInspector;
{
    NSNotificationCenter *defaultNotificationCenter;

    if (isOnScreen) {
        [inspectorWindow makeKeyAndOrderFront:self];
        return;
    }

    [[NSUserDefaults standardUserDefaults] setBool:YES forKey:OAInspectorShowInspectorDefaultKey];

    if (!inspectorWindow)
        [self loadInterface];

    // While the Inspector is visible, watch for any window to become main.  When that happens, determine if that window's delegate responds to the OAInspectableControllerProtocol, and act accordingly.
    defaultNotificationCenter = [NSNotificationCenter defaultCenter];
    [defaultNotificationCenter addObserver:self selector:@selector(_inspectWindowNotification:) name:NSWindowDidBecomeMainNotification object:nil];
    [defaultNotificationCenter addObserver:self selector:@selector(_selectionMightHaveChangedNotification:) name:NSWindowDidResignMainNotification object:nil];
    [defaultNotificationCenter addObserver:self selector:@selector(_selectionMightHaveChangedNotification:) name:OAInspectorSelectionDidChangeNotification object:nil];


    // Since we were just asked to be shown, jumpstart ourselves by inspecting the main window.
    isOnScreen = YES; // Otherwise, _inspectMainWindow will short-circuit
    [self _inspectMainWindow];
    [inspectorWindow makeKeyAndOrderFront:self];
}

// DON'T Make ourselves key here
- (void)_updateInspector;
{
    if (!isOnScreen)
        [self _showInspector];
    else
        [self _inspectMainWindow];
}

- (void)_inspectMainWindow;
{
    [self _inspectWindow:[NSApp mainWindow]];
}

- (void)_inspectWindow:(NSWindow *)window;
{
    NSView *newView;
    id <OAInspectableController> inspectableController;
    id <OAInspector, NSObject> newInspector;
    NSResponder *nextResponder;
    

    if (!isOnScreen)
        return;

    nextResponder = [window firstResponder];
    if (nextResponder == nil)
        nextResponder = window;
    inspectableController = nil;
    do {
        if ([nextResponder respondsToSelector:@selector(inspectedObject)])
            inspectableController = (id)nextResponder;
        else if ([nextResponder respondsToSelector:@selector(delegate)] && [[(id)nextResponder delegate] respondsToSelector:@selector(inspectedObject)])
            inspectableController = [(id)nextResponder delegate];
        else
            nextResponder = [nextResponder nextResponder];
    } while (nextResponder != nil && inspectableController == nil);

    if ([inspectableController conformsToProtocol:@protocol(OAInspectableController)]) {
        [currentInspectableController release];
        currentInspectableController = [inspectableController retain];
        [inspectedObject release];
        inspectedObject = [[inspectableController inspectedObject] retain];
    } else {
        [currentInspectableController release];
        currentInspectableController = nil;
        [inspectedObject release];
        inspectedObject = nil;
    }

    newInspector = nil;
    // When inspecting nil, bring up the "No Inspector" panel.
    if (inspectedObject == multipleSelectionObject)
        newView = multiInspectorBox;
    else if (!inspectedObject)
        newView = noInspectorBox;
    else {
        newInspector = [registeredInspectors objectForKey:[inspectedObject class]];
        newView = [newInspector inspectorView];
        OBASSERT(newView); // Don't vend me objects I can't inspect.
    }
    
    [inspectorWindow makeFirstResponder:inspectorWindow];

    if (newView != currentInspectorView) {
        [currentInspectorView removeFromSuperview];
        [inspectorBox addSubview:newView];
        [newView setFrameOrigin:NSMakePoint(0, 0)];

        [currentInspector inspectObject:nil];
        currentInspector = newInspector;

        [newView setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];
        [newView setFrameSize:[inspectorBox frame].size];
        currentInspectorView = newView;
        if ([currentInspector inspectorName])
            [inspectorWindow setTitle:[[currentInspector inspectorName] stringByAppendingString:NSLocalizedStringFromTableInBundle(@" Info", @"OmniAppKit", [OAInspector bundle], last word of titlebar of inspector panel with space before it)]];
        else
            [inspectorWindow setTitle:NSLocalizedStringFromTableInBundle(@"Info", @"OmniAppKit", [OAInspector bundle], generic title of inspector panel if what you are inspecting has no title of its own)];
    }
    [currentInspector inspectObject:inspectedObject];
    [inspectorBox setNeedsDisplay:YES];
}

- (void)_selectionMightHaveChangedNotification:(NSNotification *)notification;
{
    [self _inspectMainWindow];
}

- (void)_inspectWindowNotification:(NSNotification *)notification;
{
    [self _inspectWindow:(NSWindow *)[notification object]];
}

- (void)loadInterface;
{
    // This is from a category in NSBundle-OAExtensions; it raises an exception if the nib can't be loaded
    [[OAInspector bundle] loadNibNamed:@"OAInspector.nib" owner:self];

    [inspectorWindow setFrameUsingName:windowFrameSaveName];
    [inspectorWindow setFrameAutosaveName:windowFrameSaveName];

}

@end

