// Copyright 2001-2002 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OFSoftwareUpdateChecker.h"

#import <Foundation/Foundation.h>
#import <OmniBase/rcsid.h>
#import <OmniFoundation/OFInvocation.h>
#import <OmniFoundation/OFScheduledEvent.h>
#import <OmniFoundation/OFScheduler.h>
#import <OmniFoundation/NSArray-OFExtensions.h>
#import <OmniFoundation/NSObject-OFExtensions.h>
#import <OmniFoundation/NSString-OFExtensions.h>
#import <OmniFoundation/NSUserDefaults-OFExtensions.h>
#import <SystemConfiguration/SCDynamicStore.h>
#import <SystemConfiguration/SCNetwork.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniFoundation/OFSoftwareUpdateChecker.m,v 1.13 2002/03/09 01:54:00 kc Exp $");

/* Strings of interest */
static NSString *OSUCurrentVersionsURL = @"http://www.omnigroup.com/CurrentSoftwareVersions.plist";
static NSString *OSUBundleTrackInfoKey = @"OFSoftwareUpdateTrack";
static NSString *OSUBundleCheckAtLaunchKey = @"OFSoftwareUpdateAtLaunch";

// Preferences keys
static NSString *OSUCheckEnabled = @"AutomaticSoftwareUpdateCheckEnabled";
static NSString *OSUCheckFrequencyKey = @"OSUCheckInterval";
static NSString *OSUNextCheckKey = @"OSUNextScheduledCheck";
static NSString *OSUCurrentVersionsURLKey = @"OSUCurrentVersionsURL";

static NSString *OSULastLaunchKey = @"OSUNewestVersionLaunched";

// NB: All of these #defines except for MINIMUM_CHECK_INTERVAL are now unused, because the default values come from the info.plist.
// Not making use of different check intervals for different tracks in this release. Maybe later.
#ifdef DEBUG
    #define DEFAULT_PRERELEASE_CHECK_INTERVAL (60*2) // WHAM WHAM WHAM WHAM WHAM!
#else
    #define DEFAULT_PRERELEASE_CHECK_INTERVAL (60*60*8) // Check three times a day if this is a prerelease version. WHAM WHAM WHAM!
#endif
#define DEFAULT_RELEASE_CHECK_INTERVAL (60*60*24*7) // Otherwise, only check once a week. (Whew.)
#define MINIMUM_CHECK_INTERVAL (60*15)  // If you're really desperate, you can check as often as every fifteen minutes.



NSString *OFSoftwareUpdateExceptionName = @"OFSoftwareUpdateException";
NSString *OSUPreferencesChangedNotificationName = @"OSUPreferencesChangedNotification";

@interface OFSoftwareUpdateChecker (Private)

- (BOOL)_shouldCheckAtLaunch;
- (void)_scheduleNextCheck;
- (void)_initiateCheck;
- (BOOL)_interpretSoftwareStatus:(NSDictionary *)status;
- (NSURL *)_currentVersions;

- (void)_scDynamicStoreDisconnect;
- (BOOL)_postponeCheckForURL:(NSURL *)aURL;

static NSArray *OSUWinnowTracks(NSSet *visibleTracks, NSArray *downloadables);
static NSArray *OSUWinnowVersions(NSDictionary *appInfo, NSArray *downloadables);
static NSArray *extractOSUVersionFromBundle(NSDictionary *bundleInfo);
static NSString *formatOSUVersion(NSArray *osuVersion);
static NSArray *parseOSUVersionString(NSString *str);
static NSSet *computeVisibleTracks(NSDictionary *trackInfo);
static int compareOSUVersions(NSArray *software, NSArray *spec);
static void networkInterfaceWatcherCallback(SCDynamicStoreRef store, CFArrayRef keys, void *info);

/* This is kept separate from our ivars so that people who use this class don't need to pull in all the SystemConfiguration framework headers. */
struct _softwareUpdatePostponementState {
    SCDynamicStoreRef store;   // our connection to the system configuration daemon
    CFRunLoopSourceRef loopSource;  // our run loop's reference to 'store'

    SCDynamicStoreContext callbackContext;
};

@end

static inline NSDictionary *dataToPlist(NSData *input)
{
    CFDataRef cfIn = (CFDataRef)input;
    CFPropertyListRef output;
    CFStringRef errorString;
    
    /* Contrary to the name, this call handles the text-style plist as well. */
    output = CFPropertyListCreateFromXMLData(kCFAllocatorDefault, cfIn, kCFPropertyListImmutable, &errorString);
    
    return (NSDictionary *)output;
}

@implementation OFSoftwareUpdateChecker

static inline void cancelScheduledEvent(OFSoftwareUpdateChecker *self)
{
    if (self->automaticUpdateEvent != nil) {
        [[self retain] autorelease];
        [[OFScheduler mainScheduler] abortEvent:self->automaticUpdateEvent];
    	[self->automaticUpdateEvent release];
        self->automaticUpdateEvent = nil;
    }
}


// Init and dealloc

- initWithTarget:anObject action:(SEL)anAction;
{
    if ([super init] == nil)
        return nil;

    checkTarget = [anObject retain];
    checkAction = anAction;

    flags.shouldCheckAutomatically = [[NSUserDefaults standardUserDefaults] boolForKey:OSUCheckEnabled];
    flags.updateInProgress = 0;          // not currently fetching anything
    automaticUpdateEvent = nil;

    if ([self _shouldCheckAtLaunch])
        [self _initiateCheck];
    else
        [self _scheduleNextCheck];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(softwareUpdatePreferencesChanged:) name:OSUPreferencesChangedNotificationName object:nil];
    
    return self;
}

- (void)dealloc;
{
    OBASSERT(automaticUpdateEvent == nil);  // if it were non-nil, it would be retaining us and we wouldn't be being deallocated
    [self _scDynamicStoreDisconnect]; 
    [checkTarget release];
    checkTarget = nil;
    [super dealloc];
}


// API

- (BOOL)checkSynchronously;
{
    NSURL *currentVersions = [self _currentVersions];
    NSData *fetchedData;
    BOOL didSomething;

    flags.updateInProgress ++;

    NS_DURING {
        cancelScheduledEvent(self);

        fetchedData = [currentVersions resourceDataUsingCache:NO];

        didSomething = [self _interpretSoftwareStatus:dataToPlist(fetchedData)];
    
        flags.updateInProgress --;

        [self _scheduleNextCheck];
    } NS_HANDLER {
        flags.updateInProgress --;
        [self _scheduleNextCheck];
        [localException raise];
        didSomething = NO;
    } NS_ENDHANDLER;
    
    return didSomething;
}

@end

@implementation OFSoftwareUpdateChecker (NotificationsDelegatesDatasources)

- (void)softwareUpdatePreferencesChanged:(NSNotification *)aNotification;
{
    flags.shouldCheckAutomatically = [[NSUserDefaults standardUserDefaults] boolForKey:OSUCheckEnabled];
    [self _scheduleNextCheck];
}

- (void)URLResourceDidFinishLoading:(NSURL *)sender;
{
    NSData *resourceData;
    
    flags.updateInProgress --;
    
    resourceData = [sender resourceDataUsingCache:YES];
    
    [self _interpretSoftwareStatus:dataToPlist(resourceData)];
    // NB (TODO): _interpretSoftwareStatus can raise an exception, in which case we won't schedule another check, and will stop checking for updates until the next time the app is launched. Not sure if this is the best behavior or not.
    
    [self _scheduleNextCheck];
}

- (void)URLResourceDidCancelLoading:(NSURL *)sender;
{
    [self URL:sender resourceDidFailLoadingWithReason:NSLocalizedStringFromTableInBundle(@"Canceled", @"OmniFoundation", [OFSoftwareUpdateChecker bundle], reason for failure to check for update if user has canceled the request)];
}

- (void)URL:(NSURL *)sender resourceDidFailLoadingWithReason:(NSString *)reason;
{
    NSLog(@"Background software update failed: %@", reason);
    
    flags.updateInProgress --;
    [self _interpretSoftwareStatus:nil];
    // NB (TODO): _interpretSoftwareStatus can raise an exception, in which case we won't schedule another check, and will stop checking for updates until the next time the app is launched. Not sure if this is the best behavior or not.
    
    [self _scheduleNextCheck];
}

@end

@implementation OFSoftwareUpdateChecker (Private)

- (BOOL)_shouldCheckAtLaunch;
{
    id checkAtLaunch;
    NSString *lastLaunch;
    NSArray *thisLaunchVersion;
    NSUserDefaults *defaults;
    NSDictionary *myInfo;

    if (!flags.shouldCheckAutomatically)
        return NO;

    myInfo = [[NSBundle mainBundle] infoDictionary];
    checkAtLaunch = [myInfo objectForKey:OSUBundleCheckAtLaunchKey];
    if (!checkAtLaunch || ![checkAtLaunch boolValue])
        return NO;

    thisLaunchVersion = extractOSUVersionFromBundle(myInfo);
    if (!thisLaunchVersion) {
        NSLog(@"Unable to compute version number of this app");
        return NO;
    }


    defaults = [NSUserDefaults standardUserDefaults];

    lastLaunch = [defaults stringForKey:OSULastLaunchKey];
    if (lastLaunch) {
        NSArray *llVersion;

        llVersion = parseOSUVersionString(lastLaunch);

        if(compareOSUVersions(thisLaunchVersion, llVersion) <= 0)
            return NO;
    }

    [defaults setObject:formatOSUVersion(thisLaunchVersion) forKey:OSULastLaunchKey];
    [defaults autoSynchronize];

    return YES;
}

- (void)_scheduleNextCheck;
{
    NSTimeInterval checkInterval;
    NSDate *nextCheckDate, *now;
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    
    /* Make sure we haven't been disabled */
    if (![defaults boolForKey:OSUCheckEnabled])
        flags.shouldCheckAutomatically = 0;

    if (!flags.shouldCheckAutomatically || flags.updateInProgress) {
        cancelScheduledEvent(self);
        return;
    }
    
    /* Determine when we should make the next check */
    checkInterval = [defaults floatForKey:OSUCheckFrequencyKey] * 60 * 60;
    checkInterval = MAX(checkInterval, MINIMUM_CHECK_INTERVAL);
    //NSLog(@"Next OmniSoftwareUpdate check scheduled for %f seconds from now.", checkInterval);
    
    now = [NSDate date];
    nextCheckDate = [defaults objectForKey:OSUNextCheckKey];
    if (!nextCheckDate || ![nextCheckDate isKindOfClass:[NSDate class]] ||
        ([nextCheckDate timeIntervalSinceDate:now] > checkInterval)) {
        nextCheckDate = [[NSDate alloc] initWithTimeInterval:checkInterval sinceDate:now];
        [nextCheckDate autorelease];
        [defaults setObject:nextCheckDate forKey:OSUNextCheckKey];
        [defaults autoSynchronize];
    }
    
    if (automaticUpdateEvent) {
        if(fabs([[automaticUpdateEvent date] timeIntervalSinceDate:nextCheckDate]) < 1.0) {
            // We already have a scheduled check at the time we would be scheduling one, so we don't need to do anything.
            return;
        } else {
            // We have a scheduled check at a different time. Cancel the existing event and add a new one.
            cancelScheduledEvent(self);
        }
    }
    OBASSERT(automaticUpdateEvent == nil);
    automaticUpdateEvent = [[OFScheduledEvent alloc] initWithInvocation:[[[OFInvocation alloc] initForObject:self selector:@selector(_initiateCheck)] autorelease] atDate:nextCheckDate];
    [[OFScheduler mainScheduler] scheduleEvent:automaticUpdateEvent];
}

- (void)_initiateCheck;
{
    NSURL *checkURL;
    
    if (flags.updateInProgress)
        return;

    checkURL = [self _currentVersions];
    
    flags.updateInProgress ++;

    if ([self _postponeCheckForURL:checkURL]) {
        flags.updateInProgress --;  // um, never mind.
        return;
    }
    
    [checkURL loadResourceDataNotifyingClient:self usingCache:NO];
}

- (NSURL *)_currentVersions;
{
    NSString *currentVersionsURL = OSUCurrentVersionsURL;
    NSString *OSUVersion = [[[[NSBundle bundleForClass:isa] infoDictionary] objectForKey:@"CFBundleVersion"] description];
    NSURL *url;
    
    currentVersionsURL = [[NSUserDefaults standardUserDefaults] stringForKey:OSUCurrentVersionsURLKey];
    if (!currentVersionsURL || ![currentVersionsURL length])
        currentVersionsURL = OSUCurrentVersionsURL;
    
    if (OSUVersion && [OSUVersion length])
        currentVersionsURL = [NSString stringWithStrings:currentVersionsURL, @"?", OSUVersion, nil];
    // The reason for appending OSUVersion is to allow server-side hacks to compensate for bugs or changes in this class after it's in the field. Hopefully no-one will ever look at it, of course, but function favors the prepared mind.
    
    url = [NSURL URLWithString:currentVersionsURL];
    
    return url;
}

- (BOOL)_interpretSoftwareStatus:(NSDictionary *)status;
{
    NSBundle *thisApp = [NSBundle mainBundle];
    NSArray *latestVersions;
    NSSet *visibleTracks;
    
    [[NSUserDefaults standardUserDefaults] removeObjectForKey:OSUNextCheckKey];
    // Removing the nextCheckKey will cause _scheduleNextCheck to schedule a check in the future
    
    if (!status) {
        [NSException raise:OFSoftwareUpdateExceptionName format:NSLocalizedStringFromTableInBundle(@"Unable to retrieve information from omnigroup.com.", @"OmniFoundation", [OFSoftwareUpdateChecker bundle], error text generated when softwrae update is unable to retrieve the list of current software versions)];
        return NO; // happy compiler! happy compiler!
    }

    visibleTracks = computeVisibleTracks([status objectForKey:@"tracks"]);
    
    latestVersions = [status objectForKey:[thisApp bundleIdentifier]];
    latestVersions = OSUWinnowTracks(visibleTracks, latestVersions);
    latestVersions = OSUWinnowVersions([thisApp infoDictionary], latestVersions);
    
    if (!latestVersions || ![latestVersions count])
        return NO;   // nothing to do.
    
    [checkTarget performSelector:checkAction withObject:[latestVersions objectAtIndex:0]];
    
    return YES;
}

// Returns YES if we should postpone checking because our check URL requires network access but the system isn't connected to the network. This routine is also responsible for setting up or tearing down the connection to the system config daemon which we use to initiate a check when the machine reconnects to the net.
- (BOOL)_postponeCheckForURL:(NSURL *)aURL;
{
    BOOL canCheckImmediately;
    NSString *urlScheme;

    urlScheme = [aURL scheme];
    if ([urlScheme isEqual:@"file"]) {
        canCheckImmediately = YES;  // filesystem is always available. we hope.
    } else {
        NSString *urlHost = [aURL host];
        const char *hostname = [urlHost cString];

        if (urlHost == nil || hostname == NULL) {   // not sure what's up, but might as well give it a try
            canCheckImmediately = YES;
        } else {
            SCNetworkConnectionFlags reachability;

            if (!SCNetworkCheckReachabilityByName(hostname, &reachability)) {
                // Couldn't determine reachability. Docs don't indicate why this call might fail. Probably bad juju. Go ahead and try, though, I guess.
                canCheckImmediately = YES;
            } else {
                if (reachability & kSCNetworkFlagsReachable)
                    canCheckImmediately = YES;
                /*
                 // Uncomment this to allow background checks to initiate network connections (e.g., ppp dialup). This would probably be a bad idea.
                 else if(reachability & kSCNetworkFlagsConnectionRequired &&
                         reachability & kSCNetworkFlagsConnectionAutomatic)
                 canCheckImmediately = YES;
                 */
                else {
                    // We don't have a net connection righht now.
                    canCheckImmediately = NO;
                }
            }
        }
    }


    if (canCheckImmediately && (postpone != nil)) {
        // Tear down the network-watching stuff.
        [self _scDynamicStoreDisconnect];
    }


    // Set up the network-watching stuff if necessary.
    if (!canCheckImmediately && (postpone == nil)) {
        SCDynamicStoreRef store;
        NSArray *watchedRegexps;

        // SystemConfig keys to watch. These keys reflect the highest layer of the network stack, after link activity is detected, DHCP or whatever has completed, etc.
        watchedRegexps = [NSArray arrayWithObject:@"State:/Network/Global/.*"];

        postpone = malloc(sizeof(*postpone));

        postpone->loopSource = NULL;

        // We don't do any retain/release stuff here since we will always deallocate the dynamic store connection before we deallocate ourselves.
        postpone->callbackContext.version = 0;
        postpone->callbackContext.info = self;
        postpone->callbackContext.retain = NULL;
        postpone->callbackContext.release = NULL;
        postpone->callbackContext.copyDescription = NULL;

        store = SCDynamicStoreCreate(NULL, CFSTR("OFSoftwareUpdateChecker"), networkInterfaceWatcherCallback, &(postpone->callbackContext));
        if (!store) goto error0;

        if (!SCDynamicStoreSetNotificationKeys(store, NULL, (CFArrayRef)watchedRegexps) ||
            !(postpone->loopSource = SCDynamicStoreCreateRunLoopSource(NULL, store, 0)))
            goto error1;

        postpone->store = store;

        CFRunLoopAddSource(CFRunLoopGetCurrent(), postpone->loopSource, kCFRunLoopCommonModes);

        NSLog(@"%@: no network. will watch for changes.", NSStringFromClass(self->isa));
        goto no_error;

error1:
            CFRelease(store);
error0:
            free(postpone);
        postpone = NULL;
        NSLog(@"Error connecting to configd. Will not automatically perform software update.");

no_error:
            ;
    }

    return (!canCheckImmediately);
}

static void networkInterfaceWatcherCallback(SCDynamicStoreRef store, CFArrayRef keys, void *info)
{
    OFSoftwareUpdateChecker *self = info;

    NSLog(@"%@: Network configuration has changed", NSStringFromClass(self->isa));

    [self _initiateCheck];
}

- (void)_scDynamicStoreDisconnect
{
    if (postpone != nil) {

        if (postpone->loopSource) {
            CFRunLoopSourceInvalidate(postpone->loopSource);
            CFRelease(postpone->loopSource);
            postpone->loopSource = NULL;
        }

        CFRelease(postpone->store);
        postpone->store = NULL;

        free(postpone);
        postpone = NULL;

        NSLog(@"%@: no longer watching for network changes", NSStringFromClass(self->isa));
    }
}

// These functions will eventually need to be moved into a separate file to be shared with the standalone software update app

static NSArray *OSUWinnowVersions(NSDictionary *appInfo, NSArray *downloadables)
{
    id appVersion;
    NSMutableArray *winnowed;
    unsigned int dlIndex, dlCount;
    
    if (!downloadables || ![downloadables count])
        return downloadables;
    dlCount = [downloadables count];
        
    appVersion = extractOSUVersionFromBundle(appInfo);
    if (!appVersion) {
        // Unparseable application version. Pass all entries in downloadables, to encourage the user to upgrade to something parseable.
        return downloadables;
    }
        
    NSLog(@"Application version is %@ %@", [appInfo objectForKey:@"CFBundleName"], formatOSUVersion(appVersion));

    winnowed = [[NSMutableArray alloc] initWithCapacity:dlCount];
    [winnowed autorelease];
    
    for(dlIndex = 0; dlIndex < dlCount; dlIndex ++) {
        NSDictionary *downloadable = [downloadables objectAtIndex:dlIndex];
        id dlVersion;
        
        dlVersion = parseOSUVersionString([downloadable objectForKey:@"version"]);
        if (!dlVersion)
            continue;
        
        NSLog(@"Comparing to downloadable %@", formatOSUVersion(dlVersion));
        
        if (compareOSUVersions(appVersion, dlVersion) < 0)
            [winnowed addObject:downloadable];
    }
    
    return winnowed;
}

static NSArray *OSUWinnowTracks(NSSet *visibleTracks, NSArray *downloadables)
{
    int dlIndex, dlCount;
    NSMutableArray *winnowed;
    
    if (!downloadables)
        return nil;
        
    dlCount = [downloadables count];
    winnowed = [[NSMutableArray alloc] initWithCapacity:dlCount];
    [winnowed autorelease];
    for(dlIndex = 0; dlIndex < dlCount; dlIndex ++) {
        NSDictionary *downloadable = [downloadables objectAtIndex:dlIndex];
        NSString *track = [downloadable objectForKey:@"track"];
        
        // If this downloadable is on a particular track (eg, beta, sneakypeek, anything except general release) then only show it if the user knows about that track.
        if (!track || [visibleTracks containsObject:track])
            [winnowed addObject:downloadable];
    }
    
    return winnowed;
}

static NSSet *computeVisibleTracks(NSDictionary *trackInfo)
{
    NSMutableDictionary *known;
    NSString *track;
    NSEnumerator *enumerator;
    BOOL didSomething;
    NSMutableSet *wantedTracks;
    
    known = [[NSMutableDictionary alloc] init];
    [known autorelease];
    
    // TODO: Check Preferences.
    track = [[[NSBundle mainBundle] infoDictionary] objectForKey:OSUBundleTrackInfoKey];
    if (track)
        [known setObject:@"ask" forKey:track];
    else
        NSLog(@"Warning: unknown release track"); // TODO think about this
    
    // Transitively evaluate all of the "this track subsumes that track" directives in the version plist.
    do {

        didSomething = NO;
        
        enumerator = [trackInfo keyEnumerator];
        while(!didSomething && (track = [enumerator nextObject]) != nil) {
            NSArray *subsumes;
            int subsumeIndex;

            if ([known objectForKey:track] != nil)
                continue;
            subsumes = [[trackInfo objectForKey:track] objectForKey:@"subsumes"];
            if (!subsumes)
                continue;
            for(subsumeIndex = 0; subsumeIndex < [subsumes count]; subsumeIndex ++) {
                NSString *subsume = [subsumes objectAtIndex:subsumeIndex];
                if ([known objectForKey:subsume] != nil)  {
                    [known setObject:[known objectForKey:subsume] forKey:track];
                    didSomething = YES;
                    break;
                }
            }
        }
    } while (didSomething);
    
    wantedTracks = [[NSMutableSet alloc] init];
    [wantedTracks autorelease];
    
    enumerator = [known keyEnumerator];
    while( (track = [enumerator nextObject]) != nil) {
        if ([[known objectForKey:track] isEqual:@"ask"])
            [wantedTracks addObject:track];
    }
    
    return wantedTracks;
}

// Compares 'software' (a version number represented as a sequence of integers) to 'spec' (likewise). If 'software' is earlier than 'spec', returns < 0. If it's later than 'spec', returns > 0. If equal, returns 0.
static int compareSimpleVersions(NSArray *software, NSArray *spec)
{
    int index;
    int lhsCount = [software count];
    int rhsCount = [spec count];
    
    for(index = 0; index < lhsCount && index < rhsCount; index ++) {
        int lhs = [[software objectAtIndex:index] intValue];
        int rhs = [[spec objectAtIndex:index] intValue];
        
        if (lhs < rhs)
            return -1;
        if (lhs > rhs)
            return 1;
    }
    
    if (index < rhsCount)
        return -1;
    
    /* Asymmetric: the 'spec' version is implicitly padded with wildcards, so that
       software 4.0.3 is ordered-same-as spec 4.0 (but not spec 4.0.0). However, 
       software 4.0 is ordered-earlier-than spec 4.0.3. */
    
    return 0;
}

// Compares 'software' (a compound marketing + build version) to 'spec' (likewise). If 'software' is earlier than 'spec', returns < 0. If it's later than 'spec', returns > 0. If equal, returns 0.
static int compareOSUVersions(NSArray *software, NSArray *spec)
{
    int lhsCount, rhsCount, index, compare;
    
    lhsCount = [software count];
    rhsCount = [spec count];
    
    compare = 0;
    
    for(index = 0; index < lhsCount && index < rhsCount; index ++) {
        compare = compareSimpleVersions([software objectAtIndex:index],
                                        [spec objectAtIndex:index]);
        if (compare != 0)
            return compare;
    }
    
    if (lhsCount < rhsCount)
        return -1;
    
    return compare;
}

static NSString *formatOSUVersion(NSArray *osuVersion)
{
    NSMutableString *buffer;
    int index;
    
    buffer = [[NSMutableString alloc] init];
    [buffer autorelease];
    
    for(index = 0; index < [osuVersion count]; index ++) {
        if ([buffer length])
            [buffer appendString:@";"];
        [buffer appendString:[[[osuVersion objectAtIndex:index] arrayByPerformingSelector:@selector(description)] componentsJoinedByString:@","]];
    }
    
    return buffer;
}

static NSArray *extractOSUVersionFromBundle(NSDictionary *bundleInfo)
{
    NSString *marketingVersion, *bundleVersion;
    NSScanner *scanner;
    NSMutableArray *buffer;
    NSArray *marketingParsed, *bundleParsed;

    marketingVersion = [bundleInfo objectForKey:@"CFBundleShortVersionString"];
    bundleVersion = [bundleInfo objectForKey:@"CFBundleVersion"];
    
    if (!marketingVersion || ![marketingVersion isKindOfClass:[NSString class]] ||
        !bundleVersion || ![bundleVersion isKindOfClass:[NSString class]])
        return nil;
    
    bundleVersion = [bundleVersion stringByRemovingPrefix:@"v"];
    bundleVersion = [bundleVersion stringByRemovingPrefix:@"V"];
    
    scanner = [NSScanner scannerWithString:marketingVersion];
    buffer = [[NSMutableArray alloc] init];
    for(;;) {
        int anInteger;
        if (![scanner scanInt:&anInteger])
            break;
        [buffer addObject:[NSNumber numberWithInt:anInteger]];
        if (![scanner scanString:@"." intoString:NULL])
            break;
    }
    marketingParsed = [buffer copy];
    
    [buffer removeAllObjects];
    scanner = [NSScanner scannerWithString:bundleVersion];
    for(;;) {
        int anInteger;
        if (![scanner scanInt:&anInteger])
            break;
        [buffer addObject:[NSNumber numberWithInt:anInteger]];
        if (![scanner scanString:@"." intoString:NULL])
            break;
    }
    bundleParsed = [buffer copy];
    [buffer release];
    
    return [NSArray arrayWithObjects:marketingParsed, bundleParsed, nil];
}


static NSArray *parseOSUVersionString(NSString *str)
{
    NSArray *parts;
    NSArray *marketingversion, *buildversion;
    NSString *part;
    
    parts = [str componentsSeparatedByString:@";"];
    if ([parts count] != 2)
        return nil;
    
    part = [parts objectAtIndex:0];
    marketingversion = [part componentsSeparatedByString:[part containsString:@","]?@",":@"."];
    part = [parts objectAtIndex:1];
    buildversion = [part componentsSeparatedByString:[part containsString:@","]?@",":@"."];
    
    if (!marketingversion || !buildversion)
        return nil;
    
    return [NSArray arrayWithObjects:marketingversion, buildversion, nil];
}


@end
