// Copyright 1997-2007 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// <http://www.omnigroup.com/developer/sourcecode/sourcelicense/>.

// #define MAC_OS_X_VERSION_MIN_REQUIRED MAC_OS_X_VERSION_10_2

#import "ONHost.h"

#import <Foundation/Foundation.h>
#import <SystemConfiguration/SystemConfiguration.h>
#import <OmniBase/OmniBase.h>
#import <OmniBase/system.h>

#import "ONHost-InternalAPI.h"
#import "ONHostAddress.h"
#import "ONPortAddress.h"
#import "ONServiceEntry.h"

RCS_ID("$Header: svn+ssh://source.omnigroup.com/Source/svn/Omni/tags/OmniSourceRelease/2008-09-06/OmniGroup/Frameworks/OmniNetworking/ONHost.m 93428 2007-10-25 16:36:11Z kc $")

#ifndef MAX_HOSTNAME_LEN
#ifdef NI_MAXHOST
#define MAX_HOSTNAME_LEN NI_MAXHOST
#else
#define MAX_HOSTNAME_LEN 1024
#endif
#endif

@interface ONHost (Private)
+ (NSString *)_punycodeEncode:(NSString *)aString;
+ (NSString *)_punycodeDecode:(NSString *)aString;

- (BOOL)_tryLocalhost;
- (BOOL)_tryToLookupHostInfoAsDottedQuad;
@end

@implementation ONHost

BOOL ONHostNameLookupDebug = NO;
static NSRecursiveLock *ONHostLookupLock;
static NSSet *squatterAddresses;
static NSTimeInterval ONHostDefaultTimeToLiveTimeInterval = 60.0 * 60.0;
static BOOL ONHostOnlyResolvesIPv4Addresses = NO; /* This doesn't actually have any effect on the network traffic generated by Apple's current lookupd. Sigh ... */

static enum {
    /* getaddrinfo() is threadable and versatile, but triggers a bug in the name servers used by a couple of high-profile websites whose names I will not mention here. */
    Resolver_getaddrinfo,
    /* The tool uses gethostbyname(), which is not threadable (hence the tool) and not as versatile, but doesn't confuse the badly-written nameservers out there. */
    Resolver_pipe,
    /* Or, of course, we could just not look stuff up at all. */
    Resolver_none
} ONHostResolverAPI = Resolver_getaddrinfo;

/* The following variables are all protected by ONHostLookupLock */
static NSMutableDictionary *hostCache;
static NSString *domainName;
static NSString *localHostname;
static SCDynamicStoreRef systemConfigSession;
static CFRunLoopRef systemConfigListenerRunLoop;
static CFRunLoopSourceRef systemConfigNotificationSource;

/* These are used when we talk to configd */
static void systemConfigChanged(SCDynamicStoreRef, CFArrayRef, void *);
static void locked_connectToSysConfig(void);
static void locked_disconnectFromSysConfig(void);

+ (void)initialize;
{
    OBINITIALIZE;

    ONHostLookupLock = [[NSRecursiveLock alloc] init];
    hostCache = [[NSMutableDictionary alloc] initWithCapacity:16];

    localHostname = nil;
    domainName = nil;
    systemConfigSession = NULL;

    squatterAddresses = [[NSSet alloc] initWithObjects:
        [ONHostAddress addressWithIPv4UnsignedLong:0x405E6E0BUL],
        nil];
}

+ (void)setDebug:(BOOL)newDebugSetting;
{
    ONHostNameLookupDebug = newDebugSetting;
}

+ (void)setOnlyResolvesIPv4Addresses:(BOOL)onlyV4
{
    ONHostOnlyResolvesIPv4Addresses = onlyV4;
}

+ (BOOL)onlyResolvesIPv4Addresses
{
    return ONHostOnlyResolvesIPv4Addresses;
}

+ (void)setResolverType:(NSString *)resolverType
{
    if ([resolverType isEqualToString:@"getaddrinfo"])
        ONHostResolverAPI = Resolver_getaddrinfo;
    else if ([resolverType isEqualToString:@"pipe"])
        ONHostResolverAPI = Resolver_pipe;
    else if ([resolverType isEqualToString:@"gethostbyname"])
        ONHostResolverAPI = Resolver_pipe;
    else if ([resolverType isEqualToString:@"none"])
        ONHostResolverAPI = Resolver_none;
    else {
        NSLog(@"Unknown resolver type \"%@\", not changed", resolverType);
    }
}


+ (void)listenForNetworkChanges;
{
    [ONHostLookupLock lock];

    if (systemConfigListenerRunLoop == NULL) {
        systemConfigListenerRunLoop = CFRunLoopGetCurrent();
        CFRetain(systemConfigListenerRunLoop);
        
        /* Clear out the cached info, because we don't know if it might have changed since we  cached it */
        systemConfigChanged(NULL, NULL, NULL);
    }

    [ONHostLookupLock unlock];
}

static void systemConfigChanged(SCDynamicStoreRef	store,
                                CFArrayRef		changedKeys,
                                void			*info)
{
    [ONHostLookupLock lock];
    if (ONHostNameLookupDebug)
        NSLog(@"%@Flushing cached host and domain names.", (store==NULL)?@"":@"Received change notification from configd. ");
    [domainName release];
    domainName = nil;
    [localHostname release];
    localHostname = nil;
    [ONHostLookupLock unlock];
}

static CFStringRef createDnsStateKey(void)
{
    return SCDynamicStoreKeyCreateNetworkGlobalEntity(NULL, kSCDynamicStoreDomainState, kSCEntNetDNS);
}

static void locked_connectToSysConfig(void)
{
    if (systemConfigSession == NULL) {
    
        if (ONHostNameLookupDebug)
            NSLog(@"Connecting to configd");
        
        /* Open a connection to configd */
        systemConfigSession = SCDynamicStoreCreate(NULL,
                                                   (CFStringRef)[[NSBundle bundleForClass:[ONHost class]] description],
                                                   systemConfigChanged,
                                                   NULL);
    }

    /* If we've been given a run loop, then register for notification of changes to the system configuration */
    if (systemConfigListenerRunLoop != NULL &&
        systemConfigNotificationSource == NULL &&
        systemConfigSession != NULL) {

        CFStringRef notificationKeys[2];
        CFArrayRef notificationArray;

        /* Indicate which configuration keys we're interested in */
        notificationKeys[0] = createDnsStateKey();
        notificationKeys[1] = SCDynamicStoreKeyCreateHostNames(NULL);
        notificationArray = CFArrayCreate(NULL, (const void **)notificationKeys, 2, &kCFTypeArrayCallBacks);
        CFRelease(notificationKeys[0]);
        CFRelease(notificationKeys[1]);

        if (ONHostNameLookupDebug)
            NSLog(@"Requesting notifications from configd, keys=%@", notificationArray);

        SCDynamicStoreSetNotificationKeys(systemConfigSession, notificationArray, NULL);
        CFRelease(notificationArray);

        /* Add a run loop source so we actually get called */
        systemConfigNotificationSource = SCDynamicStoreCreateRunLoopSource(NULL, systemConfigSession, 0);
        CFRunLoopAddSource(systemConfigListenerRunLoop, systemConfigNotificationSource, kCFRunLoopCommonModes);
    }
}

static void locked_disconnectFromSysConfig(void)
{
    if (systemConfigSession == NULL)
        return;

    if (systemConfigNotificationSource != NULL) {
        /* Don't disconnect if we're listening for changes. */
        return;
    }

    if (ONHostNameLookupDebug)
        NSLog(@"Disconnecting from configd.");

    CFRelease(systemConfigSession);
    systemConfigSession = NULL;
}

+ (NSString *)domainName;
{
    NSString *result;
    [ONHostLookupLock lock];
    if (domainName == nil) {
        CFDictionaryRef dnsSettings;
        CFStringRef dnsStateKey;
        
        locked_connectToSysConfig();

        dnsStateKey = createDnsStateKey();
        dnsSettings = SCDynamicStoreCopyValue(systemConfigSession, dnsStateKey);
        CFRelease(dnsStateKey);

        locked_disconnectFromSysConfig();

        domainName = (NSString *)CFDictionaryGetValue(dnsSettings, kSCPropNetDNSDomainName);
        [domainName retain];
        
        CFRelease(dnsSettings);
    }
    if (domainName == nil)
        domainName = [@"local" retain];
    result = [[domainName retain] autorelease];
    [ONHostLookupLock unlock];
    return result;
}

+ (NSString *)localHostname;
{
    NSString *result;
    [ONHostLookupLock lock];
    if (localHostname == nil) {
        locked_connectToSysConfig();
        localHostname = (NSString *)SCDynamicStoreCopyLocalHostName(systemConfigSession);
        locked_disconnectFromSysConfig();
    }
    if (localHostname == nil)
        localHostname = [@"localhost" retain];
    result = [[localHostname retain] autorelease];
    [ONHostLookupLock unlock];
    return result;
}

+ (id)_lockCache:(NSMutableDictionary *)hostInfoCache toCheckForKey:aCacheKey pendingLock:(NSLock **)pendingLockPtr
{
    id cachedObject;
    NSThread *currentThread = nil;
    NSLock *pendingLock;

    if (ONHostNameLookupDebug)
        currentThread = [NSThread currentThread];

    [ONHostLookupLock lock];

    do {
        cachedObject = [hostInfoCache objectForKey:aCacheKey];

        if (cachedObject == nil)
            break;

        if ([cachedObject isKindOfClass:[NSLock class]]) {
            pendingLock = (NSLock *)cachedObject;

            if (ONHostNameLookupDebug)
                NSLog(@"<%@> Lookup already in progress for %@.  Waiting for it to finish...", currentThread, [aCacheKey description]);

            [pendingLock retain];
            [ONHostLookupLock unlock];
            [pendingLock lock];
            [pendingLock unlock];
            [pendingLock release];

            if (ONHostNameLookupDebug)
                NSLog(@"<%@> Rechecking cache...", currentThread);

            // Now look again.
            [ONHostLookupLock lock];
            cachedObject = [hostInfoCache objectForKey:aCacheKey];
        } else if ([cachedObject isKindOfClass:[ONHost class]] && [(ONHost *)cachedObject isExpired]) {
                if (ONHostNameLookupDebug)
                    NSLog(@"<%@> Found an expired host in the cache (%@)", currentThread, cachedObject);
                [hostInfoCache removeObjectForKey:aCacheKey];
                cachedObject = nil;
        } else {
            if (ONHostNameLookupDebug)
                NSLog(@"<%@> Found a object in the cache (%@)", currentThread, cachedObject);
            [cachedObject retain];
            [ONHostLookupLock unlock];
            [cachedObject autorelease];
            return cachedObject;
        }
    } while (cachedObject != nil);

    // Not found in the cache, so we should look it up. Place an NSLock in the cache to allow other lookups of the same key to block until we've finished.
    pendingLock = [[NSLock alloc] init];
    [hostInfoCache setObject:pendingLock forKey:aCacheKey];
    [pendingLock lock];
    [ONHostLookupLock unlock];

    *pendingLockPtr = pendingLock;  // caller must release this
    return nil;
}

+ (ONHost *)hostForHostname:(NSString *)aHostname;
{
    ONHost *host;
    NSThread *currentThread = nil;
    NSLock *pendingLock;
    NSException *raisedException = nil;

    if (!aHostname)
	return nil;

    if (ONHostNameLookupDebug) {
        currentThread = [NSThread currentThread];
        NSLog(@"<%@> Starting name lookup for %@", currentThread, aHostname);
    }
        
    // First, check under the current capitalization of the address.  Since this is usually correct, this will avoid needlessly creating another string
    pendingLock = nil;
    host = [self _lockCache:hostCache toCheckForKey:aHostname pendingLock:&pendingLock];
    if (host == nil) {

        // Try the lowercase string
        NSString *lowercaseHostname = [aHostname lowercaseString];

        if (lowercaseHostname != aHostname && ![aHostname isEqualToString:lowercaseHostname]) {
            NSLock *oldPendingLock = pendingLock;
            pendingLock = nil;

            host = [self _lockCache:hostCache toCheckForKey:lowercaseHostname pendingLock:&pendingLock];

            [ONHostLookupLock lock];
            [hostCache removeObjectForKey:aHostname];  // remove oldPendingLock from the hostCache
            [oldPendingLock unlock];
            [oldPendingLock release];
            if (host)   // ... so that next time we won't need to call -lowercaseString ...
                [hostCache setObject:host forKey:aHostname];
            [ONHostLookupLock unlock];

            aHostname = lowercaseHostname;
        }
    }
    if (host != nil) {
        OBASSERT(pendingLock == nil);
        OBPOSTCONDITION([host isKindOfClass:[ONHost class]]);
        return host;
    }
    
    // There were either no previous attempts to determine the address for this host name or they failed.  We will unlock the main lock while we process the request so that others can get at the cache and possibly start their own request.  If another thread asks for the same host that we are currently resolving, though, they will need to block.  We will put an lock in the cache under the inquiry name for this purpose.
    OBASSERT(pendingLock != nil);

    // Do the lookup.  If there is an error we need to remove the pendingLock, unlock it and then reraise so that other threads may try the lookup (we don't cache failures).
    NS_DURING {
        host = [[self alloc] _initWithHostname:aHostname knownAddress:nil];
    } NS_HANDLER {
        OBASSERT(host == nil);
        raisedException = localException;
    } NS_ENDHANDLER;

    // Lock the cache again now that we have our result (or error)
    [ONHostLookupLock lock];

    if (host)
        [hostCache setObject:host forKey:aHostname];
    else
        [hostCache removeObjectForKey:aHostname];

    // Unlock the pending lock and the main lock
    [pendingLock unlock];
    [pendingLock release];
    [ONHostLookupLock unlock];

    if (raisedException)
        [raisedException raise];

    return [host autorelease];
}

+ (ONHost *)hostForAddress:(ONHostAddress *)anAddress;
{
    ONHost *host;
    NSException *failureException;
    NSLock *pendingOperationLock;
    struct sockaddr *portAddress;
    int err;
    char hostnameBuffer[MAX_HOSTNAME_LEN];
    
    if (!anAddress)
        return nil;

    if (ONHostNameLookupDebug)
        NSLog(@"<%@> Starting byaddress lookup for %@", [NSThread currentThread], [anAddress description]);

    // First, check the cache.
    pendingOperationLock = nil;
    host = [self _lockCache:hostCache toCheckForKey:anAddress pendingLock:&pendingOperationLock];
    if (host) {
        OBASSERT([host isKindOfClass:[ONHost class]]);
        OBASSERT(pendingOperationLock == nil);
        return host;
    }
    OBASSERT(pendingOperationLock != nil);
    
    // Now we do the actual lookup.

    if (ONHostNameLookupDebug)
        NSLog(@"<%@> Calling getnameinfo([%@], ..., NI_NAMEREQD)", [NSThread currentThread], [anAddress description]);

    @try {
    failureException = nil;

    portAddress = [anAddress mallocSockaddrWithPort:0];
    err = getnameinfo(portAddress, portAddress->sa_len,
                      hostnameBuffer, MAX_HOSTNAME_LEN,
                      NULL, 0,
                      NI_NAMEREQD);
    free(portAddress);

    switch (err) {
        case 0:
            // Success --- we got a hostname for this address. Look up the rest of the information about the host.
            @try {
                host = [self hostForHostname:[NSString stringWithCString:hostnameBuffer]];
                // Only accept the host if it actually does refer to the IP address we started with.
                if ([[host addresses] indexOfObject:anAddress] == NSNotFound)
                    host = nil;
            } @catch (NSException *lookupException) {
                // Forward resolution failed for some reason, but that's OK really.
                host = nil;
                if (ONHostNameLookupDebug)
                    NSLog(@"<%@> Forward resolution failed: %@", [NSThread currentThread], lookupException);
            }
            if (host != nil)
                break;  // If we got a host we like, accept it.
            // Else, fall through.
        case EAI_NONAME:
            // Couldn't find a good hostname for this address. Create an unnamed ONHost.
            host = [[[self alloc] _initWithHostname:nil knownAddress:anAddress] autorelease];
            break;
        default:
            failureException = [self _exceptionForExtendedHostErrorNumber:err hostname:[anAddress description]];
    }
    } @catch (NSException *exc) {
        failureException = exc;
        host = nil;
    }
    
    // Place our result in the cache.
    [ONHostLookupLock lock];
    if (host)
        [hostCache setObject:host forKey:anAddress];
    else
        [hostCache removeObjectForKey:anAddress];
    [pendingOperationLock unlock];
    [pendingOperationLock release];
    [ONHostLookupLock unlock];

    // And finally, return the result to the caller.
    if (failureException) {
        [failureException raise];
        return nil; // pacify the compiler
    } else
        return host;
}

+ (NSString *)IDNEncodedHostname:(NSString *)aHostname;
{
    if ([aHostname canBeConvertedToEncoding:NSASCIIStringEncoding])
        return aHostname;

    NSArray *parts = [aHostname componentsSeparatedByString:@"."];
    NSMutableArray *encodedParts = [NSMutableArray array];
    unsigned int partIndex, partCount = [parts count];

    for (partIndex = 0; partIndex < partCount; partIndex++)
        [encodedParts addObject:[self _punycodeEncode:[[parts objectAtIndex:partIndex] precomposedStringWithCompatibilityMapping]]];
    return [encodedParts componentsJoinedByString:@"."];
}

+ (NSString *)IDNDecodedHostname:(NSString *)anIDNHostname;
{
    NSArray *labels = [anIDNHostname componentsSeparatedByString:@"."];
    NSMutableArray *decodedLabels;
    int labelIndex, labelCount;
    BOOL wasEncoded;
    
    labelCount = [labels count];
    decodedLabels = [[NSMutableArray alloc] initWithCapacity:labelCount];
    wasEncoded = NO;
    
    for (labelIndex = 0; labelIndex < labelCount; labelIndex++) {
        NSString *label, *decodedLabel;
        
        label = [labels objectAtIndex:labelIndex];
        decodedLabel = [self _punycodeDecode:label];
        if (!wasEncoded && ![label isEqualToString:decodedLabel])
            wasEncoded = YES;
        [decodedLabels addObject:decodedLabel];
    }
    
    if (wasEncoded) {
        NSString *result = [decodedLabels componentsJoinedByString:@"."];
        [decodedLabels release];
        return result;
    } else {
        /* This is by far the most common case. */
        [decodedLabels release];
        return anIDNHostname;
    }
}

+ (void)flushCache;
{
    NSArray *hostnames;
    unsigned int hostnameIndex, hostnameCount;

    [ONHostLookupLock lock];
    NS_DURING {
        if (ONHostNameLookupDebug)
            NSLog(@"<%@> Flushing host cache", [NSThread currentThread]);
        hostnames = [[hostCache keyEnumerator] allObjects];
        hostnameCount = [hostnames count];
        for (hostnameIndex = 0; hostnameIndex < hostnameCount; hostnameIndex++) {
            NSString *aHostname;
            ONHost *host;

            aHostname = [hostnames objectAtIndex:hostnameIndex];
            host = [hostCache objectForKey:aHostname];
            if ([host isKindOfClass:self]) {
                // Only remove the ONHost entries, not the pending locks
                [hostCache removeObjectForKey:aHostname];
            }
        }
    } NS_HANDLER {
        NSLog(@"+[ONHost flushCache]: Warning: %@", [localException reason]);
    } NS_ENDHANDLER;
    [ONHostLookupLock unlock];
}

+ (void)setDefaultTimeToLiveTimeInterval:(NSTimeInterval)newValue;
{
    ONHostDefaultTimeToLiveTimeInterval = newValue;
    [self flushCache];
}

- (void)dealloc;
{
    [hostname release];
    [canonicalHostname release];
    [addresses release];
    [expirationDate release];
    [super dealloc];
}

- (NSString *)hostname;
{
    return hostname;
}

- (NSArray *)addresses;
{
    return addresses;
}

- (NSString *)canonicalHostname;
{
    if (canonicalHostname != nil)
        return canonicalHostname;
    
    if ([addresses count] == 1)
        return [[addresses objectAtIndex:0] description];

    return nil;
}

- (NSString *)IDNEncodedHostname;
{
    return [isa IDNEncodedHostname:hostname];
}


- (NSString *)domainName;
{
    NSRange dotRange;

    if (canonicalHostname == nil)
        return nil;
        
    dotRange = [canonicalHostname rangeOfString:@"." options:NSLiteralSearch];
    if (dotRange.length == 0)
        return nil;
    return [canonicalHostname substringFromIndex:NSMaxRange(dotRange)];
}

- (BOOL)isLocalHost;
{
    // quick test
    if ([[self addresses] containsObject:[ONHostAddress loopbackAddress]])
        return YES;

    // TODO: check against [[ONInterface interfaces] addresses]
    return NO;
}

- (void)flushFromHostCache;
{
    [ONHostLookupLock lock];
    NS_DURING {
        if ([hostCache objectForKey:hostname] == self)
            [hostCache removeObjectForKey:hostname];
    } NS_HANDLER {
        NSLog(@"+[ONHost removeFromHostCache]: Warning: %@", [localException reason]);
    } NS_ENDHANDLER;
    [ONHostLookupLock unlock];
}

// Looking up service addresses

- (NSArray *)portAddressesForService:(ONServiceEntry *)servEntry
{
    const char *myHostname, *myServname;
    struct addrinfo hints;
    struct addrinfo *results, *result;
    NSLock *operationLock;
    NSArray *portAddresses;
    NSException *failure;
    int err;

    operationLock = nil;
    portAddresses = [[self class] _lockCache:serviceAddresses toCheckForKey:servEntry pendingLock:&operationLock];
    if (portAddresses != nil) {
        OBASSERT([portAddresses isKindOfClass:[NSArray class]]);
        OBASSERT(operationLock == nil);
        return portAddresses;
    }
    OBASSERT(operationLock != nil);

    myHostname = [[self IDNEncodedHostname] UTF8String]; // The result should be ASCII, but -UTF8String is a superset and there is no ASCII variant.
    myServname = [[servEntry serviceName] UTF8String];
    bzero(&hints, sizeof(hints));
    hints.ai_flags = AI_DEFAULT;
    results = NULL;
    
    if ([[servEntry protocolName] isEqualToString:ONServiceEntryTCPProtocolName])
        hints.ai_protocol = IPPROTO_TCP;
    else if ([[servEntry protocolName] isEqualToString:ONServiceEntryUDPProtocolName])
        hints.ai_protocol = IPPROTO_UDP;

    err = getaddrinfo(myHostname, myServname, &hints, &results);
    if (err == 0) {
        NSMutableArray *addressBuf;
        
        addressBuf = [[NSMutableArray alloc] init];
        for(result = results; result != NULL; result = result->ai_next) {
            ONPortAddress *address;

            address = [[ONPortAddress alloc] initWithSocketAddress:result->ai_addr];
            if (address != nil) {
                [addressBuf addObject:address];
                [address release];
            }
        }
        freeaddrinfo(results);

        portAddresses = [[NSArray alloc] initWithArray:addressBuf];
        [addressBuf release];
        failure = nil;
    } else {
        failure = [[self class] _exceptionForExtendedHostErrorNumber:err hostname:[NSString stringWithFormat:@"%@:%@/%@", hostname, [servEntry serviceName], [servEntry protocolName]]];
        portAddresses = nil;
    }

    // Place our result in the cache.
    [ONHostLookupLock lock];
    if (portAddresses)
        [serviceAddresses setObject:portAddresses forKey:servEntry];
    else
        [serviceAddresses removeObjectForKey:servEntry];
    [operationLock unlock];
    [operationLock release];
    [ONHostLookupLock unlock];

    // And finally, return the result to the caller.
    if (failure) {
        [failure raise];
        return nil; // pacify the compiler
    } else
        return portAddresses;
}

// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    if (addresses)
        [debugDictionary setObject:addresses forKey:@"addresses"];
    if (hostname)
        [debugDictionary setObject:hostname forKey:@"hostname"];
    if (expirationDate)
        [debugDictionary setObject:expirationDate forKey:@"expirationDate"];
    return debugDictionary;
}

@end


@implementation ONHost (ONInternalAPI)

+ (void)_raiseExceptionForHostErrorNumber:(int)hostErrorNumber hostname:(NSString *)aHostname;
{
    NSBundle *myBundle = [NSBundle bundleForClass:[ONHost class]];
    
    switch (hostErrorNumber) {
        case HOST_NOT_FOUND:
            [NSException raise:ONHostNotFoundExceptionName format:NSLocalizedStringFromTableInBundle(@"No such host %@", @"OmniNetworking", myBundle, @"gethostbyname error - HOST_NOT_FOUND - 'No such host is known.'"), aHostname];
            break;
        case TRY_AGAIN:
            [NSException raise:ONHostNotFoundExceptionName format:NSLocalizedStringFromTableInBundle(@"Temporary error looking up host %@, try again", @"OmniNetworking", myBundle, @"gethostbyname error - TRY_AGAIN - 'This  is  usually a temporary error and means that the local server did not  receive  a  response  from  an authoritative server.  A  retry  at some later time may succeed.'"), aHostname];
            break;
        case NO_RECOVERY:
            [NSException raise:ONHostNotFoundExceptionName format:NSLocalizedStringFromTableInBundle(@"Unexpected server failure looking up host %@", @"OmniNetworking", myBundle, @"gethostbyname error - NO_RECOVERY - 'Some  unexpected server failure was encountered.  This is a  non-recoverable error.'"), aHostname];
            break;
        case NO_DATA:
            [NSException raise:ONHostHasNoAddressesExceptionName format:NSLocalizedStringFromTableInBundle(@"Found no addresses for host %@", @"OmniNetworking", myBundle, @"gethostbyname error - NO_DATA - 'The  requested  name  is  valid but does not have an IP  address;  this is  not  a  temporary  error.  This means that the name is known to the name server but there is no address associated with this name.'"), aHostname];
            break;
        default:
            [NSException raise:ONHostNameLookupErrorExceptionName format:NSLocalizedStringFromTableInBundle(@"Error looking up host %@", @"OmniNetworking", myBundle, @"gethostbyname error - other errors - specific cause of error is not known"), aHostname];
            break;
    }
}

+ (NSException *)_exceptionForExtendedHostErrorNumber:(int)eaiError hostname:(NSString *)name
{
    NSBundle *myBundle = [NSBundle bundleForClass:[ONHost class]];
    NSString *exceptionName, *exceptionReason;
    NSDictionary *userInfo;

    userInfo = NULL;
    
    switch(eaiError) {
        // Temporary name lookup failure.
        case EAI_AGAIN:
            exceptionName = ONHostNotFoundExceptionName;
            exceptionReason = [NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Temporary error looking up host %@, try again", @"OmniNetworking", myBundle, @"getaddrinfo error - EAI_AGAIN - 'This  is  usually a temporary error and means that the local server did not  receive  a  response  from  an authoritative server.  A  retry  at some later time may succeed.'"), name];
            break;

        // These are the "success, but no success" errors...
        case EAI_NONAME:
            exceptionName = ONHostHasNoAddressesExceptionName;
            exceptionReason = [NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Found no hostname for  %@", @"OmniNetworking", myBundle, @"gethostbyname error - EAI_NONAME - The specified address is valid but does not have a corresponding hostname. This probably means that the address has no reverse DNS information."), name];
            break;
        case EAI_NODATA:
#warning EAI_NODATA seems to be used for both no-such-host and host-has-no-address cases. Is there a way to distinguish between the two when using getaddrinfo?
            exceptionName = ONHostHasNoAddressesExceptionName;
            exceptionReason = [NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Found no address for host %@", @"OmniNetworking", myBundle, @"gethostbyname error - EAI_NODATA - The specified hostname is valid but does not have not have an IP  address; this is not a temporary error. This means that the name is known to the name server but there is no address associated with this name."), name];
            break;
        case EAI_ADDRFAMILY:
            exceptionName = ONHostHasNoAddressesExceptionName;
            exceptionReason = [NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Found no appropriate addresses for host %@", @"OmniNetworking", myBundle, @"gethostbyname error - EAI_ADDRFAMILY - The specified name is valid but does not have an address we can use. This probably means it has an address of some other address family."), name];
            break;
            
        // Catchall for errors which probably indicate either something wrong with OmniNetworking, or something wrong with the system at a low enough level that we can't do anything about it. We use gai_strerror() to display the particular error in case that's useful for debugging.
        default:
            exceptionName = ONHostNameLookupErrorExceptionName;
            exceptionReason = [NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Unexpected server failure (%d: %s) looking up host %@", @"OmniNetworking", myBundle, @"getaddrinfo error - miscellaneous unrecoverable or internal errors - bad parameter passed to get_info call or other unexpected problem"), eaiError, gai_strerror(eaiError), name];
            break;

        case EAI_SYSTEM:
            // Some system-level error occurred; error code is in errno
            exceptionName = ONHostNameLookupErrorExceptionName;
            exceptionReason = [NSString stringWithFormat:NSLocalizedStringFromTableInBundle(@"Error looking up host %@: %s", @"OmniNetworking", myBundle, @"getnameinfo or getaddrinfo error - EAI_SYSTEM - specific cause of error is not known"), name, strerror(OMNI_ERRNO())];
            userInfo = [NSDictionary dictionaryWithObject:[NSNumber numberWithInt:OMNI_ERRNO()] forKey:OBExceptionPosixErrorNumberKey];
            break;
    }

    return [NSException exceptionWithName:exceptionName
                                   reason:exceptionReason
                                 userInfo:userInfo];
}

- _initWithHostname:(NSString *)aHostname knownAddress:(ONHostAddress *)knownAddress;
{    
    if ([super init] == nil)
	return nil;

    if (aHostname == nil && knownAddress != nil) {
        hostname = [[knownAddress description] retain];
        addresses = [[NSArray arrayWithObject:knownAddress] retain];
        return self;
    }

    OBASSERT(aHostname != nil && knownAddress == nil);
    
    hostname = [aHostname retain];

    if ([self _tryLocalhost] || [self _tryToLookupHostInfoAsDottedQuad])
        return self;

    NS_DURING {
        switch(ONHostResolverAPI) {
            case Resolver_getaddrinfo:
                [self _lookupHostInfoUsingGetaddrinfo];
                break;
            case Resolver_pipe:
                [self _lookupHostInfoByPipe];
                break;
            default:
                addresses = [[NSArray alloc] init];
                break;
        }
    } NS_HANDLER {
        [self release];
        [localException raise];
    } NS_ENDHANDLER;
    if (expirationDate == nil)
        expirationDate = [[NSDate alloc] initWithTimeIntervalSinceNow:ONHostDefaultTimeToLiveTimeInterval];


    return self;
}

- (BOOL)isExpired;
{
    return [expirationDate timeIntervalSinceNow] < 0.0;
}

// The normal gethostbyname() function is not thread-safe.  Additionally, if the name ends up being unregistered, a long timeout period will take effect during which we can do no other host name lookups, assuming that we were to solve this problem by simply putting a lock around our calls to gethostbyname().

// Instead, we have a tool subproject that executes the call to gethostbyname() in its own process.  Under NEXTSTEP and Rhapsody, this will cause two calls to the lookupd process which will in turn contact DNS, NetInfo, NIS and possibly other services (LDAP) in the future in order to resolve the name.  Since lookupd itself is multi-threaded, this will allow two namelookups to proceed in parallel.

// Other operating systems should also support parallel name lookups at this level and if they don't then there probably isn't much that we could do about it.

// This method invokes the ONGetHostByName tool.  If there is an error looking up the addresses for the hostname, it will be returned in the exit status.  Otherwise, the network addresses for the host will be output as raw bytes in the byte order that they would have been returned to us from gethostbyname().  This allows us to easily parse the ints and stick them in a in_addr struct.

- (void)_lookupHostInfoByPipe;
{
    static NSString *ONGetHostByNamePath;
    const unsigned char *addressBytes;
    unsigned long int addressCount, addressIndex;
    int addressFamily, addressLength;
    NSData *addressData;
    ONHostAddress **addressIds;
    unsigned int canonicalHostnameLength, hostnameLength;
    NSRange nextRange;
    NSData *outputData;

    OBPRECONDITION(addresses == nil);
    
    if (!ONGetHostByNamePath) {
        NSBundle *thisBundle = [NSBundle bundleForClass:[ONHost class]];
        NSString *toolExtension = @"";

        ONGetHostByNamePath = [[thisBundle pathForResource:@"ONGetHostEntry" ofType:toolExtension] retain];
        if (!ONGetHostByNamePath) {
            NSString *noONGetHostEntryMsg = NSLocalizedStringFromTableInBundle(@"Cannot find the ONGetHostEntry tool", @"OmniNetworking", thisBundle, @"error - resource is missing from framework bundle");
            [NSException raise:ONGetHostByNameNotFoundExceptionName format:noONGetHostEntryMsg];
        }
    }

    {
        int pipeReadDescriptor, pipeWriteDescriptor;
        {
            int pipeFD[2];
    
            if (pipe(pipeFD) != 0) {
                [NSException raise:ONHostNameLookupErrorExceptionName posixErrorNumber:OMNI_ERRNO() format:NSLocalizedStringFromTableInBundle(@"Error looking up host %@: %s", @"OmniNetworking", [NSBundle bundleForClass:[ONHost class]], @"gethostbyname error - other errors - specific cause of error is not known"), hostname, strerror(OMNI_ERRNO())];
            }
            pipeReadDescriptor = pipeFD[0];
            pipeWriteDescriptor = pipeFD[1];
        }

        const char *toolPath = [[NSFileManager defaultManager] fileSystemRepresentationWithPath:ONGetHostByNamePath];
        const char *hostnameParameter = [[self IDNEncodedHostname] UTF8String]; // The result should be ASCII, but -UTF8String is a superset and there is no ASCII variant.
        pid_t child = vfork();
        switch (child) {
            case -1: // Error
                close(pipeReadDescriptor);
                close(pipeWriteDescriptor);
                [NSException raise:ONHostNameLookupErrorExceptionName posixErrorNumber:OMNI_ERRNO() format:NSLocalizedStringFromTableInBundle(@"Error looking up host %@: %s", @"OmniNetworking", [NSBundle bundleForClass:[ONHost class]], @"gethostbyname error - other errors - specific cause of error is not known"), hostname, strerror(OMNI_ERRNO())];
                OBASSERT_NOT_REACHED("Raising an exception should not return");
    
            case 0: // Child
                close(pipeReadDescriptor); // Close the parent's half of the pipe
    
                if (dup2(pipeWriteDescriptor, STDOUT_FILENO) != STDOUT_FILENO)
                    _exit(NETDB_INTERNAL); // Use _exit() not exit(): don't flush the parent's file buffers
    
                close(pipeWriteDescriptor); // We've copied it to STDOUT_FILENO, we can close the other descriptor now
                close(STDIN_FILENO); // The child doesn't need stdin
                close(STDERR_FILENO); // The child doesn't need stderr
    
                execl(toolPath, toolPath, "name", hostnameParameter, NULL);
                _exit(NETDB_INTERNAL); // Use _exit() not exit(): don't flush the parent's file buffers
                OBASSERT_NOT_REACHED("_exit() should not return");
    
            default: // Parent
                break;
        }

        unsigned int offset = 0, capacity = 8192;
        int childStatus;

        close(pipeWriteDescriptor); // Close the child's half of the pipe

        NSMutableData *readData = [NSMutableData dataWithLength:capacity];
        void *bytes = [readData mutableBytes];
        while (YES) {
            int length = read(pipeReadDescriptor, bytes + offset, capacity - offset);

            if (length == 0)
                break;

            if (length == -1) {
                if (OMNI_ERRNO() == EINTR)
                    continue;

                close(pipeReadDescriptor);
                waitpid(child, &childStatus, 0);
                [NSException raise:ONHostNameLookupErrorExceptionName posixErrorNumber:OMNI_ERRNO() format:NSLocalizedStringFromTableInBundle(@"Error looking up host %@: %s", @"OmniNetworking", [NSBundle bundleForClass:[ONHost class]], @"gethostbyname error - other errors - specific cause of error is not known"), hostname, strerror(OMNI_ERRNO())];
            }
    
            offset += length;
            if (offset == capacity) {
                capacity += capacity;
                [readData setLength:capacity];
                bytes = [readData mutableBytes];
            }
        }
    
        [readData setLength:offset];
        outputData = readData;
        close(pipeReadDescriptor);
        do {
            waitpid(child, &childStatus, 0);
        } while (!WIFEXITED(childStatus));
        unsigned int terminationStatus = WEXITSTATUS(childStatus);
        if (terminationStatus != 0)
            [ONHost _raiseExceptionForHostErrorNumber:terminationStatus hostname:hostname];
    }


    nextRange = NSMakeRange(0, sizeof(canonicalHostnameLength));
    [outputData getBytes:&canonicalHostnameLength range:nextRange];
    hostnameLength = MIN(canonicalHostnameLength, [outputData length] - sizeof(canonicalHostnameLength));
    nextRange = NSMakeRange(NSMaxRange(nextRange), hostnameLength);
    canonicalHostname = [[NSString alloc] initWithData:[outputData subdataWithRange:nextRange] encoding:[NSString defaultCStringEncoding]];
    nextRange = NSMakeRange(NSMaxRange(nextRange), [outputData length] - NSMaxRange(nextRange));
    if (nextRange.length < 2*sizeof(int))
        [ONHost _raiseExceptionForHostErrorNumber:-1 hostname:hostname];
    [outputData getBytes:&addressFamily range:NSMakeRange(nextRange.location, sizeof(int))];
    [outputData getBytes:&addressLength range:NSMakeRange(nextRange.location + sizeof(int), sizeof(int))];
    nextRange.location += 2*sizeof(int);
    nextRange.length   -= 2*sizeof(int);
    addressData = [outputData subdataWithRange:nextRange];
    addressCount = [addressData length] / addressLength;
    OBASSERT(([addressData length] % addressLength) == 0);
    if (addressCount == 0)
        [ONHost _raiseExceptionForHostErrorNumber:NO_DATA hostname:hostname];

    addressBytes = [addressData bytes];
    addressIds = malloc(addressCount * sizeof(*addressIds));
    for (addressIndex = 0; addressIndex < addressCount; addressIndex ++) {
        addressIds[addressIndex] = [ONHostAddress hostAddressWithInternetAddress:addressBytes family:addressFamily];
        addressBytes += addressLength;
    }
    addresses = [[NSArray alloc] initWithObjects:addressIds count:addressCount];
    free(addressIds);
}

// Somebody out there has finally twigged to the fact that it would be nice to be able to resolve host information in multiple threads; the getaddrinfo() API is threadsafe, and the implementation in Darwin is nearly-threadsafe as of MacOS 10.2, according to the Darwin CVS repository. By "nearly threadsafe" I mean that it's approximately as threadsafe as our lookupd hack (in fact, it has pretty much the same problem we used to have: a race condition when linking to the lookupd process). 

- (void)_lookupHostInfoUsingGetaddrinfo;
{
    const char *myHostname;
    struct addrinfo hints;
    struct addrinfo *results, *result;
    NSMutableArray *addressBuf;
    int err;

    OBPRECONDITION(addresses == nil);
    
    myHostname = [[self IDNEncodedHostname] UTF8String]; // The result should be ASCII, but -UTF8String is a superset and there is no ASCII variant.
    bzero(&hints, sizeof(hints));
    hints.ai_flags = AI_CANONNAME;
    results = NULL;
    
    if (ONHostOnlyResolvesIPv4Addresses)
        hints.ai_family = PF_INET;
    else
        hints.ai_family = PF_UNSPEC;

    err = getaddrinfo(myHostname, NULL, &hints, &results);
    if (err != 0) {
        [[[self class] _exceptionForExtendedHostErrorNumber:err hostname:hostname] raise];
    }

    if (results->ai_canonname != NULL) {
        canonicalHostname = [[NSString alloc] initWithCString:results->ai_canonname];
    }

    addressBuf = [[NSMutableArray alloc] init];
    for(result = results; result != NULL; result = result->ai_next) {
        ONHostAddress *address = [ONHostAddress hostAddressWithSocketAddress:result->ai_addr];
        if (address != nil && ![squatterAddresses containsObject:address])
            [addressBuf addObject:address];
    }
    freeaddrinfo(results);

    addresses = [[NSArray alloc] initWithArray:addressBuf];
    [addressBuf release];
}


@end

@implementation ONHost (Private)

// Punycode is defined in RFC 3492

#define ACEPrefix @"xn--"   // Prefix for encoded labels, defined in RFC3490 [5]

#define encode_character(c) (c) < 26 ? (c) + 'a' : (c) - 26 + '0'

static const short punycodeDigitValue[0x7B] = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 0x00 - 0x0F
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 0x10 - 0x1F
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 0x20 - 0x2F
    26, 27, 28, 29, 30, 31, 32, 33, 34, 35, -1, -1, -1, -1, -1, -1, // 0x30 - 0x3F
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, // 0x40 - 0x4F
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1, // 0x50 - 0x5F
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, // 0x60 - 0x6F
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25                      // 0x70 - 0x7A
};


static int adaptPunycodeDelta(int delta, int number, BOOL firstTime)
{
    int power;
    
    delta = firstTime ? delta / 700 : delta / 2;
    delta += delta / number;
    
    for (power = 0; delta > (35 * 26) / 2; power += 36)
        delta /= 35;
    return power + (35 + 1) * delta / (delta + 38);
}

/* Minimal validity checking. This should be elaborated to include the full IDN stringprep profile. */
static BOOL validIDNCodeValue(unsigned codepoint)
{
    /* Valid Unicode, non-basic codepoint? (implied by rfc3492) */
    if (codepoint < 0x9F || codepoint > 0x10FFFF)
        return NO;
    
    /* Some prohibited values from rfc3454 referenced by rfc3491[5] */
    if (codepoint == 0x00A0 ||
        (codepoint >= 0x2000 && codepoint <= 0x200D) ||
        codepoint == 0x202F || codepoint == 0xFEFF ||
        ( codepoint >= 0xFFF9 && codepoint <= 0xFFFF ))
        return NO; /* Miscellaneous whitespace & non-printing characters */
    
    int plane = ( codepoint & ~(0xFFFF) );

    if (plane == 0x0F0000 || plane == 0x100000 ||
        (codepoint >= 0xE000 && codepoint <= 0xF8FF))
        return NO;  /* Private use areas */
        
    if ((codepoint & 0xFFFE) == 0xFFFE ||
        (codepoint >= 0xD800 && codepoint <= 0xDFFF) ||
        (codepoint >= 0xFDD0 && codepoint <= 0xFDEF))
        return NO; /* Various non-character code points */
    
    /* end of gauntlet */
    return YES;
}

+ (NSString *)_punycodeEncode:(NSString *)aString;
{
    // setup buffers
    char outputBuffer[MAX_HOSTNAME_LEN]; 
    int stringLength = [aString length];
    unichar *inputBuffer = alloca(stringLength * sizeof(unichar));
    unichar *inputPtr, *inputEnd = inputBuffer + stringLength;
    char *outputEnd = outputBuffer + MAX_HOSTNAME_LEN;
    char *outputPtr = outputBuffer;
    
    // check once for hostname too long here and just refuse to encode if it is (this handles it if all ASCII)
    // there are additional checks for running over the buffer during the encoding loop
    if (stringLength > MAX_HOSTNAME_LEN)
        return aString;
    
    [aString getCharacters:inputBuffer];
    
    // handle ASCII characters
    for (inputPtr = inputBuffer; inputPtr < inputEnd; inputPtr++) {
        if (*inputPtr < 0x80) 
            *outputPtr++ = *inputPtr;            
    }
    int handled = outputPtr - outputBuffer;
    
    if (handled == stringLength)
        return aString;
    
    // add dash separator
    if (handled > 0 && outputPtr < outputEnd)
        *outputPtr++ = '-';
    
    // encode the rest
    int n = 0x80;
    int delta = 0;
    int bias = 72;
    BOOL firstTime = YES;
    
    while (handled < stringLength) {
        unichar max = (unichar)-1;
        for (inputPtr = inputBuffer; inputPtr < inputEnd; inputPtr++) {
            if (*inputPtr >= n && *inputPtr < max)
                max = *inputPtr;
        }
        
        delta += (max - n) * (handled + 1);
        n = max;
                
        for (inputPtr = inputBuffer; inputPtr < inputEnd; inputPtr++) {
            if (*inputPtr < n) 
                delta++;
            else if (*inputPtr == n) {
                int oldDelta = delta;
                int power = 36;
                
                // NSLog(@"encode: delta=%d pos=%d bias=%d codepoint=%05x", delta, inputPtr-inputBuffer, bias, *inputPtr);
                
                while (1) {
                    int t;
                    if (power <= bias)
                        t = 1;
                    else if (power >= bias + 26)
                        t = 26;
                    else
                        t = power - bias;
                    if (delta < t)
                        break;
                    if (outputPtr >= outputEnd)
                        return aString;
                    *outputPtr++ = encode_character(t + (delta - t) % (36 - t));
                    delta = (delta - t) / (36 - t);
                    power += 36;
		}
                
                if (outputPtr >= outputEnd)
                    return aString;
                *outputPtr++ = encode_character(delta);
                bias = adaptPunycodeDelta(oldDelta, ++handled, firstTime);
                firstTime = NO;
                delta = 0;
	    }
	}
        delta++;
        n++;
    }
    if (outputPtr >= outputEnd)
        return aString;
    *outputPtr++ = 0;
#ifdef DEBUG_toon    
    NSLog(@"Punycode encoded \"%@\" into \"%s\"", aString, outputBuffer);
#endif    
    return [ACEPrefix stringByAppendingString:[NSString stringWithCString:outputBuffer]];
}

+ (NSString *)_punycodeDecode:(NSString *)aString;
{
    NSMutableString *decoded;
    NSRange deltas;
    unsigned int *delta;
    unsigned deltaCount, deltaIndex;
    unsigned labelLength, decodedLabelLength;
    const unsigned acePrefixLength = 4;
        
    /* Check that the string has the IDNA ACE prefix. Most strings won't. */
    labelLength = [aString length];
    OBASSERT([ACEPrefix length] == acePrefixLength);
    if (labelLength < acePrefixLength ||
        ([aString compare:ACEPrefix options:NSCaseInsensitiveSearch range:(NSRange){0,acePrefixLength}] != NSOrderedSame))
        return aString;
    
    /* Also, any valid encoded string will be all-ASCII */
    if (![aString canBeConvertedToEncoding:NSASCIIStringEncoding])
        return aString;
    
    /* Find the delimiter that marks the end of the basic-code-points section. */
    NSRange delimiter = [aString rangeOfString:@"-"
                                       options:NSBackwardsSearch
                                         range:(NSRange){acePrefixLength, labelLength-acePrefixLength}];
    if (delimiter.length > 0) {
        decoded = [[aString substringWithRange:(NSRange){acePrefixLength, delimiter.location - acePrefixLength}] mutableCopy];
        deltas = (NSRange){NSMaxRange(delimiter), labelLength - NSMaxRange(delimiter)};
    } else {
        /* No delimiter means no basic code point section: it's all encoded deltas (RFC3492 [3.1]) */
        decoded = [[NSMutableString alloc] init];
        deltas = (NSRange){acePrefixLength, labelLength - acePrefixLength};
    }
    
    /* If there aren't any deltas, it's not a valid IDN label, because you're not supposed to encode something that didn't need to be encoded. */
    if (deltas.length == 0) {
        [decoded release];
        return aString;
    }
    
    decodedLabelLength = [decoded length];
    
    /* Convert the variable-length-integers in the deltas section into machine representation */
    {
        unichar *enc;
        unsigned i, bias, value, weight, position;
        BOOL reset;
        const int base = 36, tmin = 1, tmax = 26;
        
        enc = malloc(sizeof(*enc) * deltas.length);  // code points from encoded string
        delta = malloc(sizeof(*delta) * deltas.length); // upper bound on number of decoded integers
        deltaCount = 0;
        bias = 72;
        reset = YES;
        value = weight = position = 0;
                
        [aString getCharacters:enc range:deltas];
        for(i = 0; i < deltas.length; i++) {
            int digit, threshold;
            
            if (reset) {
                value = 0;
                weight = 1;
                position = 0;
                reset = NO;
            }
            
            if (enc[i] <= 0x7A)
                digit = punycodeDigitValue[enc[i]];
            else {
                free(enc);
                goto fail;
            }
            if (digit < 0) { // unassigned value
                free(enc);
                goto fail;
            }
            
            value += weight * digit;
            threshold = base * (position+1) - bias;
            
            // clamp to tmin=1 tmax=26 (rfc3492 [5])
            threshold = MIN(threshold, tmax);
            threshold = MAX(threshold, tmin);
            
            if (digit < threshold) {
                delta[deltaCount++] = value;
                // NSLog(@"decode: delta[%d]=%d bias=%d from=%@", deltaCount-1, value, bias, [aString substringWithRange:(NSRange){deltas.location + i - position, position+1}]);
                bias = adaptPunycodeDelta(value, deltaCount + decodedLabelLength, (deltaCount==1) ? YES : NO);
                reset = YES;
            } else {
                weight *= (base - threshold);
                position ++;
            }
        }
        
        free(enc);
        
        if (!reset) {
            /* The deltas section ended in the middle of an integer: something's wrong */
            goto fail;
        }
        
        /* deltas[] now holds deltaCount integers */
    }

    /* now use the decoded integers to insert characters into the decoded string */
    {
        unsigned position, codeValue;
        unichar ch[1];
        
        position = 0;
        codeValue = 0x80;
        
        for (deltaIndex = 0; deltaIndex < deltaCount; deltaIndex ++) {
            position += delta[deltaIndex];

            codeValue += ( position / (decodedLabelLength + 1) );
            position = ( position % (decodedLabelLength + 1) );
            
            if (!validIDNCodeValue(codeValue))
                goto fail;
            
            /* TODO: This will misbehave for code points greater than 0x0FFFF, because NSString uses a 16-bit encoding internally; the position values will be off by one afterwards [actually, we'll just get bad results because I'm using initWithCharacters:length: (BMP-only) instead of initWithCharacter: (all planes but only exists in OmniFoundation)] */
            ch[0] = codeValue;
            NSString *insertion = [[NSString alloc] initWithCharacters:ch length:1];
            [decoded replaceCharactersInRange:(NSRange){position, 0} withString:insertion];
            [insertion release];
            
            position ++;
            decodedLabelLength ++;
        }
    }
    
    if ([decoded length] != decodedLabelLength)
        goto fail;
    
    free(delta);
    
    NSString *normalized = [decoded precomposedStringWithCompatibilityMapping];  // Applies normalization KC
    if ([normalized compare:decoded options:NSLiteralSearch] != NSOrderedSame) {
        // Decoded string was not normalized, therefore could not have been the result of decoding a correctly encoded IDN.
        [decoded release];
        return aString;
    } else {
        [decoded autorelease];
        return normalized;
    }
    
fail:
    free(delta);
    [decoded release];
    return aString;
}

- (BOOL)_tryLocalhost;
    // Make sure localhost works even when the system is misconfigured.
{
    OBPRECONDITION(addresses == nil);
    
    if ([hostname caseInsensitiveCompare:@"localhost"] != NSOrderedSame)
        return NO;

    canonicalHostname = [@"localhost" retain];
    addresses = [[NSArray arrayWithObject:[ONHostAddress loopbackAddress]] retain];
    return YES;
}

- (BOOL)_tryToLookupHostInfoAsDottedQuad;
{
    ONHostAddress *address;

    OBPRECONDITION(addresses == nil);
    
    address = [ONHostAddress hostAddressWithNumericString:hostname];
    if (address == nil)
        return NO;

    // Oh ho!  They gave us an IP number in dotted quad notation!  I guess we'll return the dotted quad as the canonical hostname (computed in -canonicalHostname), and the converted address as the address.
    // (We're not calculating the real canonical hostname because it might return more addresses, and that wouldn't be what the user wants since they specifically specified a single address.)
    canonicalHostname = nil;
    addresses = [[NSArray arrayWithObject:address] retain];
    return YES;
}

@end

NSString *ONHostNotFoundExceptionName = @"ONHostNotFoundExceptionName";
NSString *ONHostNameLookupErrorExceptionName = @"ONHostNameLookupErrorExceptionName";
NSString *ONHostHasNoAddressesExceptionName = @"ONHostHasNoAddressesExceptionName";
NSString *ONGetHostByNameNotFoundExceptionName = @"ONGetHostByNameNotFoundExceptionName";
