// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniHTML/OHAppletCell.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OmniAppKit/OmniAppKit.h>
#import <OWF/OWF.h>

#import <OmniHTML/OHHTMLDocument.h>
#import <OmniHTML/OHHTMLView.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniHTML/Cells.subproj/OHAppletCell.m,v 1.23 2000/04/06 02:04:18 wjs Exp $")

@interface OHAppletCell (Private)
- (NSImage *)statusImage;
- (void)drawStatusImage:(NSImage *)statusImage withFrame:(NSRect)contentFrame inView:(OHHTMLView *)controlView;
@end

@implementation OHAppletCell

static OWContentType *appletContentType;
static NSImage *appletImage;
static NSImage *appletErrorImage;
static NSUserDefaults *defaults;

+ (void)initialize;
{
    static BOOL initialized = NO;

    [super initialize];
    if (initialized)
	return;
    initialized = YES;

    appletContentType = [OWContentType contentTypeForString:@"Omni/Applet"];
    appletImage = [[NSImage imageNamed:@"OHApplet" inBundleForClass:self] retain];
    [appletImage setDataRetained:YES];
    [appletImage setScalesWhenResized:YES];
    appletErrorImage = [[NSImage imageNamed:@"OHAppletError" inBundleForClass:self] retain];
    [appletErrorImage setDataRetained:YES];
    [appletErrorImage setScalesWhenResized:YES];
    defaults = [[NSUserDefaults standardUserDefaults] retain];
}

// Init and dealloc

- initWithCode:(NSString *)aClassName archive:(NSString *)anArchive codeBaseAddress:(OWAddress *)anAddress referringAddress:(OWAddress *)aReferringAddress htmlDocument:(OHHTMLDocument *)aDocument;
{
    OWWebPipeline *appletPipeline;
    OWAddress *loadAddress;

    if (![super init])
        return nil;

    nonretainedHTMLDocument = aDocument;
    codeBaseAddress = [anAddress retain];
    archive = [anArchive retain];
    if (![aClassName hasSuffix:@".class"])
        aClassName = [aClassName stringByAppendingString:@".class"];
    appletClassName = [aClassName retain];
    status = OHAppletStatusUnloaded;
    cellName = nil;

    if (archive)
        loadAddress = [codeBaseAddress addressForRelativeString:archive];
    else
        loadAddress = [codeBaseAddress addressForRelativeString:appletClassName];
    appletPipeline = [[OWWebPipeline alloc] initWithContent:loadAddress target:self useCachedErrorContent:YES];
    [appletPipeline setReferringAddress:aReferringAddress];
    [appletPipeline release];

    return self;
}

- (void)dealloc;
{    
    [OWPipeline cancelTarget:self];
    [appletController nullifyAppletCell];
    [appletClassName release];
    [codeBaseAddress release];
    [parameters release];
    [appletController release];
    [cellName release];
    [super dealloc];
}

//

- (void)addParameter:(NSString *)name value:(NSString *)value;
{
    if (!parameters)
	parameters = [[NSMutableDictionary alloc] init];
    if (value)
        [parameters setObject:value forKey:name];
}

- (void)addParameterDictionary:(NSDictionary *)dictionary;
{
    [parameters addEntriesFromDictionary:dictionary];
}

//

- (OWAddress *)codeBaseAddress;
{
    return codeBaseAddress;
}

- (NSString *)archive;
{
    return archive;
}

- (OHHTMLDocument *)document;
{
    return nonretainedHTMLDocument;
}

- (NSDictionary *)parameters;
{
    return parameters;
}

- (NSString *)name
{
    return cellName;
}

- (void)setName:(NSString *)appletName
{
    [appletName retain];
    [cellName release];
    cellName = appletName;
}

//

- (id <OHAppletController>)appletController;
{
    return appletController;
}

// Actions

- (IBAction)start:(id)sender;
{
    OWWebPipeline *appletPipeline;

    switch (status) {
        case OHAppletStatusUnloaded:
            appletPipeline = (OWWebPipeline *)[OWPipeline currentPipelineForTarget:self];
            [appletPipeline setContextObject:appletClassName forKey:@"AppletClassName"];
            [appletPipeline setContextObject:parameters forKey:@"AppletParameters"];
            [appletPipeline fetch];
            status = OHAppletStatusLoading;
            break;
        case OHAppletStatusReady:
        case OHAppletStatusStopped:
            status = OHAppletStatusRunning;
            [appletController start];
            break;
        default:
            return;
    }
    [self queueSelector:@selector(relayoutCell)];
}

- (IBAction)stop:(id)sender;
{
    if (status != OHAppletStatusRunning)
        return;
    [appletController stop];
    status = OHAppletStatusStopped;
}

// OHBasicCell subclass

- (NSSize)naturalSize;
{
    NSImage *statusImage;

    statusImage = [self statusImage];
    if (statusImage)
        return [statusImage size];
    else
        return [super naturalSize];
}

- (void)drawRect:(NSRect)drawRect inHTMLView:(OHHTMLView *)htmlView;
{
    NSImage *statusImage;
    NSRect contentFrame;

    statusImage = [self statusImage];
    contentFrame = [self contentFrame];
    if (statusImage) {
        [self drawStatusImage:statusImage withFrame:contentFrame inView:htmlView];
    } else {
        [super drawRect:contentFrame inHTMLView:htmlView];
    }
}

- (NSColor *)missingInteriorFillColor;
{
    return [NSColor colorWithDeviceRed:0.1 green:0.1 blue:0.8 alpha:0.15];
}

- (void)resetContentViewSize;
{
    if (![self statusImage])
        [super resetContentViewSize];
}

- (BOOL)wantsToTrackMouse;
{
    return YES;
}

- (BOOL)trackMouse:(NSEvent *)event inHTMLView:(OHHTMLView *)htmlView;
{
    if (!appletController)
        [self start:nil];
    return YES;
}

- (BOOL)hasLinks;
{
    return YES;
}

// OHViewCell subclass

- (void)setupContentView;
{
    [self setContentView:[appletController view]];
}

// OWTarget protocol

- (OWContentType *)targetContentType;
{
    return appletContentType;
}

- (OWTargetContentDisposition)pipeline:(OWPipeline *)aPipeline hasContent:(id <OWContent>)content;
{
    appletController = [content retain];
    status = OHAppletStatusReady;
    return OWTargetContentDisposition_ContentAccepted;
}

- (NSString *)targetTypeFormatString;
{
    return @"%@ Applet";
}

- (OWContentInfo *)parentContentInfo;
{
    return [nonretainedHTMLDocument contentInfo];
}

// OWOptionalTarget protocol

- (OWTargetContentDisposition)pipeline:(OWPipeline *)aPipeline hasAlternateContent:(id <OWContent>)someContent;
{
    if ([someContent isKindOfClass:[OWDataStream class]] && [(OWDataStream *)someContent resetContentTypeAndEncoding]) {
        // OK, we've reset the content type and encoding, let's try again
        [aPipeline addContent:someContent];
        [aPipeline startProcessingContent];
        return OWTargetContentDisposition_ContentUpdatedOrTargetChanged;
    } else {
        return OWTargetContentDisposition_ContentRejectedSavePipeline;
    }
}

- (void)pipelineDidEnd:(OWPipeline *)aPipeline;
{
    switch (status) {
        case OHAppletStatusReady:
            // OK, start that app
            [self queueSelector:@selector(start:)];
            break;
        case OHAppletStatusLoading:
            // The load failed
            status = OHAppletStatusError;
            [self queueSelector:@selector(redisplayCell)];
            break;
        default:
            // This isn't really expected, but if it does happen:  do nothing.
            break;
    }
}

// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];

    if (appletClassName)
        [debugDictionary setObject:appletClassName forKey:@"appletClassName"];
    if (codeBaseAddress)
        [debugDictionary setObject:codeBaseAddress forKey:@"codeBaseAddress"];
    if (parameters)
        [debugDictionary setObject:parameters forKey:@"parameters"];

    if (appletController)
        [debugDictionary setObject:appletController forKey:@"appletController"];

    return debugDictionary;
}

@end

@implementation OHAppletCell (Private)

- (NSImage *)statusImage;
{
    switch (status) {
        default:
        case OHAppletStatusUnloaded:
        case OHAppletStatusLoading:
            // TODO: Should probably have more status images
        case OHAppletStatusReady:
        case OHAppletStatusStopped:
            return appletImage;

        case OHAppletStatusRunning:
            if ([self hasContentView])
                return nil;
            [self setupContentView];
            if ([self hasContentView])
                return nil;

            status = OHAppletStatusError;
            // NO BREAK
        case OHAppletStatusError:
            return appletErrorImage;
    }
}

- (void)drawStatusImage:(NSImage *)statusImage withFrame:(NSRect)contentFrame inView:(OHHTMLView *)controlView;
{
    NSSize imageSize;
    float scale;

    [self drawMissingInteriorWithFrame:contentFrame inView:controlView];
    
    imageSize = [statusImage size];
    scale = MIN(MIN(NSWidth(contentFrame) / imageSize.width, NSHeight(contentFrame) / imageSize.height), 1.0);
    if (scale != 1.0)
        [statusImage setSize:NSMakeSize(floor(imageSize.width * scale), floor(imageSize.height * scale))];

    [self drawImage:statusImage withFrame:contentFrame inView:controlView];

    if (scale != 1.0)
        [statusImage setSize:imageSize];
}

@end
