// Copyright 1997-2000 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import <OmniHTML/OHViewCell.h>

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OmniAppKit/OmniAppKit.h>

#import <OmniHTML/OHHTMLView.h>

RCS_ID("$Header: /Network/Source/CVS/OmniGroup/Frameworks/OmniHTML/Cells.subproj/OHViewCell.m,v 1.19 2000/03/25 06:37:05 wjs Exp $")

@interface OHViewCell (Private)
- (void)contentViewFrameChanged:(NSNotification *)notification;
- (void)ensureContentView;
@end

@implementation OHViewCell

- init;
{
    if (![super init])
        return nil;

    _contentViewSize = NSZeroSize;

    return self;
}

- (void)dealloc;
{
    [self setContentView:nil];
    [super dealloc];
}

// API

- (BOOL)hasContentView;
{
    return _contentView != nil;
}

- (NSView *)contentView;
{
    [self ensureContentView];
    return _contentView;
}

- (void)setupContentView;
{
    // Subclasses override this to set up their content view
}

- (void)setContentView:(NSView *)newContentView;
{
    if (_contentView) {
        [[NSNotificationCenter defaultCenter] removeObserver:self name:NSViewFrameDidChangeNotification object:_contentView];
        [_contentView setNextKeyView:nil];
        [_contentView removeFromSuperview];
        [_contentView release];
    }
    _contentView = [newContentView retain];
    if (_contentView) {
        _contentViewSize = [_contentView frame].size;
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(contentViewFrameChanged:) name:NSViewFrameDidChangeNotification object:_contentView];
    }
}

- (void)resetContentViewOrigin;
{
    NSPoint newOrigin;

    newOrigin = [self contentFrame].origin;
    if (NSEqualPoints(newOrigin, [_contentView frame].origin))
        return;
    [_contentView setFrameOrigin:newOrigin];
}

- (void)resetContentViewSize;
{
    NSSize newSize;

    [self ensureContentView];
    newSize = [self contentFrame].size;
    if (NSEqualSizes(newSize, [_contentView frame].size))
        return;
    viewCellFlags.doNotRelayoutWhenContentViewFrameChanges = YES;
    [_contentView setFrameSize:newSize];
    viewCellFlags.doNotRelayoutWhenContentViewFrameChanges = NO;
}

// OHBasicCell subclass

- (void)setOrigin:(NSPoint)newOrigin;
{
    [super setOrigin:newOrigin];

    if (!_contentView)
        [self setupContentView];
    
    if (!_contentView)
        return; // Don't worry about it, sometimes they are created lazily

    // We do this both here and in drawRect:... below because if we don't do it here we won't get moved when our page resizes until we are drawn again, and so if we're resized off the bottom of the screen we'll leave a turd around until the user scrolls down.  If we don't do it below we could fail because the first time this cell lays out it may not have a contentView.
    [self resetContentViewOrigin];
    if ([_contentView superview] != [self containingView]) {
        [[self containingView] addSubview:_contentView];
    }
}

- (void)setLayoutBounds:(NSRect)bounds;
{
    [super setLayoutBounds:bounds];
    [self resetContentViewSize];
}

- (NSSize)naturalSize;
{
    [self ensureContentView];
    return _contentViewSize;
}

- (void)drawRect:(NSRect)drawRect inHTMLView:(OHHTMLView *)htmlView;
{
    [self ensureContentView];
    [self resetContentViewOrigin];
    if ([_contentView superview] != htmlView) {
	[htmlView addSubview:_contentView];
    }
}

- (int)baselineOffsetChangeToAlignText;
{
    return -3;
}

@end

@implementation OHViewCell (Private)

- (void)ensureContentView;
{
    if (!_contentView)
        [self setupContentView];
    OBASSERT(_contentView != nil);
}

// Notification

- (void)contentViewFrameChanged:(NSNotification *)notification;
{
    NSSize newContentViewSize;

    newContentViewSize = [_contentView frame].size;
    if (!NSEqualSizes(_contentViewSize, newContentViewSize)) {
        _contentViewSize = newContentViewSize;
        if (!viewCellFlags.doNotRelayoutWhenContentViewFrameChanges)
            [self relayoutCell];
    }
}

- (void)_setNextKeyViewCell:(OHViewCell *)nextKeyViewCell;
{
    NSView *nextContentView;

    nextContentView = [nextKeyViewCell contentView];
    [_contentView setNextKeyView:nextContentView];
}

@end
